// =====================================================================================
//
// Project:
// ImageAccess
// 
// Author:
// Daniel Sage
// Biomedical Imaging Group (BIG)
// Ecole Polytechnique Federale de Lausanne (EPFL)
// Lausanne, Switzerland
//
// Date:
// 11 July 2003
//
// Reference:
// D. Sage, M. Unser
// Teaching Image-Processing Programming in Java
// IEEE Signal Processing Magazine, vol. 20, no. 6, pp. 43-52, November 2003.
//
// Conditions of use:
// You'll be free to use this software for research purposes, but you
// should not redistribute it without our consent. In addition, we
// expect you to include a citation or acknowledgment whenever
// you present or publish results that are based on it.
//
// Permission for use in JIBS granted by Daniel Sage on 13 April 2011 under the
// following terms:
// You include the Resize in your distribution if you clearly mention the
// credits and put a reference on the scientific paper:
// 
// Credit:
// Resize: An ImageJ plugin to resize an image using high-quality interpolation
// Written by Arrate Muoz, David Leroux, Daniel Sage and Michael Unser at the
// Biomedical Image Group (BIG), EPFL, Switzerland
// 
// Reference
// A. Muoz Barrutia, T. Blu, M. Unser,
// "Least-Squares Image Resizing Using Finite Differences," IEEE Transactions on
// Image Processing, vol. 10, no. 9, pp. 1365-1378, September 2001.
// =====================================================================================

package ch.epfl.bigwww;

import ij.ImagePlus;
import ij.process.ByteProcessor;
import ij.process.ColorProcessor;
import ij.process.FloatProcessor;
import ij.process.ImageProcessor;

/**
 * ImageAccess is an interface layer to facilitate the access
 * to the pixels of ImageJ images. 
 * Methods of ImageAccess provides an easy and robust way to 
 * access to the pixels of images.
 *
 * The data are stored in a double array. 
 * Many methods get/put allows to access to the data. If the
 * user try to access outside of the image, the mirror boundary
 * conditions are applied. 
 * 
 * @author 	<br>
 * <a href="mailto:daniel.sage@epfl.ch?subject=ImageAccess Docs">Daniel Sage</a><br>
 * <b>Swiss Federal Institute of Technology Lausanne</b><br>
 * <a href="http://bigwww.epfl.ch">Biomedical Imaging Group</a><br>
 * CH-1015 Lausanne EPFL<br>
 * Switzerland<br>
 */

public class ImageAccess
{
	public static final int PATTERN_SQUARE_3x3 = 0;
	public static final int PATTERN_CROSS_3x3 = 1;

	private double pixels[] = null; // store the pixel data
	private int nx = 0; // size in X axis
	private int ny = 0; // size in Y axis
	private int size = 0; // size = nx*ny

	/**
	* Creates a new ImageAccess object from a 2D double array of pixels.
	* The size of the array determines the size of the image.
	*
	* @param array    an array of pixel (2D)
	*/
	public ImageAccess(double[][] array)
	{
		if (array == null) throw new ArrayStoreException(
				"Constructor: array == null.");
		this.ny = array[0].length;
		this.nx = array.length;
		this.size = this.nx * this.ny;
		this.pixels = new double[this.size];
		int k = 0;
		for (int j = 0; j < this.ny; j++)
			for (int i = 0; i < this.nx; i++)
				this.pixels[k++] = array[i][j];
	}

	/**
	* Creates a new object of the class ImageAccess from an 
	* ImageProcessor object.
	*
	* ImageProcessor object contains the image data, the size and 
	* the type of the image. The ImageProcessor is provided by ImageJ,
	* it should by a 8-bit, 16-bit. 
	*
	* @param ip    an ImageProcessor object provided by ImageJ
	*/
	public ImageAccess(ImageProcessor ip)
	{
		if (ip == null) throw new ArrayStoreException(
				"Constructor: ImageProcessor == null.");
		this.nx = ip.getWidth();
		this.ny = ip.getHeight();
		this.size = this.nx * this.ny;
		this.pixels = new double[this.size];
		if (ip.getPixels() instanceof byte[])
		{
			byte[] bsrc = (byte[]) ip.getPixels();
			for (int k = 0; k < this.size; k++)
				this.pixels[k] = (bsrc[k] & 0xFF);

		}
		else if (ip.getPixels() instanceof short[])
		{
			short[] ssrc = (short[]) ip.getPixels();
			for (int k = 0; k < this.size; k++)
				this.pixels[k] = (ssrc[k] & 0xFFFF);
		}
		else if (ip.getPixels() instanceof float[])
		{
			float[] fsrc = (float[]) ip.getPixels();
			for (int k = 0; k < this.size; k++)
				this.pixels[k] = fsrc[k];
		}
		else
		{
			throw new ArrayStoreException("Constructor: Unexpected image type.");
		}
	}

	/**
	* Creates a new object of the class ImageAccess from an 
	* ColorProcessor object.
	*
	* ImageProcessor object contains the image data, the size and 
	* the type of the image. The ColorProcessor is provided by ImageJ,
	* The ImageAccess contains one plane (red, green or blue) selected
	* with the colorPlane parameter.
	*
	* @param cp    			an ColorProcessor object
	* @param colorPlane   	index of the color plane 0, 1 or 2
	*/
	public ImageAccess(ColorProcessor cp, int colorPlane)
	{
		if (cp == null) throw new ArrayStoreException(
				"Constructor: ColorProcessor == null.");
		if (colorPlane < 0) throw new ArrayStoreException(
				"Constructor: colorPlane < 0.");
		if (colorPlane > 2) throw new ArrayStoreException(
				"Constructor: colorPlane > 2.");
		this.nx = cp.getWidth();
		this.ny = cp.getHeight();
		this.size = this.nx * this.ny;
		this.pixels = new double[this.size];
		byte[] r = new byte[this.size];
		byte[] g = new byte[this.size];
		byte[] b = new byte[this.size];
		cp.getRGB(r, g, b);
		if (colorPlane == 0) for (int k = 0; k < this.size; k++)
			this.pixels[k] = (r[k] & 0xFF);
		else if (colorPlane == 1) for (int k = 0; k < this.size; k++)
			this.pixels[k] = (g[k] & 0xFF);
		else if (colorPlane == 2) for (int k = 0; k < this.size; k++)
			this.pixels[k] = (b[k] & 0xFF);
	}

	/**
	* Creates a new object of the class ImageAccess.
	*
	* The size of the image are given as parameter.
	* The data pixels are empty and are not initialized.
	*
	* @param nx       	the size of the image along the X-axis
	* @param ny       	the size of the image along the Y-axis
	*/
	public ImageAccess(int nxIn, int nyIn)
	{
		if (nxIn < 1) throw new ArrayStoreException("Constructor: nx < 1.");
		if (nyIn < 1) throw new ArrayStoreException("Constructor: ny < 1.");
		this.nx = nxIn;
		this.ny = nyIn;
		this.size = this.nx * this.ny;
		this.pixels = new double[this.size];
	}

	/**
	* Return the width of the image.
	*
	* @return     	the image width
	*/
	public int getWidth()
	{
		return this.nx;
	}

	/**
	* Return the height of the image.
	*
	* @return     	the image height
	*/
	public int getHeight()
	{
		return this.ny;
	}

	/**
	* Return the maximum value of ImageAccess.
	*
	* @return     	the maximum value
	*/
	public double getMaximum()
	{
		double maxi = this.pixels[0];
		for (int i = 1; i < this.size; i++)
			if (this.pixels[i] > maxi) maxi = this.pixels[i];
		return maxi;
	}

	/**
	* Return the minimum value of ImageAccess.
	*
	* @return     	the minimum value
	*/
	public double getMinimum()
	{
		double mini = this.pixels[0];
		for (int i = 1; i < this.size; i++)
			if (this.pixels[i] < mini) mini = this.pixels[i];
		return mini;
	}

	/**
	* Return the mean value of ImageAccess.
	*
	* @return     	the mean value
	*/
	public double getMean()
	{
		double mean = 0.0;
		for (int i = 0; i < this.size; i++)
			mean += this.pixels[i];
		mean /= (this.size);
		return mean;
	}

	/**
	* Returns a copy of the pixel data organize in a
	* 2D array.
	*
	* @return     	the 2D double array
	*/
	public double[][] getArrayPixels()
	{
		double[][] array = new double[this.nx][this.ny];
		int k = 0;
		for (int j = 0; j < this.ny; j++)
			for (int i = 0; i < this.nx; i++)
				array[i][j] = this.pixels[k++];
		return array;
	}

	/**
	* Returns a reference to the pixel data in double (1D).
	*
	* @return     	the 1D double array
	*/
	public double[] getPixels()
	{
		return this.pixels;
	}

	/**
	* Create a FloatProcessor from the pixel data.
	* The double values of the pixel are simply casted in float.
	*
	* @return     	the FloatProcessor
	*/
	public FloatProcessor createFloatProcessor()
	{
		FloatProcessor fp = new FloatProcessor(this.nx, this.ny);
		float[] fsrc = new float[this.size];
		for (int k = 0; k < this.size; k++)
			fsrc[k] = (float) (this.pixels[k]);
		fp.setPixels(fsrc);
		return fp;
	}

	/**
	* Create a ByteProcessor from the pixel data.
	* The double values of the pixel are clipped in the [0..255] range.
	*
	* @return     	the ByteProcessor
	*/
	public ByteProcessor createByteProcessor()
	{
		ByteProcessor bp = new ByteProcessor(this.nx, this.ny);
		byte[] bsrc = new byte[this.size];
		double p;
		for (int k = 0; k < this.size; k++)
		{
			p = this.pixels[k];
			if (p < 0) p = 0.0;
			if (p > 255.0) p = 255.0;
			bsrc[k] = (byte) p;
		}
		bp.setPixels(bsrc);
		return bp;
	}

	/**
	* Create a new ImageAccess object by duplication of the current the 
	* ImageAccess object.
	*
	* @return   a new ImageAccess object
	**/
	public ImageAccess duplicate()
	{
		ImageAccess ia = new ImageAccess(this.nx, this.ny);
		for (int i = 0; i < this.size; i++)
			ia.pixels[i] = this.pixels[i];
		return ia;
	}

	/**
	* An ImageAccess object calls this method for getting
	* the gray level of a selected pixel.
	*
	* Mirror border conditions are applied.
	*
	* @param x		input, the integer x-coordinate of a pixel
	* @param y		input, the integer y-coordinate of a pixel
	* @return     	the gray level of the pixel (double) 
	*/
	public double getPixel(int x, int y)
	{
		int periodx = 2 * this.nx - 2;
		int periody = 2 * this.ny - 2;
		if (x < 0)
		{
			while (x < 0)
				x += periodx; // Periodize	
			if (x >= this.nx) x = periodx - x; // Symmetrize
		}
		else if (x >= this.nx)
		{
			while (x >= this.nx)
				x -= periodx; // Periodize	
			if (x < 0) x = -x; // Symmetrize
		}

		if (y < 0)
		{
			while (y < 0)
				y += periody; // Periodize	
			if (y >= this.ny) y = periody - y; // Symmetrize	
		}
		else if (y >= this.ny)
		{
			while (y >= this.ny)
				y -= periody; // Periodize	
			if (y < 0) y = -y; // Symmetrize
		}
		return this.pixels[x + y * this.nx];
	}

	/**
	* An ImageAccess object calls this method for getting
	* the gray level of a selected pixel using a bilinear interpolation.
	* The coordinates can be given in double and the 
	* bilinear interpolation is applied the find the gray level.
	*
	* Mirror border conditions are applied.
	*
	* @param x		input, the double x-coordinate of a pixel
	* @param y		input, the double y-coordinate of a pixel
	* @return     	the gray level of the pixel (double) 
	*/
	public double getInterpolatedPixel(double x, double y)
	{
		if (Double.isNaN(x)) return 0;
		if (Double.isNaN(y)) return 0;

		if (x < 0)
		{
			int periodx = 2 * this.nx - 2;
			while (x < 0)
				x += periodx; // Periodize
			if (x >= this.nx) x = periodx - x; // Symmetrize
		}
		else if (x >= this.nx)
		{
			int periodx = 2 * this.nx - 2;
			while (x >= this.nx)
				x -= periodx; // Periodize
			if (x < 0) x = -x; // Symmetrize
		}

		if (y < 0)
		{
			int periody = 2 * this.ny - 2;
			while (y < 0)
				y += periody; // Periodize
			if (y >= this.ny) y = periody - y; // Symmetrize
		}
		else if (y >= this.ny)
		{
			int periody = 2 * this.ny - 2;
			while (y >= this.ny)
				y -= periody; // Periodize
			if (y < 0) y = -y; // Symmetrize
		}
		int i;
		if (x >= 0.0) i = (int) x;
		else
		{
			final int iAdd = (int) x - 1;
			i = ((int) (x - iAdd) + iAdd);
		}
		int j;
		if (y >= 0.0) j = (int) y;
		else
		{
			final int iAdd = (int) y - 1;
			j = ((int) (y - iAdd) + iAdd);
		}

		double dx = x - i;
		double dy = y - j;
		int di;
		if (i >= this.nx - 1) di = -1;
		else di = 1;
		int index = i + j * this.nx;
		double v00 = this.pixels[index];
		double v10 = this.pixels[index + di];
		if (j >= this.ny - 1) index -= this.nx;
		else index += this.nx;
		double v01 = this.pixels[index];
		double v11 = this.pixels[index + di];
		return (dx * (v11 * dy - v10 * (dy - 1.0)) - (dx - 1.0)
				* (v01 * dy - v00 * (dy - 1.0)));
	}

	/**
	* An ImageAccess object calls this method for getting a 
	* whole column of the image.
	*
	* The column should already created with the correct size [ny].
	*
	* @param x       	input, the integer x-coordinate of a column
	* @param column     output, an array of the type double
	*/
	public void getColumn(int x, double[] column)
	{
		if (x < 0) throw new IndexOutOfBoundsException("getColumn: x < 0.");
		if (x >= this.nx) throw new IndexOutOfBoundsException(
				"getColumn: x >= nx.");
		if (column == null) throw new ArrayStoreException(
				"getColumn: column == null.");
		if (column.length != this.ny) throw new ArrayStoreException(
				"getColumn: column.length != ny.");
		for (int i = 0; i < this.ny; i++)
		{
			column[i] = this.pixels[x];
			x += this.nx;
		}
	}

	/**
	* An ImageAccess object calls this method for getting a part 
	* of column.
	* The starting point is given by the y parameter and the ending
	* determine by the size of the column parameter. The column 
	* parameter should already created.
	*
	* @param x       	input, the integer x-coordinate of a column
	* @param y       	input, starting point
	* @param column     output, an array of the type double
	*/
	public void getColumn(int x, int y, double[] column)
	{
		if (x < 0) throw new IndexOutOfBoundsException("getColumn: x < 0.");
		if (x >= this.nx) throw new IndexOutOfBoundsException(
				"getColumn: x >= nx.");
		if (column == null) throw new ArrayStoreException(
				"getColumn: column == null.");
		int by = column.length;
		if (y >= 0) if (y < this.ny - by - 1)
		{
			int index = y * this.nx + x;
			for (int i = 0; i < by; i++)
			{
				column[i] = this.pixels[index];
				index += this.nx;
			}
			return;
		}
		// Getting information outside of the image
		int yt[] = new int[by];
		for (int k = 0; k < by; k++)
		{
			int ya = y + k;
			int periody = 2 * this.ny - 2;
			while (ya < 0)
				ya += periody; // Periodize
			while (ya >= this.ny)
			{
				ya = periody - ya; // Symmetrize
				if (ya < 0) ya = -ya;
			}
			yt[k] = ya;
		}
		int index = 0;
		for (int i = 0; i < by; i++)
		{
			index = yt[i] * this.nx + x;
			column[i] = this.pixels[index];
		}
	}

	/**
	* An ImageAccess object calls this method for getting a 
	* whole row of the image.
	*
	* The row should already created with the correct size [nx].
	*
	* @param y       	input, the integer y-coordinate of a row
	* @param row        output, an array of the type double
	*/
	public void getRow(int y, double[] row)
	{
		if (y < 0) throw new IndexOutOfBoundsException("getRow: y < 0.");
		if (y >= this.ny) throw new IndexOutOfBoundsException("getRow: y >= ny.");
		if (row == null) throw new ArrayStoreException("getColumn: row == null.");
		if (row.length != this.nx) throw new ArrayStoreException(
				"getColumn: row.length != nx.");
		y *= this.nx;
		for (int i = 0; i < this.nx; i++)
			row[i] = this.pixels[y++];
	}

	/**
	* An ImageAccess object calls this method for getting a part 
	* of row.
	* The starting point is given by the y parameter and the ending
	* determine by the size of the row parameter. The row 
	* parameter should already created.
	*
	* @param x       	input, starting point
	* @param y       	input, the integer y-coordinate of a row
	* @param row        output, an array of the type double
	*/
	public void getRow(int x, int y, double[] row)
	{
		if (y < 0) throw new IndexOutOfBoundsException("getRow: y < 0.");
		if (y >= this.ny) throw new IndexOutOfBoundsException("getRow: y >= ny.");
		if (row == null) throw new ArrayStoreException("getRow: row == null.");
		int bx = row.length;
		if (x >= 0) if (x < this.nx - bx - 1)
		{
			int index = y * this.nx + x;
			for (int i = 0; i < bx; i++)
			{
				row[i] = this.pixels[index++];
			}
			return;
		}
		int periodx = 2 * this.nx - 2;
		int xt[] = new int[bx];
		for (int k = 0; k < bx; k++)
		{
			int xa = x + k;
			while (xa < 0)
				xa += periodx; // Periodize
			while (xa >= this.nx)
			{
				xa = periodx - xa; // Symmetrize
				if (xa < 0) xa = -xa;
			}
			xt[k] = xa;
		}
		int somme = 0;
		int index = y * this.nx;
		for (int i = 0; i < bx; i++)
		{
			somme = index + xt[i];
			row[i] = this.pixels[somme];
		}
	}

	/**
	* An ImageAccess object calls this method for getting a neighborhood
	* arround a pixel position.
	*
	* The neigh parameter should already created. The size of the array 
	* determines the neighborhood block.
	*
	* <br>Mirror border conditions are applied.
	* <br>
	* <br>The pixel value of (x-n/2, y-n/2) is put into neigh[0][0]
	* <br>...
	* <br>The pixel value of (x+n/2, y+n/2) is put into neigh[n-1][n-1]
	* <br>
	* <br>For example if neigh is a double[4][4]:
	* <br>The pixel value of (x-1, y-1) is put into neigh[0][0]
	* <br>The pixel value of (x  , y  ) is put into neigh[1][1]
	* <br>The pixel value of (x, y+1) is put into neigh[1][2]
	* <br>The pixel value of (x+1, y-2) is put into neigh[2][0]
	* <br>...
	* <br>For example if neigh is a double[5][5]:
	* <br>The pixel value of (x-2, y-2) is put into neigh[0][0]
	* <br>The pixel value of (x-1, y-1) is put into neigh[1][1]
	* <br>The pixel value of (x  , y  ) is put into neigh[2][2]
	* <br>The pixel value of (x, y+1) is put into neigh[2][3]
	* <br>The pixel value of (x+2, y-2) is put into neigh[4][0]

	* @param x		the integer x-coordinate of a selected central pixel
	* @param y		the integer y-coordinate of a selected central pixel
	* @param neigh	output, a 2D array s
	*/
	public void getNeighborhood(int x, int y, double neigh[][])
	{
		int bx = neigh.length;
		int by = neigh[0].length;
		int bx2 = (bx - 1) / 2;
		int by2 = (by - 1) / 2;
		if (x >= bx2) if (y >= by2) if (x < this.nx - bx2 - 1) if (y < this.ny
				- by2 - 1)
		{
			int index = (y - by2) * this.nx + (x - bx2);
			for (int j = 0; j < by; j++)
			{
				for (int i = 0; i < bx; i++)
				{
					neigh[i][j] = this.pixels[index++];
				}
				index += (this.nx - bx);
			}
			return;
		}
		int xt[] = new int[bx];
		for (int k = 0; k < bx; k++)
		{
			int xa = x + k - bx2;
			int periodx = 2 * this.nx - 2;
			while (xa < 0)
				xa += periodx; // Periodize
			while (xa >= this.nx)
			{
				xa = periodx - xa; // Symmetrize
				if (xa < 0) xa = -xa;
			}
			xt[k] = xa;
		}
		int yt[] = new int[by];
		for (int k = 0; k < by; k++)
		{
			int ya = y + k - by2;
			int periody = 2 * this.ny - 2;
			while (ya < 0)
				ya += periody; // Periodize
			while (ya >= this.ny)
			{
				ya = periody - ya; // Symmetrize
				if (ya < 0) ya = -ya;
			}
			yt[k] = ya;
		}
		int sum = 0;
		for (int j = 0; j < by; j++)
		{
			int index = yt[j] * this.nx;
			for (int i = 0; i < bx; i++)
			{
				sum = index + xt[i];
				neigh[i][j] = this.pixels[sum];
			}
		}
	}

	/**
	* An ImageAccess object calls this method for getting a
	* neighborhood of a predefined pattern around a selected pixel (x,y).
	* <br>The available patterns are:
	* <br>- a 3*3 block: PATTERN_SQUARE_3x3 (8-connect)
	* <br>- a 3*3 cross: PATTERN_CROSS_3x3  (4-connect)
	* <br>
	* <br>Mirror border conditions are applied.
	* <br>The pixel is arranged in a 1D array according the following rules:
	* <br>
	* <br>If the pattern is PATTERN_SQUARE_3x3  (8-connect)
	* <br>The pixel value of (x-1, y-1) are put into neigh[0]
	* <br>The pixel value of (x  , y-1) are put into neigh[1]
	* <br>The pixel value of (x+1, y-1) are put into neigh[2]
	* <br>The pixel value of (x-1, y  ) are put into neigh[3]
	* <br>The pixel value of (x  , y  ) are put into neigh[4]
	* <br>The pixel value of (x+1, y  ) are put into neigh[5]
	* <br>The pixel value of (x-1, y+1) are put into neigh[6]
	* <br>The pixel value of (x  , y+1) are put into neigh[7]
	* <br>The pixel value of (x+1, y+1) are put into neigh[8]
	* <br>
	* <br>If the pattern is PATTERN_CROSS_3x3   (4-connect)
	* <br>The pixel value of (x  , y-1) are put into neigh[0]
	* <br>The pixel value of (x-1, y  ) are put into neigh[1]
	* <br>The pixel value of (x  , y  ) are put into neigh[2]
	* <br>The pixel value of (x+1, y  ) are put into neigh[3]
	* <br>The pixel value of (x  , y+1) are put into neigh[4]
	* <br>
	* <br>The neigh should already created as a double array of 9 elements
	* for the PATTERN_SQUARE_3x3 or 5 elements for the PATTERN_CROSS_3x3.
	* 
	* @param x			x-coordinate of a selected central pixel
	* @param y			y-coordinate of a selected central pixel
	* @param neigh		output, an array consisting of 9 or 5 elements
	* @param pattern	PATTERN_SQUARE_3x3 or PATTERN_CROSS_3x3.
	*/
	public void getPattern(int x, int y, double neigh[], int pattern)
	{
		if (neigh == null) throw new ArrayStoreException(
				"getPattern: neigh == null.");
		switch (pattern)
		{
		case PATTERN_SQUARE_3x3:
			if (neigh.length != 9) throw new ArrayStoreException(
					"getPattern: neigh.length != 9.");
			getPatternSquare3x3(x, y, neigh);
			break;
		case PATTERN_CROSS_3x3:
			if (neigh.length != 5) throw new ArrayStoreException(
					"getPattern: neigh.length != 5");
			getPatternCross3x3(x, y, neigh);
			break;
		default:
			throw new ArrayStoreException("getPattern: unexpected pattern.");
		}
	}

	/**
	* An ImageAccess object calls this method for getting
	* a 3x3 neighborhood of 8-connected pixels around a selected pixel.
	*
	* @param x		input, the integer x-coordinate of a selected central pixel
	* @param y		input, the integer y-coordinate of a selected central pixel
	* @param neigh	output, an array consisting of 9 elements of the type double
	*/
	private void getPatternSquare3x3(int x, int y, double neigh[])
	{
		if (x >= 1) if (y >= 1) if (x < this.nx - 1) if (y < this.ny - 1)
		{
			int index = (y - 1) * this.nx + (x - 1);
			neigh[0] = this.pixels[index++];
			neigh[1] = this.pixels[index++];
			neigh[2] = this.pixels[index];
			index += (this.nx - 2);
			neigh[3] = this.pixels[index++];
			neigh[4] = this.pixels[index++];
			neigh[5] = this.pixels[index];
			index += (this.nx - 2);
			neigh[6] = this.pixels[index++];
			neigh[7] = this.pixels[index++];
			neigh[8] = this.pixels[index];
			return;
		}
		int x1 = x - 1;
		int x2 = x;
		int x3 = x + 1;
		int y1 = y - 1;
		int y2 = y;
		int y3 = y + 1;
		if (x == 0) x1 = x3;
		if (y == 0) y1 = y3;
		if (x == this.nx - 1) x3 = x1;
		if (y == this.ny - 1) y3 = y1;
		int offset = y1 * this.nx;
		neigh[0] = this.pixels[offset + x1];
		neigh[1] = this.pixels[offset + x2];
		neigh[2] = this.pixels[offset + x3];
		offset = y2 * this.nx;
		neigh[3] = this.pixels[offset + x1];
		neigh[4] = this.pixels[offset + x2];
		neigh[5] = this.pixels[offset + x3];
		offset = y3 * this.nx;
		neigh[6] = this.pixels[offset + x1];
		neigh[7] = this.pixels[offset + x2];
		neigh[8] = this.pixels[offset + x3];
	}

	/**
	* An ImageAccess object calls this method for getting
	* a 3x3 neighborhood of 4-connected pixels around a selected pixel.
	*
	* @param x		input, the integer x-coordinate of a selected central pixel
	* @param y		input, the integer y-coordinate of a selected central pixel
	* @param neigh	output, an array consisting of 5 elements of the type double
	*/
	private void getPatternCross3x3(int x, int y, double neigh[])
	{
		if (x >= 1) if (y >= 1) if (x < this.nx - 1) if (y < this.ny - 1)
		{
			int index = (y - 1) * this.nx + x;
			neigh[0] = this.pixels[index];
			index += (this.nx - 1);
			neigh[1] = this.pixels[index++];
			neigh[2] = this.pixels[index++];
			neigh[3] = this.pixels[index];
			index += (this.nx - 1);
			neigh[4] = this.pixels[index];
			return;
		}
		int x1 = x - 1;
		int x2 = x;
		int x3 = x + 1;
		int y1 = y - 1;
		int y2 = y;
		int y3 = y + 1;
		if (x == 0) x1 = x3;
		if (y == 0) y1 = y3;
		if (x == this.nx - 1) x3 = x1;
		if (y == this.ny - 1) y3 = y1;
		int offset = y1 * this.nx;
		neigh[0] = this.pixels[offset + x2];
		offset = y2 * this.nx;
		neigh[1] = this.pixels[offset + x1];
		neigh[2] = this.pixels[offset + x2];
		neigh[3] = this.pixels[offset + x3];
		offset = y3 * this.nx;
		neigh[4] = this.pixels[offset + x2];
	}

	/**
	* An ImageAccess object calls this method to get a sub-image 
	* with the upper left corner in the coordinate (x,y).
	*
	* The sub-image ouptut should be already created.
	*
	* @param x      x-coordinate in the source image
	* @param y      y-coordinate in the source image
	* @param output an ImageAccess object with the sub-image;
	*/
	public void getSubImage(int x, int y, ImageAccess output)
	{
		if (output == null) throw new ArrayStoreException(
				"getSubImage: output == null.");
		if (x < 0) throw new ArrayStoreException(
				"getSubImage: Incompatible image size");
		if (y < 0) throw new ArrayStoreException(
				"getSubImage: Incompatible image size");
		if (x >= this.nx) throw new ArrayStoreException(
				"getSubImage: Incompatible image size");
		if (y >= this.ny) throw new ArrayStoreException(
				"getSubImage: Incompatible image size");
		int nxcopy = output.getWidth();
		int nycopy = output.getHeight();
		double[][] neigh = new double[nxcopy][nycopy];
		int nx2 = (nxcopy - 1) / 2;
		int ny2 = (nycopy - 1) / 2;
		this.getNeighborhood(x + nx2, y + ny2, neigh);
		output.putArrayPixels(neigh);
	}

	/**
	* An ImageAccess object calls this method in order a value
	* of the gray level to be put to a position inside it
	* given by the coordinates.
	*
	* @param x		input, the integer x-coordinate of a pixel
	* @param y		input, the integer y-coordinate of a pixel
	* @param value	input, a value of the gray level of the type double
	*/
	public void putPixel(int x, int y, double value)
	{
		if (x < 0) return;
		if (x >= this.nx) return;
		if (y < 0) return;
		if (y >= this.ny) return;
		this.pixels[x + y * this.nx] = value;
	}

	/**
	* An ImageAccess object calls this method to put a whole 
	* column in a specified position into the image.
	*
	* @param x       	input, the integer x-coordinate of a column
	* @param column     input, an array of the type double
	*/
	public void putColumn(int x, double[] column)
	{
		if (x < 0) throw new IndexOutOfBoundsException("putColumn: x < 0.");
		if (x >= this.nx) throw new IndexOutOfBoundsException(
				"putColumn: x >= nx.");
		if (column == null) throw new ArrayStoreException(
				"putColumn: column == null.");
		if (column.length != this.ny) throw new ArrayStoreException(
				"putColumn: column.length != ny.");
		for (int i = 0; i < this.ny; i++)
		{
			this.pixels[x] = column[i];
			x += this.nx;
		}
	}

	/**
	* An ImageAccess object calls this method to put a part of column
	* into the image. The starting poisition in given by y and the ending
	* position is determined by the size of the column array.
	*
	* @param x       	input, the integer x-coordinate of a column
	* @param y       	input, the integer y-coordinate of a column
	* @param column     input, an array of the type double
	*/
	public void putColumn(int x, int y, double[] column)
	{
		if (x < 0) throw new IndexOutOfBoundsException("putColumn: x < 0.");
		if (x >= this.nx) throw new IndexOutOfBoundsException(
				"putColumn: x >= nx.");
		if (column == null) throw new ArrayStoreException(
				"putColumn: column == null.");
		int by = column.length;
		int index = y * this.nx + x;
		int top = 0;
		int bottom = 0;
		if (y >= 0)
		{
			if (y < this.ny - by) bottom = by;
			else bottom = -y + this.ny;
			for (int i = top; i < bottom; i++)
			{
				this.pixels[index] = column[i];
				index += this.nx;
			}
		}
		else
		{
			index = x;
			top = -y;
			if (y < this.ny - by) bottom = by;
			else bottom = -y + this.ny;
			for (int i = top; i < bottom; i++)
			{
				this.pixels[index] = column[i];
				index += this.nx;
			}
		}
	}

	/**
	* An ImageAccess object calls this method to put a whole 
	* row in a specified position into the image.
	*
	* @param y       input, the integer y-coordinate of a row
	* @param row     input, an array of the type double
	*/
	public void putRow(int y, double[] row)
	{
		if (y < 0) throw new IndexOutOfBoundsException("putRow: y < 0.");
		if (y >= this.ny) throw new IndexOutOfBoundsException("putRow: y >= ny.");
		if (row == null) throw new ArrayStoreException("putRow: row == null.");

		if (row.length != this.nx) throw new ArrayStoreException(
				"putRow: row.length != nx.");
		y *= this.nx;
		for (int i = 0; i < this.nx; i++)
		{
			this.pixels[y++] = row[i];
		}

	}

	/**
	* An ImageAccess object calls this method to put a part of row
	* into the image. The starting poisition in given by x and the ending
	* position is determined by the size of the row array.
	*
	* @param x       	input, the integer x-coordinate of a column
	* @param y       	input, the integer y-coordinate of a row
	* @param row		input, an array of the type double
	*/
	public void putRow(int x, int y, double[] row)
	{
		if (y < 0) throw new IndexOutOfBoundsException("putRow: y < 0.");
		if (y >= this.ny) throw new IndexOutOfBoundsException("putRow: y >= ny.");
		if (row == null) throw new ArrayStoreException("putRow: row == null.");
		int bx = row.length;
		int index = y * this.nx + x;
		int left = 0;
		int right = 0;
		if (x >= 0)
		{
			if (x < this.nx - bx) right = bx;
			else right = -x + this.nx;

			for (int i = left; i < right; i++)
			{
				this.pixels[index++] = row[i];
			}
		}
		else
		{
			index = y * this.nx;
			left = -x;

			if (x < this.nx - bx) right = bx;
			else right = -x + this.nx;

			for (int i = left; i < right; i++)
			{
				this.pixels[index++] = row[i];
			}
		}
	}

	/**
	* An ImageAccess object calls this method in order to put 
	* an 2D array of double in an ImageAccess.
	*
	* @param array		input, the double array
	*/
	public void putArrayPixels(double[][] array)
	{
		if (array == null) throw new IndexOutOfBoundsException(
				"putArrayPixels: array == null.");
		int bx = array.length;
		int by = array[0].length;
		if (bx * by != this.size) throw new IndexOutOfBoundsException(
				"putArrayPixels: imcompatible size.");
		int k = 0;
		for (int j = 0; j < by; j++)
			for (int i = 0; i < bx; i++)
				this.pixels[k++] = array[i][j];
	}

	/**
	* An ImageAccess object calls this method to put a sub-image 
	* with the upper left corner in the coordinate (x,y).
	*
	* The sub-image input should be already created.
	*
	* @param x      x-coordinate in the source image
	* @param y      y-coordinate in the source image
	* @param input  an ImageAccess object that we want to put;
	*/
	public void putSubImage(int x, int y, ImageAccess input)
	{
		if (input == null) throw new ArrayStoreException(
				"putSubImage: input == null.");
		if (x < 0) throw new IndexOutOfBoundsException("putSubImage: x < 0.");
		if (y < 0) throw new IndexOutOfBoundsException("putSubImage: y < 0.");
		if (x >= this.nx) throw new IndexOutOfBoundsException(
				"putSubImage: x >= nx.");
		if (y >= this.ny) throw new IndexOutOfBoundsException(
				"putSubImage: y >= ny.");
		int nxcopy = input.getWidth();
		int nycopy = input.getHeight();
		// Reduces the size of the area to copy if it is too large
		if (x + nxcopy > this.nx) nxcopy = this.nx - x;
		if (y + nycopy > this.ny) nycopy = this.ny - y;
		// Copies lines per lines
		double[] dsrc = input.getPixels();
		for (int j = 0; j < nycopy; j++)
			System.arraycopy(dsrc, j * nxcopy, this.pixels, (j + y) * this.nx + x,
					nxcopy);
	}

	/**
	* An ImageAccess object calls this method to set a constant
	* value to all pixels of the image.
	*
	* @param constant a constant value 
	*/
	public void setConstant(double constant)
	{
		for (int k = 0; k < this.size; k++)
			this.pixels[k] = constant;
	}

	/**
	* Stretches the contrast inside an image so that the gray levels 
	* are in the range 0 to 255.
	*/
	public void normalizeContrast()
	{
		double minGoal = 0.0;
		double maxGoal = 255.0;
		// Search the min and max
		double minImage = getMinimum();
		double maxImage = getMaximum();
		// Compute the parameter to rescale the gray levels
		double a;
		if (minImage - maxImage == 0)
		{
			a = 1.0;
			minImage = (maxGoal - minGoal) / 2.0;
		}
		else a = (maxGoal - minGoal) / (maxImage - minImage);
		for (int i = 0; i < this.size; i++)
		{
			this.pixels[i] = (float) (a * (this.pixels[i] - minImage) + minGoal);
		}
	}

	/**
	* Display an image at a specific position (x, y).
	*
	* @param title  a string for the title
	* @param loc   	Point for the location
	*/
	public void show(String title, java.awt.Point loc)
	{
		FloatProcessor fp = createFloatProcessor();
		fp.resetMinAndMax();
		ImagePlus impResult = new ImagePlus(title, fp);
		impResult.show();
		ij.gui.ImageWindow window = impResult.getWindow();
		window.setLocation(loc.x, loc.y);
		impResult.show();
	}

	/**
	* Display an image.
	*
	* @param title   a string for the title of the window
	*/
	public void show(String title)
	{
		FloatProcessor fp = createFloatProcessor();
		fp.resetMinAndMax();
		ImagePlus impResult = new ImagePlus(title, fp);
		impResult.show();
	}

	/**
	* Compute the absolute value.
	*/
	public void abs()
	{
		for (int k = 0; k < this.size; k++)
			this.pixels[k] = Math.abs(this.pixels[k]);
	}

	/**
	* Compute the square root of an ImageAccess.
	*/
	public void sqrt()
	{
		for (int k = 0; k < this.size; k++)
		{
			this.pixels[k] = Math.sqrt(this.pixels[k]);
		}
	}

	/**
	* Raised an ImageAccess object to the power a.
	*
	* @param a 	input
	*/
	public void pow(final double a)
	{
		for (int k = 0; k < this.size; k++)
		{
			this.pixels[k] = Math.pow(this.pixels[k], a);
		}
	}

	/**
	* An ImageAccess object calls this method for adding
	* a constant to each pixel. 
	*
	* @param constant   a constant to be added
	*/
	public void add(double constant)
	{
		for (int k = 0; k < this.size; k++)
			this.pixels[k] += constant;
	}

	/**
	* An ImageAccess object calls this method for multiplying
	* a constant to each pixel. 
	*
	* @param constant   a constant to be multiplied
	*/
	public void multiply(final double constant)
	{
		for (int k = 0; k < this.size; k++)
			this.pixels[k] *= constant;
	}

	/**
	* An ImageAccess object calls this method for adding
	* a constant to each pixel. 
	*
	* @param constant   a constant to be subtracted
	*/
	public void subtract(final double constant)
	{
		for (int k = 0; k < this.size; k++)
			this.pixels[k] -= constant;
	}

	/**
	* An ImageAccess object calls this method for dividing
	* a constant to each pixel. 
	*
	* @param constant   a constant to be divided
	*/
	public void divide(final double constant)
	{
		if (constant == 0.0) throw new ArrayStoreException("divide: Divide by 0");
		for (int k = 0; k < this.size; k++)
			this.pixels[k] /= constant;
	}

	/**
	* An ImageAccess object calls this method for adding
	* two ImageAccess objects.
	*
	* [this = im1 + im2]
	*
	* The resulting ImageAccess and the two operands should have 
	* the same size.
	*
	* @param im1		an ImageAccess object to be added
	* @param im2		an ImageAccess object to be added
	*/
	public void add(ImageAccess im1, ImageAccess im2)
	{
		if (im1.getWidth() != this.nx) throw new ArrayStoreException(
				"add: incompatible size.");
		if (im1.getHeight() != this.ny) throw new ArrayStoreException(
				"add: incompatible size.");
		if (im2.getWidth() != this.nx) throw new ArrayStoreException(
				"add: incompatible size.");
		if (im2.getHeight() != this.ny) throw new ArrayStoreException(
				"add: incompatible size.");
		double[] doubleOperand1 = im1.getPixels();
		double[] doubleOperand2 = im2.getPixels();
		for (int k = 0; k < this.size; k++)
			this.pixels[k] = doubleOperand1[k] + doubleOperand2[k];
	}

	/**
	* An ImageAccess object calls this method for multiplying
	* two ImageAccess objects.
	*
	* The resulting ImageAccess and the two operands should have 
	* the same size.
	*
	* [this = im1 * im2]
	*
	* @param im1		an ImageAccess object to be multiplied
	* @param im2		an ImageAccess object to be multiplied
	*/

	public void multiply(ImageAccess im1, ImageAccess im2)
	{
		if (im1.getWidth() != this.nx) throw new ArrayStoreException(
				"multiply: incompatible size.");
		if (im1.getHeight() != this.ny) throw new ArrayStoreException(
				"multiply: incompatible size.");
		if (im2.getWidth() != this.nx) throw new ArrayStoreException(
				"multiply: incompatible size.");
		if (im2.getHeight() != this.ny) throw new ArrayStoreException(
				"multiply: incompatible size.");
		double[] doubleOperand1 = im1.getPixels();
		double[] doubleOperand2 = im2.getPixels();
		for (int k = 0; k < this.size; k++)
			this.pixels[k] = doubleOperand1[k] * doubleOperand2[k];
	}

	/**
	* An ImageAccess object calls this method for subtracting
	* two ImageAccess objects.
	*
	* The resulting ImageAccess and the two operands should have 
	* the same size.
	*
	* [this = im1 - im2]
	*
	* @param im1		an ImageAccess object to be subtracted
	* @param im2		an ImageAccess object to be subtracted
	*/

	public void subtract(ImageAccess im1, ImageAccess im2)
	{
		if (im1.getWidth() != this.nx) throw new ArrayStoreException(
				"subtract: incompatible size.");
		if (im1.getHeight() != this.ny) throw new ArrayStoreException(
				"subtract: incompatible size.");
		if (im2.getWidth() != this.nx) throw new ArrayStoreException(
				"subtract: incompatible size.");
		if (im2.getHeight() != this.ny) throw new ArrayStoreException(
				"subtract: incompatible size.");
		double[] doubleOperand1 = im1.getPixels();
		double[] doubleOperand2 = im2.getPixels();
		for (int k = 0; k < this.size; k++)
			this.pixels[k] = doubleOperand1[k] - doubleOperand2[k];
	}

	/**
	* An ImageAccess object calls this method for dividing
	* two ImageAccess objects.
	*
	* [this = im1 / im2]
	*
	* The resulting ImageAccess and the two operands should have 
	* the same size.
	*
	* @param im1		numerator
	* @param im2		denominator
	*/

	public void divide(ImageAccess im1, ImageAccess im2)
	{
		if (im1.getWidth() != this.nx) throw new ArrayStoreException(
				"divide: incompatible size.");
		if (im1.getHeight() != this.ny) throw new ArrayStoreException(
				"divide: incompatible size.");
		if (im2.getWidth() != this.nx) throw new ArrayStoreException(
				"divide: incompatible size.");
		if (im2.getHeight() != this.ny) throw new ArrayStoreException(
				"divide: incompatible size.");
		double[] doubleOperand1 = im1.getPixels();
		double[] doubleOperand2 = im2.getPixels();
		for (int k = 0; k < this.size; k++)
			this.pixels[k] = doubleOperand1[k] / doubleOperand2[k];
	}

}
