// =====================================================================================
//
// Project:
// Resize ImageJ plugin
// 
// Authors:
// Arrate Munoz, David Leroux, Daniel Sage
// Biomedical Imaging Group (BIG)
// Ecole Polytechnique Federale de Lausanne (EPFL)
// Lausanne, Switzerland
//
// Date:
// 11 July 2001
//
// Reference:
// A. Muoz Barrutia, T. Blu, M. Unser
// Least-Squares Image Resizing Using Finite Differences
// IEEE Transactions on Image Processing, vol. 10, no. 9, pp. 1365-1378,
// September 2001.
//
// Conditions of use:
// You'll be free to use this software for research purposes, but you
// should not redistribute it without our consent. In addition, we
// expect you to include a citation or acknowledgment whenever
// you present or publish results that are based on it.
//
// Permission for use in JIBS granted by Daniel Sage on 13 April 2011 under the
// following terms:
// You include the Resize in your distribution if you clearly mention the
// credits and put a reference on the scientific paper:
// 
// Credit:
// Resize: An ImageJ plugin to resize an image using high-quality interpolation
// Written by Arrate Muoz, David Leroux, Daniel Sage and Michael Unser at the
// Biomedical Image Group (BIG), EPFL, Switzerland
// 
// Reference
// A. Muoz Barrutia, T. Blu, M. Unser,
// "Least-Squares Image Resizing Using Finite Differences," IEEE Transactions on
// Image Processing, vol. 10, no. 9, pp. 1365-1378, September 2001.
//
// =====================================================================================

package ch.epfl.bigwww;

import ij.IJ;
import ij.ImagePlus;
import ij.ImageStack;
import ij.gui.GUI;
import ij.plugin.filter.PlugInFilter;
import ij.process.ColorProcessor;
import ij.process.ImageProcessor;
import ij.process.ShortProcessor;

import java.awt.Button;
import java.awt.Choice;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Label;
import java.awt.List;
import java.awt.Point;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

public class ResizeExpert_ implements PlugInFilter, ActionListener,
		ItemListener
{

	private ImagePlus imp = null; // Reference to the input image
	private ImageStack stack; // Handler to ImageAccess
	private double zoomY;
	private double zoomX;
	private double shiftY;
	private double shiftX;
	private int interpDegree;
	private int analyDegree;
	private int syntheDegree;
	private boolean inversable = false;
	private int[] size = new int[4];

	private int xsize;
	private int ysize;
	private String method = "";
	private boolean dialogOK = true;

	/*
	* Implementation of the "run" method of the PlugInFilter class. 
	*/
	public void run(ImageProcessor ip)
	{

		// Processing
		double t = System.currentTimeMillis();
		Resize resize = new Resize();

		if (this.imp.getType() == ImagePlus.COLOR_RGB)
		{
			ImageAccess out[] = new ImageAccess[3];
			for (int c = 0; c < 3; c++)
			{
				ImageAccess in = new ImageAccess((ColorProcessor) ip, c);
				out[c] = new ImageAccess(this.size[3], this.size[2]);
				resize.computeZoom(in, out[c], this.analyDegree, this.syntheDegree,
						this.interpDegree, this.zoomY, this.zoomX, 0, 0, false);
			}
			byte[] r = (byte[]) out[0].createByteProcessor().getPixels();
			byte[] g = (byte[]) out[1].createByteProcessor().getPixels();
			byte[] b = (byte[]) out[2].createByteProcessor().getPixels();
			ColorProcessor cp = new ColorProcessor(this.size[3], this.size[2]);
			cp.setRGB(r, g, b);
			this.stack.addSlice("", cp);
		}
		else
		{
			ImageAccess in = new ImageAccess(ip);
			ImageAccess out = new ImageAccess(this.xsize, this.ysize);
			resize.computeZoom(in, out, this.analyDegree, this.syntheDegree,
					this.interpDegree, this.zoomY, this.zoomX, 0, 0, false);
			switch (this.imp.getType())
			{
			case ImagePlus.GRAY8:
				this.stack.addSlice("", out.createByteProcessor());
				break;
			case ImagePlus.GRAY16:
				this.stack.addSlice("", createShortProcessor(out));
				break;
			case ImagePlus.GRAY32:
				this.stack.addSlice("", out.createFloatProcessor());
				break;
			}
		}
		IJ.showStatus("Time of resizing "
				+ IJ.d2s(System.currentTimeMillis() - t) + " ms");

		// Display
		if (this.stack.getSize() == this.imp.getStack().getSize())
		{
			ImagePlus impResult = new ImagePlus(this.imp.getTitle(), this.stack);
			impResult.updateAndDraw();
			impResult.show();
		}
	}

	/**
	*/
	void showAbout()
	{
		IJ.showMessage("Zoom In Java",
				"Compute a resized version of an input image \n"
						+ "using either PROJECTION or STANDARD method\n" + "\n"
						+ "Swiss Federal Institute of Technology Lausanne (EPFL) \n"
						+ "Biomedical Imaging Group\n");
	}

	/**
	* Implementation of the "setup" method of the PlugInFilter class.
	*/
	public int setup(String arg, ImagePlus impIn)
	{
		if (IJ.versionLessThan("1.21a")) return (DONE);

		this.imp = impIn;
		if (arg.equals("about"))
		{
			showAbout();
			return DONE;
		}

		if (this.imp == null)
		{
			IJ.error("Input image required");
			return (DONE);
		}

		if (checkSizeImage() == false) return (DONE);

		doDialog();
		if (!this.dialogOK) return (DONE);

		if (checkParameters() == false) return (DONE);

		int nx = this.imp.getWidth();
		int ny = this.imp.getHeight();

		int[] size_local = Resize.calculatefinalsize(this.inversable, ny, nx, this.zoomY, this.zoomX);
		//int slices = imp.getStackSize();
		//int type = imp.getType();

		if (size_local[0] < 1)
		{
			IJ.error("X-Scale  too small.");
			return (DONE);
		}

		if (size_local[1] < 1)
		{
			IJ.error("Y-Scale  too small.");
			return (DONE);
		}
		this.stack = new ImageStack(size_local[3], size_local[2]);
		this.xsize = size_local[3]; // Workaround bug 25-05-2009
		this.ysize = size_local[2]; // Workaround bug 25-05-2009
		return DOES_ALL + DOES_STACKS;
	}

	/**
	* Check the size image.
	*
	* Process only the black and white image and no stacks.
	*/
	private boolean checkSizeImage()
	{
		int nx = this.imp.getProcessor().getWidth();
		int ny = this.imp.getProcessor().getHeight();
		if (nx <= 7)
		{
			IJ.error("Size X too small.");
			return false;
		}
		if (ny <= 7)
		{
			IJ.error("Size Y too small.");
			return false;
		}

		return true;
	}

	/**
	* Check the parameters.
	*/
	private boolean checkParameters()
	{
		if (this.zoomY <= 0.0)
		{
			IJ.error("X-Scale  too small.");
			return false;
		}

		if (this.zoomX < 0.0)
		{
			IJ.error("Y-Scale  too small.");
			return false;
		}

		if (Math.abs(this.shiftY) >= 0.5)
		{
			IJ.error("X-Shift  too large.");
			return false;
		}

		if (Math.abs(this.shiftX) >= 0.5)
		{
			IJ.error("Y-Shift  too large.");
			return false;
		}

		return true;
	}

	/**
	* Build the dialog box.
	*/
	private Dialog dlg = new Dialog(new Frame(), "Resize Expert");
	private List lstMethod = new List();
	private Choice chcInversable = new Choice();
	private Choice chcInterpolation = new Choice();
	private Choice chcAnalysis = new Choice();
	private Choice chcSynthesis = new Choice();
	private Button bnOK = new Button();
	private Button bnCancel = new Button();
	private Label lblMethod = new Label();
	private Label lblInversable = new Label();
	private Label lblInterpolation = new Label();
	private Label lblAnalysis = new Label();
	private Label lblSynthesis = new Label();
	private Label lblZoom = new Label();
	private Label lblzoomY = new Label();
	private Label lblzoomX = new Label();
	private Label lblShift = new Label();
	private Label lblshiftY = new Label();
	private Label lblshiftX = new Label();
	private TextField txtzoomY = new TextField();
	private TextField txtzoomX = new TextField();
	private TextField txtshiftY = new TextField();
	private TextField txtshiftX = new TextField();

	private void doDialog()
	{
		this.lblMethod.setLocation(new Point(150, 30));
		this.lblMethod.setSize(new Dimension(60, 20));
		this.lblMethod.setVisible(true);
		this.lblMethod.setText("Method");
		this.dlg.add(this.lblMethod);

		this.lstMethod.setLocation(new Point(150, 50));
		this.lstMethod.setSize(new Dimension(120, 60));
		this.lstMethod.setVisible(true);
		this.lstMethod.add("Projection");
		this.lstMethod.add("Standard");
		this.lstMethod.select(0);
		this.dlg.add(this.lstMethod);
		this.lstMethod.addItemListener(this);

		this.lblInversable.setLocation(new Point(20, 270));
		this.lblInversable.setSize(new Dimension(100, 20));
		this.lblInversable.setVisible(true);
		this.lblInversable.setText("Inversable");
		this.dlg.add(this.lblInversable);

		this.chcInversable.setLocation(new Point(20, 290));
		this.chcInversable.setSize(new Dimension(60, 20));
		this.chcInversable.setVisible(true);
		this.chcInversable.addItem("Yes");
		this.chcInversable.addItem("No");
		this.chcInversable.select(1);
		this.dlg.add(this.chcInversable);

		this.lblInterpolation.setLocation(new Point(150, 120));
		this.lblInterpolation.setSize(new Dimension(150, 20));
		this.lblInterpolation.setVisible(true);
		this.lblInterpolation.setText("Interpolation degree");
		this.dlg.add(this.lblInterpolation);

		this.chcInterpolation.setLocation(new Point(150, 140));
		this.chcInterpolation.setSize(new Dimension(60, 20));
		this.chcInterpolation.setVisible(true);
		this.chcInterpolation.addItem("0");
		this.chcInterpolation.addItem("1");
		this.chcInterpolation.addItem("2");
		this.chcInterpolation.addItem("3");
		this.chcInterpolation.select(0);
		this.dlg.add(this.chcInterpolation);

		this.lblAnalysis.setLocation(new Point(150, 170));
		this.lblAnalysis.setSize(new Dimension(150, 20));
		this.lblAnalysis.setVisible(true);
		this.lblAnalysis.setText("Analysis degree");
		this.dlg.add(this.lblAnalysis);

		this.chcAnalysis.setLocation(new Point(150, 190));
		this.chcAnalysis.setSize(new Dimension(60, 20));
		this.chcAnalysis.setVisible(true);
		this.chcAnalysis.addItem("0");
		this.chcAnalysis.addItem("1");
		this.chcAnalysis.addItem("2");
		this.chcAnalysis.addItem("3");
		this.chcAnalysis.select(0);
		this.dlg.add(this.chcAnalysis);

		this.lblSynthesis.setLocation(new Point(150, 220));
		this.lblSynthesis.setSize(new Dimension(150, 20));
		this.lblSynthesis.setVisible(true);
		this.lblSynthesis.setText("Synthesis degree");
		this.dlg.add(this.lblSynthesis);

		this.chcSynthesis.setLocation(new Point(150, 240));
		this.chcSynthesis.setSize(new Dimension(60, 20));
		this.chcSynthesis.setVisible(true);
		this.chcSynthesis.addItem("0");
		this.chcSynthesis.addItem("1");
		this.chcSynthesis.addItem("2");
		this.chcSynthesis.addItem("3");
		this.chcSynthesis.select(0);
		this.dlg.add(this.chcSynthesis);

		this.lblZoom.setLocation(new Point(20, 30));
		this.lblZoom.setSize(new Dimension(150, 20));
		this.lblZoom.setVisible(true);
		this.lblZoom.setText("Zoom factor");
		this.dlg.add(this.lblZoom);

		this.lblzoomX.setLocation(new Point(20, 50));
		this.lblzoomX.setSize(new Dimension(150, 20));
		this.lblzoomX.setVisible(true);
		this.lblzoomX.setText("X-Scale");
		this.dlg.add(this.lblzoomX);

		this.txtzoomX.setLocation(new Point(20, 70));
		this.txtzoomX.setSize(new Dimension(100, 20));
		this.txtzoomX.setVisible(true);
		this.txtzoomX.setText("0.5");
		this.dlg.add(this.txtzoomX);

		this.lblzoomY.setLocation(new Point(20, 100));
		this.lblzoomY.setSize(new Dimension(150, 20));
		this.lblzoomY.setVisible(true);
		this.lblzoomY.setText("Y-Scale");
		this.dlg.add(this.lblzoomY);

		this.txtzoomY.setLocation(new Point(20, 120));
		this.txtzoomY.setSize(new Dimension(100, 20));
		this.txtzoomY.setVisible(true);
		this.txtzoomY.setText("0.5");
		this.dlg.add(this.txtzoomY);

		this.lblShift.setLocation(new Point(20, 150));
		this.lblShift.setSize(new Dimension(150, 20));
		this.lblShift.setVisible(true);
		this.lblShift.setText("Shift value");
		this.dlg.add(this.lblShift);

		this.lblshiftX.setLocation(new Point(20, 170));
		this.lblshiftX.setSize(new Dimension(150, 20));
		this.lblshiftX.setVisible(true);
		this.lblshiftX.setText("X-Scale");
		this.dlg.add(this.lblshiftX);

		this.txtshiftX.setLocation(new Point(20, 190));
		this.txtshiftX.setSize(new Dimension(100, 20));
		this.txtshiftX.setVisible(true);
		this.txtshiftX.setText("0.0");
		this.dlg.add(this.txtshiftX);

		this.lblshiftY.setLocation(new Point(20, 220));
		this.lblshiftY.setSize(new Dimension(150, 20));
		this.lblshiftY.setVisible(true);
		this.lblshiftY.setText("Y-Scale");
		this.dlg.add(this.lblshiftY);

		this.txtshiftY.setLocation(new Point(20, 240));
		this.txtshiftY.setSize(new Dimension(100, 20));
		this.txtshiftY.setVisible(true);
		this.txtshiftY.setText("0.0");
		this.dlg.add(this.txtshiftY);

		this.bnCancel.setLocation(new Point(20, 340));
		this.bnCancel.setSize(new Dimension(100, 30));
		this.bnCancel.setVisible(true);
		this.bnCancel.setLabel("Cancel");
		this.dlg.add(this.bnCancel);
		this.bnCancel.addActionListener(this);

		this.bnOK.setLocation(new Point(180, 340));
		this.bnOK.setSize(new Dimension(100, 30));
		this.bnOK.setVisible(true);
		this.bnOK.setLabel("OK");
		this.dlg.add(this.bnOK);
		this.bnOK.addActionListener(this);

		/*		dlg.setModal(true);
				dlg.pack();
				//dlg.setResizable(false);
				GUI.center(dlg);
				dlg.setVisible(true);
				IJ.wait(250); 	// work around for Sun/WinNT bug

		*/
		this.dlg.setSize(new Dimension(300, 390));
		this.dlg.setLayout(null);
		this.dlg.setModal(true);
		this.dlg.setResizable(false);
		GUI.center(this.dlg);
		this.dlg.setVisible(true);

	}

	/**
	 * Implements the actionPerformed for the ActionListener.
	 */
	public synchronized void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == this.bnCancel)
		{
			this.dialogOK = false;
			this.dlg.dispose();
		}
		else if (e.getSource() == this.bnOK)
		{
			this.dialogOK = true;
			this.interpDegree = getIntegerFromString(this.chcInterpolation
					.getSelectedItem());
			this.analyDegree = getIntegerFromString(this.chcAnalysis
					.getSelectedItem());
			this.syntheDegree = getIntegerFromString(this.chcSynthesis
					.getSelectedItem());

			if (this.lstMethod.getSelectedItem() == "Standard")
			{
				this.analyDegree = -1;
				this.syntheDegree = this.interpDegree;
			}
			else
			{
				if (this.analyDegree > this.syntheDegree) this.analyDegree = this.syntheDegree;
			}

			this.zoomY = getDoubleValue(this.txtzoomY, Double.MIN_VALUE, 0.5,
					Double.MAX_VALUE);
			this.zoomX = getDoubleValue(this.txtzoomX, Double.MIN_VALUE, 0.5,
					Double.MAX_VALUE);
			this.shiftY = getDoubleValue(this.txtshiftY, Double.MIN_VALUE, 0.0,
					Double.MAX_VALUE);
			this.shiftX = getDoubleValue(this.txtshiftX, Double.MIN_VALUE, 0.0,
					Double.MAX_VALUE);

			this.inversable = ((this.chcInversable.getSelectedItem())
					.equals("Yes"));
			this.dlg.setVisible(false);
			this.dlg.dispose();
		}
		notify();
	}

	/**
	 * Implements the actionPerformed for the ItemListener.
	 */
	public synchronized void itemStateChanged(ItemEvent e)
	{
		this.method = this.lstMethod.getSelectedItem();
		if ((this.method == "Projection"))
		{
			this.chcAnalysis.setEnabled(true);
			this.chcSynthesis.setEnabled(true);
		}
		else
		{
			this.chcAnalysis.setEnabled(false);
			this.chcSynthesis.setEnabled(false);
		}
	}

	/*
	* Get a double value from a TextField between minimal and maximal values.
	*/
	private double getDoubleValue(TextField text, double mini, double defaut,
			double maxi)
	{
		double d;
		try
		{
			d = (new Double(text.getText())).doubleValue();
			if (d < mini) text.setText("" + mini);
			if (d > maxi) text.setText("" + maxi);
		}

		catch (Exception e)
		{
			if (e instanceof NumberFormatException) text.setText("" + defaut);
		}
		d = (new Double(text.getText())).doubleValue();
		return d;
	}

	/*
	* Get a integer value from a string between minimal and maximal values.
	*/
	private int getIntegerFromString(String value)
	{
		int d;
		try
		{
			d = (new Integer(value)).intValue();
		}

		catch (Exception e)
		{
			if (e instanceof NumberFormatException) value = "3";
		}
		d = (new Integer(value)).intValue();
		return d;
	}

	/**
	*/
	private ShortProcessor createShortProcessor(ImageAccess image)
	{
		double pixels[] = image.getPixels();

		int nx = this.imp.getWidth();
		int ny = this.imp.getHeight();
		ShortProcessor sp = new ShortProcessor(nx, ny);
		short[] ssrc = new short[pixels.length];
		double p;
		for (int k = 0; k < pixels.length; k++)
		{
			p = pixels[k];
			if (p < 0) p = 0.0;
			if (p > Short.MAX_VALUE) p = Short.MAX_VALUE;
			ssrc[k] = (short) p;
		}
		sp.setPixels(ssrc);
		return sp;
	}

}
