// =====================================================================================
//
// Project:
// Resize ImageJ plugin
// 
// Authors:
// Arrate Munoz, David Leroux, Daniel Sage
// Biomedical Imaging Group (BIG)
// Ecole Polytechnique Federale de Lausanne (EPFL)
// Lausanne, Switzerland
//
// Date:
// 11 July 2001
//
// Reference:
// A. Muoz Barrutia, T. Blu, M. Unser
// Least-Squares Image Resizing Using Finite Differences
// IEEE Transactions on Image Processing, vol. 10, no. 9, pp. 1365-1378,
// September 2001.
//
// Conditions of use:
// You'll be free to use this software for research purposes, but you
// should not redistribute it without our consent. In addition, we
// expect you to include a citation or acknowledgment whenever
// you present or publish results that are based on it.
//
// Permission for use in JIBS granted by Daniel Sage on 13 April 2011 under the
// following terms:
// You include the Resize in your distribution if you clearly mention the
// credits and put a reference on the scientific paper:
// 
// Credit:
// Resize: An ImageJ plugin to resize an image using high-quality interpolation
// Written by Arrate Muoz, David Leroux, Daniel Sage and Michael Unser at the
// Biomedical Image Group (BIG), EPFL, Switzerland
// 
// Reference
// A. Muoz Barrutia, T. Blu, M. Unser,
// "Least-Squares Image Resizing Using Finite Differences," IEEE Transactions on
// Image Processing, vol. 10, no. 9, pp. 1365-1378, September 2001.

// =====================================================================================

package ch.epfl.bigwww;

import ij.IJ;
import ij.ImagePlus;
import ij.ImageStack;
import ij.gui.GUI;
import ij.plugin.filter.PlugInFilter;
import ij.process.ColorProcessor;
import ij.process.ImageProcessor;
import ij.process.ShortProcessor;

import java.awt.Button;
import java.awt.Checkbox;
import java.awt.Choice;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Label;
import java.awt.Panel;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.text.DecimalFormat;

public class Resize_ implements PlugInFilter, ActionListener
{
	private boolean dialogOK = true; // True if the user choose OK, false otherwise
	private ImagePlus imp; // Reference to the input image
	private ImageStack stack; // Handler to the output stack
	private double sizeX, sizeY; // Request size of the image in the dialog box
	private double zoomX, zoomY;
	private int interpDegree;
	private int analyDegree;
	private int syntheDegree;
	boolean UnitPixelsX = true;
	boolean UnitPixelsY = true;
	boolean constraintRatio = true;
	private String interpolation = ""; // Cubic or Linear
	private String method = ""; // Interpolation or Least-Squares

	int nx; // Size of the input image
	int ny;

	/**
	* Implementation of the "run" method of the PlugInFilter class. 
	*/
	public void run(ImageProcessor ip)
	{
		this.interpDegree = 3;
		this.syntheDegree = 3;
		this.analyDegree = 3;

		// ---------------------------------------
		// To work around the bug on large image
		int nx_local = this.imp.getWidth();
		int ny_local = this.imp.getHeight();
		if (nx_local > 3000) this.analyDegree = 1;
		if (ny_local > 3000) this.analyDegree = 1;
		// ---------------------------------------
		if (this.interpolation.equalsIgnoreCase("Linear"))
		{
			this.interpDegree = 1;
			this.syntheDegree = 1;
			this.analyDegree = 1;
		}

		if (this.method.equalsIgnoreCase("Interpolation"))
		{
			this.analyDegree = -1;
		}

		Resize resize = new Resize();

		if (this.imp.getType() == ImagePlus.COLOR_RGB)
		{
			ImageAccess out[] = new ImageAccess[3];
			for (int c = 0; c < 3; c++)
			{
				ImageAccess in = new ImageAccess((ColorProcessor) ip, c);
				out[c] = new ImageAccess((int) this.sizeX, (int) this.sizeY);
				resize.computeZoom(in, out[c], this.analyDegree, this.syntheDegree,
						this.interpDegree, this.zoomY, this.zoomX, 0, 0, false);
			}
			byte[] r = (byte[]) out[0].createByteProcessor().getPixels();
			byte[] g = (byte[]) out[1].createByteProcessor().getPixels();
			byte[] b = (byte[]) out[2].createByteProcessor().getPixels();
			ColorProcessor cp = new ColorProcessor((int) this.sizeX,
					(int) this.sizeY);
			cp.setRGB(r, g, b);
			this.stack.addSlice("", cp);
		}
		else
		{
			ImageAccess in = new ImageAccess(ip);
			ImageAccess out = new ImageAccess((int) this.sizeX, (int) this.sizeY);
			resize.computeZoom(in, out, this.analyDegree, this.syntheDegree,
					this.interpDegree, this.zoomY, this.zoomX, 0, 0, false);
			switch (this.imp.getType())
			{
			case ImagePlus.GRAY8:
				this.stack.addSlice("", out.createByteProcessor());
				break;
			case ImagePlus.GRAY16:
				this.stack.addSlice("", createShortProcessor(out));
				break;
			case ImagePlus.GRAY32:
				this.stack.addSlice("", out.createFloatProcessor());
				break;
			}
		}

		// Display
		if (this.stack.getSize() == this.imp.getStack().getSize())
		{
			ImagePlus impResult = new ImagePlus(this.imp.getTitle(), this.stack);
			impResult.updateAndDraw();
			impResult.show();
		}
	}

	/**
	* Implementation of the "setup" method of the PlugInFilter class.
	*/
	public int setup(String arg, ImagePlus impIn)
	{
		if (IJ.versionLessThan("1.21a")) return (DONE);

		this.imp = impIn;
		if (arg.equals("about"))
		{
			showAbout();
			return DONE;
		}

		if (this.imp == null)
		{
			IJ.error("Input image required");
			return (DONE);
		}

		if (checkSizeImage() == false) return (DONE);

		doDialog();
		if (!this.dialogOK) return (DONE);

		if (checkParameters() == false) return (DONE);

		int nx_local = this.imp.getWidth();
		int ny_local = this.imp.getHeight();

		if (this.UnitPixelsX == true)
		{
			this.zoomX = this.sizeX / nx_local; // sizeX is in pixels 
		}
		else
		{
			this.zoomX = this.sizeX / 100.0; // sizeX is in percent 
		}

		if (this.UnitPixelsY == true)
		{
			this.zoomY = this.sizeY / ny_local; // sizeY is in pixels 
		}
		else
		{
			this.zoomY = this.sizeY / 100.0; // sizeY is in percent 
		}

		this.sizeX = (int) Math.round(nx_local * this.zoomX);
		this.sizeY = (int) Math.round(ny_local * this.zoomY);

		this.stack = new ImageStack((int) this.sizeX, (int) this.sizeY);

		return DOES_8G + DOES_16 + DOES_32 + DOES_RGB + DOES_STACKS;
	}

	/**
	* Implementation of the "showAbout" method of the PlugInFilter class.
	*/
	public void showAbout()
	{
		IJ
				.showMessage(
						"About Resize_...",
						"Compute a resized version of an input image \n"
								+ "using either interpolation or least-squares approximation method\n"
								+ "\n"
								+ "Swiss Federal Institute of Technology Lausanne (EPFL) \n"
								+ "Biomedical Imaging Group\n");
	}

	/**
	* Check the size image.
	*
	* Process only the black and white image and no stacks.
	*/
	private boolean checkSizeImage()
	{
		this.nx = this.imp.getProcessor().getWidth();
		this.ny = this.imp.getProcessor().getHeight();

		if (this.nx <= 3)
		{
			IJ.error("Size X too small.");
			return false;
		}
		if (this.ny <= 3)
		{
			IJ.error("Size Y too small.");
			return false;
		}

		return true;
	}

	/**
	* Check the parameters.
	*/
	private boolean checkParameters()
	{
		if (this.sizeY <= 3)
		{
			IJ.error("Final Height too small.");
			return false;
		}
		if (this.sizeX <= 3)
		{
			IJ.error("Final Width too small.");
			return false;
		}
		return true;
	}

	/**
	* Build the dialog box.
	*/
	private Dialog dlg = new Dialog(new Frame(), "Resize");
	Button bnOK = new Button();
	private Button bnCancel = new Button();
	TextField txtXScale = new TextField(10);
	TextField txtYScale = new TextField(10);
	Choice choiceUnitX = new Choice();
	Choice choiceUnitY = new Choice();
	private Choice choiceInterpolation = new Choice();
	private Choice choiceMethod = new Choice();
	Checkbox chkAspectRatio;

	private void doDialog()
	{
		GridBagLayout grid = new GridBagLayout();
		GridBagConstraints c = new GridBagConstraints();

		this.dlg.setLayout(grid);

		this.txtXScale.setText("" + this.nx / 2);
		this.txtXScale.setColumns(10);
		this.txtYScale.setText("" + this.ny / 2);
		this.txtYScale.setColumns(10);

		this.choiceUnitX.add("Pixels");
		this.choiceUnitX.add("Percent");
		this.choiceUnitX.select("Pixels");

		this.choiceUnitY.add("Pixels");
		this.choiceUnitY.add("Percent");
		this.choiceUnitY.select("Pixels");

		Panel pnX = new Panel();
		pnX.setLayout(new FlowLayout(FlowLayout.CENTER, 5, 0));
		pnX.add(this.txtXScale);
		pnX.add(this.choiceUnitX);

		Panel pnY = new Panel();
		pnY.setLayout(new FlowLayout(FlowLayout.CENTER, 5, 0));
		pnY.add(this.txtYScale);
		pnY.add(this.choiceUnitY);

		this.chkAspectRatio = new Checkbox("Constrain Aspect Ratio", true);

		this.choiceInterpolation.add("Linear");
		this.choiceInterpolation.add("Cubic");
		this.choiceInterpolation.select("Cubic");

		this.choiceMethod.add("Interpolation");
		this.choiceMethod.add("Least-Squares");
		this.choiceMethod.select("Least-Squares");

		buildCell(grid, c, 0, 0, 1, 1, new Label("Width"));
		buildCell(grid, c, 1, 0, 1, 1, pnX);

		buildCell(grid, c, 0, 1, 1, 1, new Label("Height"));
		buildCell(grid, c, 1, 1, 1, 1, pnY);

		buildCell(grid, c, 1, 3, 1, 1, this.chkAspectRatio);

		buildCell(grid, c, 0, 4, 1, 1, new Label("Method"));
		buildCell(grid, c, 1, 4, 1, 1, this.choiceMethod);
		buildCell(grid, c, 0, 5, 1, 1, new Label("Spline"));
		buildCell(grid, c, 1, 5, 1, 1, this.choiceInterpolation);

		// Panel buttons 
		Panel buttons = new Panel();
		buttons.setLayout(new FlowLayout(FlowLayout.CENTER, 5, 0));
		this.bnCancel = new Button("Cancel");
		this.bnCancel.addActionListener(this);
		buttons.add(this.bnCancel);
		this.bnOK = new Button("  OK  ");
		buttons.add(this.bnOK);
		this.bnOK.addActionListener(this);

		c.gridx = 1;
		c.gridy = 6;
		c.gridwidth = 1;
		c.gridheight = 1;
		c.anchor = GridBagConstraints.EAST;
		c.insets = new Insets(20, 5, 5, 5);
		grid.setConstraints(buttons, c);
		this.dlg.add(buttons);

		this.chkAspectRatio.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{
				if (Resize_.this.chkAspectRatio.getState() == true
						&& Resize_.this.constraintRatio == false)
				{
					Resize_.this.constraintRatio = true;
					constraintAspectRatioX();
				}
				if (Resize_.this.chkAspectRatio.getState() == false
						&& Resize_.this.constraintRatio == true)
				{
					Resize_.this.constraintRatio = false;
					constraintAspectRatioX();
				}
			}
		});

		this.choiceUnitX.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{
				String unit = Resize_.this.choiceUnitX.getSelectedItem();
				double x = getDoubleValue(Resize_.this.txtXScale, 0,
						Resize_.this.nx, Double.MAX_VALUE);
				if ((unit.equals("Pixels")) && (Resize_.this.UnitPixelsX == false))
				{
					Resize_.this.UnitPixelsX = true;
					long px = Math.round(x / 100.0 * Resize_.this.nx);
					Resize_.this.txtXScale.setText(new Long(px).toString());
				}
				if ((unit.equals("Percent")) && (Resize_.this.UnitPixelsX == true))
				{
					Resize_.this.UnitPixelsX = false;
					double px = x / Resize_.this.nx * 100.0;
					Resize_.this.txtXScale.setText(convertDoubleToString3(px));
				}
				//constraintAspectRatioY();
			}
		});

		this.choiceUnitY.addItemListener(new ItemListener()
		{
			public void itemStateChanged(ItemEvent e)
			{
				String unit = Resize_.this.choiceUnitY.getSelectedItem();
				double y = getDoubleValue(Resize_.this.txtYScale, 0,
						Resize_.this.ny, Double.MAX_VALUE);
				if ((unit.equals("Pixels")) && (Resize_.this.UnitPixelsY == false))
				{
					Resize_.this.UnitPixelsY = true;
					long py = Math.round(y / 100.0 * Resize_.this.ny);
					Resize_.this.txtYScale.setText(new Long(py).toString());
				}
				if ((unit.equals("Percent")) && (Resize_.this.UnitPixelsY == true))
				{
					Resize_.this.UnitPixelsY = false;
					double py = y / Resize_.this.ny * 100.0;
					Resize_.this.txtYScale.setText(convertDoubleToString3(py));
				}
				//constraintAspectRatioX();
			}
		});

		this.txtXScale.addKeyListener(new KeyListener()
		{
			public void keyReleased(KeyEvent e)
			{
				Resize_.this.bnOK.setEnabled(false);
				if (!Resize_.this.txtXScale.getText().equals(""))
				{
					double p = 0.0;
					try
					{
						if (Resize_.this.UnitPixelsX == true)
						{
							p = Integer.parseInt(Resize_.this.txtXScale.getText());
						}
						else
						{
							p = (Double.valueOf(Resize_.this.txtXScale.getText()))
									.doubleValue();
						}
					}
					catch (NumberFormatException n)
					{
						n.printStackTrace();
					}
					if (p > 0.0)
					{
						Resize_.this.bnOK.setEnabled(true);
						constraintAspectRatioY();
					}
				}
			}

			public void keyTyped(KeyEvent e)
			{
				// we don't care
			}

			public void keyPressed(KeyEvent e)
			{
				// we don't care
			}
		});

		this.txtYScale.addKeyListener(new KeyListener()
		{
			public void keyReleased(KeyEvent e)
			{
				Resize_.this.bnOK.setEnabled(false);
				if (!Resize_.this.txtYScale.getText().equals(""))
				{
					double p = 0.0;
					try
					{
						if (Resize_.this.UnitPixelsY == true)
						{
							p = Integer.parseInt(Resize_.this.txtYScale.getText());
						}
						else
						{
							p = (Double.valueOf(Resize_.this.txtYScale.getText()))
									.doubleValue();
						}
					}
					catch (NumberFormatException n)
					{
						n.printStackTrace();
					}
					if (p > 0.0)
					{
						Resize_.this.bnOK.setEnabled(true);
						constraintAspectRatioX();
					}
				}
			}

			public void keyTyped(KeyEvent e)
			{
				// we don't care
			}

			public void keyPressed(KeyEvent e)
			{
				// we don't care
			}
		});

		// Building the main panel
		this.dlg.setModal(true);
		this.dlg.pack();
		//dlg.setResizable(false);
		GUI.center(this.dlg);
		this.dlg.setVisible(true);
		IJ.wait(250); // work around for Sun/WinNT bug
	}

	/**
	* Build one cell of the dialog box.
	*/
	private void buildCell(GridBagLayout gbl, GridBagConstraints gbc, int x,
			int y, int w, int h, Component Comp)
	{
		gbc.gridx = x;
		gbc.gridy = y;
		gbc.gridwidth = w;
		gbc.gridheight = h;
		gbc.anchor = GridBagConstraints.NORTHWEST;
		gbc.insets = new Insets(5, 5, 5, 5);
		gbl.setConstraints(Comp, gbc);
		this.dlg.add(Comp);
	}

	/**
	 * Implements the actionPerformed for the ActionListener.
	 */
	public void actionPerformed(ActionEvent e)
	{
		if (e.getSource() == this.bnCancel)
		{
			this.dialogOK = false;
			this.dlg.dispose();
		}
		else if (e.getSource() == this.bnOK)
		{
			this.dialogOK = true;
			this.sizeX = getDoubleValue(this.txtXScale, 0, 10.0, Double.MAX_VALUE);
			this.sizeY = getDoubleValue(this.txtYScale, 0, 10.0, Double.MAX_VALUE);
			this.method = this.choiceMethod.getSelectedItem();
			this.interpolation = this.choiceInterpolation.getSelectedItem();
			this.UnitPixelsX = (this.choiceUnitX.getSelectedItem())
					.equals("Pixels");
			this.UnitPixelsY = (this.choiceUnitY.getSelectedItem())
					.equals("Pixels");
			this.dlg.dispose();
		}
	}

	/**
	* Get a double value from a TextField between minimal and maximal values.
	*/
	double getDoubleValue(TextField text, double mini, double defaut,
			double maxi)
	{
		double d;
		try
		{
			String s = text.getText();
			if (s.charAt(0) == '-')
			{
				s = s.substring(1);
			}
			d = (new Double(s)).doubleValue();
			if (d < mini) text.setText("" + mini);
			if (d > maxi) text.setText("" + maxi);
		}

		catch (Exception e)
		{
			if (e instanceof NumberFormatException) text.setText("" + defaut);
		}
		d = (new Double(text.getText())).doubleValue();
		return d;
	}

	/**
	* Constraint the size Y. 
	*/
	void constraintAspectRatioY()
	{
		if (this.constraintRatio == true)
		{
			boolean unitX = (this.choiceUnitX.getSelectedItem()).equals("Pixels");
			boolean unitY = (this.choiceUnitY.getSelectedItem()).equals("Pixels");
			double x = getDoubleValue(this.txtXScale, 0, this.nx, Double.MAX_VALUE);
			if (unitX == false) x = x / 100.0 * this.nx;
			double y = Math.round(x * this.ny / this.nx); // in pixels
			if (unitY == true) this.txtYScale.setText("" + Math.round(y));
			else this.txtYScale
					.setText(convertDoubleToString3(y * 100.0 / this.ny));
		}
	}

	/**
	* Constraint the size X. 
	*/
	void constraintAspectRatioX()
	{
		if (this.constraintRatio == true)
		{
			boolean unitX = (this.choiceUnitX.getSelectedItem()).equals("Pixels");
			boolean unitY = (this.choiceUnitY.getSelectedItem()).equals("Pixels");
			double y = getDoubleValue(this.txtYScale, 0, this.ny, Double.MAX_VALUE);
			if (unitY == false) y = y / 100.0 * this.ny;
			double x = Math.round(y * this.nx / this.ny); // in pixels
			if (unitX == true) this.txtXScale.setText("" + Math.round(x));
			else this.txtXScale.setText(convertDoubleToString3(x * 100.0 / this.nx));
		}

	}

	/**
	* Convert a double to a String with 3 digit in the decimal part.
	*/
	String convertDoubleToString3(double d)
	{
		if (d >= Double.MAX_VALUE) return "3.4e38";
		if (d < 0.0) d = -d;
		double dr = Math.round(d * 1000.0) / 1000.0;
		DecimalFormat df = new DecimalFormat("0.000");
		return df.format(dr);
	}

	/**
	*/
	private ShortProcessor createShortProcessor(ImageAccess image)
	{
		double pixels[] = image.getPixels();
		ShortProcessor sp = new ShortProcessor(image.getWidth(), image.getHeight());
		short[] ssrc = new short[pixels.length];
		double p;
		for (int k = 0; k < pixels.length; k++)
		{
			p = pixels[k];
			if (p < 0) p = 0.0;
			if (p > Short.MAX_VALUE) p = Short.MAX_VALUE;
			ssrc[k] = (short) p;
		}
		sp.setPixels(ssrc);
		return sp;
	}

}
