/*
 * ****************************************************************************
 *
 * (C) Copyright 2004 Science Applications International Cororation.
 * All Rights Reserved.
 *
 * Developed by Science Applications International Corporation under
 * contract to the Department of Defense. All government restrictions
 * on software distribution apply.
 *
 * Subject to the GNU LESSER GENERAL PUBLIC LICENSE,
 * Version 2.1, February 1999 http://www.gnu.org/copyleft/lesser.html
 * A copy of that license is included with the source distribution.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Peer Reviewed:
 *
 * Revision History:
 * $Log
 * ****************************************************************************
 */
package com.saic.isd.util.report;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.print.Printable;
import java.util.Date;

import javax.print.Doc;
import javax.print.DocFlavor;
import javax.print.DocPrintJob;
import javax.print.PrintException;
import javax.print.PrintService;
import javax.print.PrintServiceLookup;
import javax.print.ServiceUI;
import javax.print.SimpleDoc;
import javax.print.attribute.DocAttributeSet;
import javax.print.attribute.HashDocAttributeSet;
import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.swing.JEditorPane;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.ScrollPaneConstants;

import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.startup.AppStarter;

import org.gerhardb.lib.print.PrintableText;

import com.saic.isd.printscreen.Util;
import com.saic.isd.swing.filechooser.SimpleFileFilter;

/**
 * Displays a text or html report in a frame with the ability to print the
 * report.
 * <p>
 * @author   Gerhard Beck   (original)
 * @author   $Author: gerhardb $     (latest)
 * @version  $Revision: 1.8 $
 */
public class Report
   extends JFrame
{
   //static final Logger clsLogger =
   //    Logger.getLogger( "com.saic.isd.util.report.Report" );

   protected JEditorPane myEditorPane = new JEditorPane();
   protected String myHTML = null;

   /*
    * =========================================================================
    *                         Constructor
    * =========================================================================
    */

   /**
    * Shows a report in a frame with printing enabled.
    * @param title Shown on the frames title bar.
    * @param report The text/html to show.  If HTML, needs html tags.
    * @param html Use true for html, false for text.
    */
   public static void showReport(String title, String report, boolean html)
   {
      String type = "text/plain"; //$NON-NLS-1$
      if (html)
      {
         type = "text/html"; //$NON-NLS-1$
      }
      new Report(title, report, type);
   }

   /**
    * Shows a report putting a header on the report with
    * the application and report title.
    * @param appTitle
    * @param reportTitle "Report" will be appended to the title.
    * @param report report body without html or body tags.
    */
   public static void showReport(
      String appTitle, String reportTitle, String report)
   {
      String title = reportTitle + " " + AppStarter.getString("report");  //$NON-NLS-1$ //$NON-NLS-2$
      String reportHTML = "<html><body>" //$NON-NLS-1$
                          + "<center><strong><big>" //$NON-NLS-1$
                          + appTitle + "<br>" //$NON-NLS-1$
                          + reportTitle + " " + AppStarter.getString("report") //$NON-NLS-1$ //$NON-NLS-2$
                          + "</big></strong><br>" //$NON-NLS-1$
                          + "<strong>" + new Date() + "</strong></center>" //$NON-NLS-1$ //$NON-NLS-2$
                          + "<hr>" //$NON-NLS-1$
                          + report
                          + "</body></html>"; //$NON-NLS-1$
      new Report(title, reportHTML, "text/html"); //$NON-NLS-1$
   }

   // =========================================================================
   //                        Constructor
   // =========================================================================

   /**
    * Shows a report in a frame with printing enabled.
    * @param title Shown on the frames title bar.
    * @param report The text/html/rtf to show.  If HTML, needs html tags.
    * @param type Any valid type as described in
    * javax.swing.JEditorPane.  Currentl types are:
    * "text/plain", "text/html", "text/rtf"
    */
   public Report(String title, String report, String type)
   {
      this(title, type);
      this.myEditorPane.setText(report);
      setVisible(true);
   }

   /**
    * Shows a report in a frame with printing enabled.
    * @param title Shown on the frames title bar.
    * @param report The text/html/rtf to show.  If HTML, needs html tags.
    * @param type Any valid type as described in
    * javax.swing.JEditorPane.  Currentl types are:
    * "text/plain", "text/html", "text/rtf"
    */
   public Report(String title, String type)
   {
      super(title);
      Dimension size = new Dimension(700, 400);
      setSize(size);

      this.myEditorPane.setEditable(false);
      this.myEditorPane.setDisabledTextColor(Color.black);
      this.myEditorPane.setEditorKit(
         JEditorPane.createEditorKitForContentType(type));

      JScrollPane center = new JScrollPane(
                           this.myEditorPane,
                           ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED,
                           ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER
                           );

      getContentPane().setLayout(new BorderLayout());
      getContentPane().add(center, BorderLayout.CENTER);

      this.setJMenuBar(this.makeMenus());

      SwingUtils.centerOnScreen(this);
   }

   // =========================================================================
   //                        Private Methods
   // =========================================================================
   private JMenuBar makeMenus()
   {
      JMenu fileMenu = new JMenu(AppStarter.getString("SortScreen.menu.file.label")); //$NON-NLS-1$
      JMenu editMenu = new JMenu(AppStarter.getString("SortScreen.menu.edit.label")); //$NON-NLS-1$

      JMenuItem closeMenu = new JMenuItem(AppStarter.getString("close"));  //$NON-NLS-1$
      closeMenu.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            Report.this.dispose();
         }
      });

      JMenuItem saveMenu = new JMenuItem(
      		AppStarter.getString("SortScreen.menu.file.saveas.label")); //$NON-NLS-1$
      saveMenu.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            String contents = Report.this.myEditorPane.getText();
            Util.saveToFile(
               contents,
               new SimpleFileFilter("html", AppStarter.getString("report")), //$NON-NLS-1$ //$NON-NLS-2$
               Report.this.myEditorPane);
         }
      });

      JMenuItem printPreviewMenu = 
      	new JMenuItem(AppStarter.getString("preview") + "..."); //$NON-NLS-1$ //$NON-NLS-2$
      printPreviewMenu.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            Util.previewHTML(
            		Report.this, 
            		AppStarter.getString("preview"),  //$NON-NLS-1$
            		Report.this.myEditorPane.getText());
         }
      });

      JMenuItem printImageMenu = new JMenuItem(
      		AppStarter.getString("print") + "...", KeyEvent.VK_P); //$NON-NLS-1$ //$NON-NLS-2$
      printImageMenu.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            new PrintableText(Report.this.myEditorPane).printContent();
         }
      });

      JMenuItem printTextMenu = new JMenuItem(AppStarter.getString("Report.22"), KeyEvent.VK_T); //$NON-NLS-1$
      printTextMenu.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            //printHtml(myHTML);
            printWithService();
         }
      });

      JMenuItem copyMenu = new JMenuItem(AppStarter.getString("Report.23")); //$NON-NLS-1$
      copyMenu.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent ae)
         {
            String contents = Report.this.myEditorPane.getText();
            Util.copyToClipboard(contents);
         }
      });

      fileMenu.add(saveMenu);
      fileMenu.add(printPreviewMenu);
      fileMenu.add(printImageMenu);
      fileMenu.add(new JSeparator());
      fileMenu.add(closeMenu);

      editMenu.add(copyMenu);

      JMenuBar menuBar = new JMenuBar();
      menuBar.add(fileMenu);
      menuBar.add(editMenu);

      return menuBar;
   }

   /**
    * Print using graphic component.
    */
   void printWithService()
   {
      Printable printable = new PrintableText(this.myEditorPane);
      DocFlavor flavor = DocFlavor.SERVICE_FORMATTED.PRINTABLE;

      PrintRequestAttributeSet pras = new HashPrintRequestAttributeSet();
      //pras.add(new Copies(1));
      //pras.add(MediaSize.getMediaSizeForName(MediaSizeName.ISO_C6));
      //pras.add(Sides.DUPLEX);
      PrintService[] allServices =
         PrintServiceLookup.lookupPrintServices(flavor, pras);
      if (allServices.length == 0)
      {
         //System.out.println("No print services found");
         return;
      }
      PrintService defaultService =
         PrintServiceLookup.lookupDefaultPrintService();
      PrintService service =
         ServiceUI.printDialog(null, 200, 200,
         allServices, defaultService, flavor, pras);
      if (service == null)
      {
         //System.out.println("User cancelled print");
         return;
      }
      DocPrintJob job = service.createPrintJob();
      DocAttributeSet das = new HashDocAttributeSet();
      Doc doc = new SimpleDoc(printable, flavor, das);
      try
      {
         job.print(doc, pras);
      }
      catch (PrintException pe)
      {
         pe.printStackTrace();
      }
   }

   /**
    * Attempt to use the Java Print Service.
    * Unfortunately, HTML does not seem to be implemented in the
    * current version of java.
    * java.sun.com/j2se/1.4.1/docs/guide/jps/spec/JPSTOC.fm.html
    * access1.sun.com/tools/JPSDemo/jpsdemo.html
    * javaalmanac.com/egs/javax.print/pkg.html
    * forum.java.sun.com/thread.jsp?thread=146439&forum=31&message=1185367
    * java.sun.com/products/java-media/2D/forDevelopers/sdk12print.html
    *
    * RECOMMENDED SITE TO TRY FOR NEXT TIME:
    * http://java.sun.com/products/java-media/2D/forDevelopers/sdk12print.html
    *
    * Two other approaches:
    * a) convert html to postscript using open source widget
    *    and print postscript.
    * b) java help seems to be able to do this.  Bundle with distribution.
    *    License looks like it would work.
    */
   /*
   DocFlavor flavor = DocFlavor.STRING.TEXT_HTML;
   DocFlavor flavor = DocFlavor.STRING.TEXT_PLAIN;
   DocFlavor flavor = DocFlavor.INPUT_STREAM.TEXT_HTML_UTF_8;
*/
/*
   static void printHtml(String html)
   {
      DocFlavor flavor = DocFlavor.INPUT_STREAM.TEXT_HTML_US_ASCII;
      InputStream htmlIn = Util.stringToInputStream(html);

      PrintRequestAttributeSet pras = new HashPrintRequestAttributeSet();
      //pras.add(new Copies(1));
      //pras.add(MediaSize.getMediaSizeForName(MediaSizeName.ISO_C6));
      //pras.add(Sides.DUPLEX);
      PrintService[] allServices =
         PrintServiceLookup.lookupPrintServices(flavor, pras);
      if (allServices.length == 0)
      {
         //System.out.println("No print services found");
         return;
      }
      PrintService defaultService =
         PrintServiceLookup.lookupDefaultPrintService();
      PrintService service =
         ServiceUI.printDialog(null, 200, 200,
         allServices, defaultService, flavor, pras);
      if (service == null)
      {
         //System.out.println("User cancelled print");
         return;
      }
      DocPrintJob job = service.createPrintJob();
      DocAttributeSet das = new HashDocAttributeSet();
      Doc doc = new SimpleDoc(htmlIn, flavor, das);
      try
      {
         job.print(doc, pras);
      }
      catch (PrintException pe)
      {
         pe.printStackTrace();
      }
   }
*/
}