/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.jibs.concatenater;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.DecimalFormat;
import java.util.prefs.Preferences;

import javax.swing.*;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.swing.JFileChooserExtra;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.StopCheckButton;

/**
 */

public class Splitter extends JFrame
{
	public static String APP_NAME = "Splitter";
	private static final String LAST_FILE = "LastFile"; //$NON-NLS-1$
	private static final String LAST_DIR = "LastDir"; //$NON-NLS-1$
	private static final String LAST_MEGS = "LastMegs"; //$NON-NLS-1$
	private static final String READY = "Ready"; //$NON-NLS-1$

	private static final Preferences clsPrefs = Preferences.userRoot().node("/org/gerhardb/jibs/concatenater/splitter"); //$NON-NLS-1$

	JTextField myFileFld = new JTextField(60);
	JTextField myRootDir = new JTextField(60);
   private JSlider myMegsSlider;
	DefaultBoundedRangeModel myRange = new DefaultBoundedRangeModel();
	JProgressBar myBar = new JProgressBar(this.myRange);
	StopCheckButton myStopCheckBtn;

	public Splitter(boolean iExitOnClose)
	{
		super("Splitter"); //$NON-NLS-1$
		this.myStopCheckBtn = new StopCheckButton();
		this.myMegsSlider = new JSlider(SwingConstants.HORIZONTAL, 0, 100, clsPrefs.getInt(LAST_MEGS, 10));

		layoutComponents();
		this.myRootDir.setText(clsPrefs.get(LAST_DIR, null));
		
		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());

		if (iExitOnClose)
		{
			this.addWindowListener(new WindowAdapter()
			{
				@Override
				public void windowClosing(WindowEvent evt)
				{
					System.exit(0);
				}
			});
		}

		EventQueue.invokeLater(new Runnable()
		{
			public void run()
			{
				Splitter.this.pack();
				SwingUtils.centerOnScreen(Splitter.this);
				Splitter.this.setVisible(true);
			}
		});
	}

	private void layoutComponents()
	{
		this.myBar.setStringPainted(true);
		this.myBar.setString(READY); 
		
      this.myMegsSlider.setSnapToTicks(true);
		this.myMegsSlider.setPaintLabels(true);
		this.myMegsSlider.setPaintTicks(true);
		this.myMegsSlider.setMajorTickSpacing(10);
		this.myMegsSlider.setMajorTickSpacing(1);
		
		// Set up application
		this.setSize(new Dimension(600, 600));

		JButton goBtn = new JButton("Split"); //$NON-NLS-1$
		goBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				go();
			}
		});

		JButton fileBtn = new JButton("..."); //$NON-NLS-1$
		fileBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				selectFile();
			}
		});

		JButton rootBtn = new JButton("..."); //$NON-NLS-1$
		rootBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent evt)
			{
				selectRoot();
			}
		});

		JPanelRows topPanel = new JPanelRows(FlowLayout.CENTER);
		JPanel aRow = topPanel.topRow();
		aRow.add(new JLabel(
						"Automatically splits files in a sequence like frodo.mpg.001, frodo.mpg.002")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel(" ")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel(
				"Select the file to split.")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel("File" + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		aRow.add(this.myFileFld);
		aRow.add(fileBtn);
		
		aRow = topPanel.nextRow();
		aRow.add(new JLabel(" ")); //$NON-NLS-1$

		aRow = topPanel.nextRow(new BorderLayout());
		aRow.add(new JLabel("Megabytes" + Jibs.getString("colon") + " "), BorderLayout.WEST); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$		
		aRow.add(this.myMegsSlider, BorderLayout.CENTER);

		aRow = topPanel.nextRow();
		aRow.add(new JLabel(" ")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel(
				"Select the diretory where the split files go.")); //$NON-NLS-1$

		aRow = topPanel.nextRow();
		aRow.add(new JLabel("Directory" + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		aRow.add(this.myRootDir);
		aRow.add(rootBtn);

		aRow = topPanel.nextRow();
		aRow.add(goBtn);
		aRow.add(this.myStopCheckBtn.getStopButton());

		aRow = topPanel.nextRow(new BorderLayout());
		aRow.add(this.myBar, BorderLayout.CENTER);

		this.setContentPane(topPanel);
	}

	void selectFile()
	{
		this.myBar.setString(READY); 
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(LAST_FILE,
				null));
		chooser.setSaveName(APP_NAME, Jibs.getString("RecreateDirectories.21")); //$NON-NLS-1$
		chooser.setApproveButtonText(Jibs.getString("RecreateDirectories.22")); //$NON-NLS-1$
		chooser.setDialogTitle("Select file to split"); //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
		chooser.setMultiSelectionEnabled(false);

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked != null)
			{
				this.myFileFld.setText(picked.toString());
				try
				{
					clsPrefs.put(LAST_FILE, picked.toString());
					clsPrefs.flush();
					if (null == clsPrefs.get(LAST_DIR, null))
					{
						clsPrefs.put(LAST_DIR, picked.getParent().toString());
					}
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
	}

	void selectRoot()
	{
		this.myBar.setString(READY); 
		JFileChooserExtra chooser = new JFileChooserExtra(clsPrefs.get(LAST_DIR,
				null));
		chooser.setSaveName(APP_NAME, Jibs.getString("RecreateDirectories.21")); //$NON-NLS-1$
		chooser.setApproveButtonText(Jibs.getString("RecreateDirectories.22")); //$NON-NLS-1$
		chooser.setDialogTitle("Select directory where split files go"); //$NON-NLS-1$
		chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		chooser.setMultiSelectionEnabled(false);

		int returnVal = chooser.showOpenDialog(this);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			File picked = chooser.getSelectedFile();
			if (picked != null)
			{
				this.myRootDir.setText(picked.toString());
				try
				{
					clsPrefs.put(LAST_DIR, picked.toString());
					clsPrefs.flush();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}
	}
	
	void go()
	{
		try
		{
			clsPrefs.putInt(LAST_MEGS, this.myMegsSlider.getValue());
			clsPrefs.flush();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		
		String rootString = this.myRootDir.getText();
		File rootFile = new File(rootString);
		if (!rootFile.exists())
		{
			JOptionPane
					.showMessageDialog(this, "Directory does not exist: "
							+ rootString, "Problem with Diretory",
							JOptionPane.ERROR_MESSAGE);
			return;
		}
		
		String fileString = this.myFileFld.getText();
		File fileToSpit = new File(fileString);
		if (!fileToSpit.exists())
		{
			JOptionPane
					.showMessageDialog(this, "File does not exist: "
							+ fileToSpit, "Problem with File",
							JOptionPane.ERROR_MESSAGE);
			return;
		}

		if (this.myMegsSlider.getValue() < 1)
		{
			JOptionPane
			.showMessageDialog(this, "Select at least one megabyte", "Problem",
					JOptionPane.ERROR_MESSAGE);			
		}
		//System.out.println("RootString: " + rootString);

		if (this.myStopCheckBtn.isStopped()) { return; }
		setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try
		{
			SplitAll sa = new SplitAll(fileToSpit, rootFile, this.myMegsSlider.getValue());
			new Thread(sa).start(); 
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this, ex.getMessage(), "Splitter", //$NON-NLS-1$
					JOptionPane.ERROR_MESSAGE);
		}
		setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}
	
	private class SplitAll implements Runnable
	{
		File myFileToSplit;
		File myDirToSplitInto;
		int myMegs;
		int myCount;
		String myBaseFileName;
		
		SplitAll(File fileToSplit, File dirToSplitInto, int megs)
		{
			this.myFileToSplit = fileToSplit;
			this.myDirToSplitInto = dirToSplitInto;
			this.myMegs = megs * 1024 * 1024;
			this.myBaseFileName = this.myDirToSplitInto.getAbsolutePath() 
			+ System.getProperty("file.separator") + this.myFileToSplit.getName() 
			+ ".";
			System.out.println("myMegs: " + this.myMegs);
		}
		
		private String getNextFileName(DecimalFormat formatter)
		{
			this.myCount++;
			String newFileName = this.myBaseFileName + formatter.format(this.myCount);
			System.out.println("New File Name: " + newFileName);
			return newFileName;			
		}
		
		@SuppressWarnings("null")
		public void run()
		{
			System.out.println("File to split is: " + this.myFileToSplit);
			System.out.println("Directory to split into is: " + this.myDirToSplitInto);
			

			int estimatedCount = (int)(this.myFileToSplit.length() / this.myMegs) + 1;
			Splitter.this.myBar.setStringPainted(false);
			Splitter.this.myBar.getModel().setRangeProperties(0, 0, 0, estimatedCount, false);

			String format = "00";
			if (estimatedCount > 99)
			{
				format = "000";
			}
			else if (estimatedCount > 999)
			{
				format = "0000";
			}
			else if (estimatedCount > 9999)
			{
				format = "#0000";
			}				
		   DecimalFormat formatter = new DecimalFormat(format); //$NON-NLS-1$

			BufferedInputStream in = null;				
			BufferedOutputStream out = null;
			int written = 0;
			try
			{	
				in = new BufferedInputStream(
					new FileInputStream(this.myFileToSplit));			
			
				out = new BufferedOutputStream(
					new FileOutputStream(getNextFileName(formatter)));

				byte[] buf = new byte[1024];
				int len = 0;
				while ((len = in.read(buf)) > 0 && !Splitter.this.myStopCheckBtn.isStopped())
				{
					out.write(buf, 0, len);
					written = written + len;
					// Start a new file if we exceeded our limit.
					if (written >= this.myMegs)
					{
						Splitter.this.myBar.setValue(this.myCount);
						out.close();
						written = 0;
						out = new BufferedOutputStream(
							new FileOutputStream(getNextFileName(formatter)));						
					}
				}
				Splitter.this.myBar.setValue(this.myCount);
				in.close();
				out.close();
			}
			catch (IOException ex)
			{
				JOptionPane.showMessageDialog(Splitter.this, ex.getMessage(), "Splitter", //$NON-NLS-1$
						JOptionPane.ERROR_MESSAGE);
				try
				{
					in.close();
					out.close();
				}
				catch (Exception doubleEx)
				{
					doubleEx.printStackTrace();
				}
			}

			Splitter.this.myBar.setStringPainted(true);
			Splitter.this.myBar.setString("Done"); 	
			System.out.println("ALL DONE!!!");
		}	
		
	}

	// ===========================================================================
	// Main
	// ===========================================================================
	public static void main(String[] args)
	{
		new Splitter(true);
	}
}
