/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.dirtools;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Date;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.io.EzLogger;
import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.io.TargetFileExistsException;


/**
 * Does all the actual filtering work.
 * <p>
 * Orginally intended to be able to be driven from a command line, but that
 * go lost in the suffle.
 * <p>
 * Notes: All files without an image file name are automatically DELETED!!!.
 */
public class WorkerCopy
   implements Runnable
{
   static final SimpleDateFormat LONG_DATE =
      new SimpleDateFormat("yyyy.MMMMM.dd HH:mm:ss"); //$NON-NLS-1$
   static final String FAILED = "FAILED	";
   static final String SKIPPED = "SKIPPED	";
   static final String SUCCEEDED = "SUCCEEDED	";
   static final String ABORTED = "==> ABORTED <==";
   //static final String[] EXCLUDE = {"Recycled", "System Volume Information", "trash"};
   
   // OTHER VARIABLES
   boolean iStop;
   DirCopyPanel myPanel;
   File myToDir;
   EzLogger myLog;
   File[] myExcludeDirectories;
   File[] myIncludeDirectories;
   
   //===========================================================================
   //                              Constructor
   //===========================================================================
   public WorkerCopy(
   	DirCopyPanel panel,
      String to
      )
   throws Exception
   {
   	this.myPanel = panel;
   	this.myToDir = new File(to);
   	this.myExcludeDirectories = this.myPanel.myDirTools.myExcludeList.getDirArray();
   	this.myIncludeDirectories = this.myPanel.myDirTools.myIncludeList.getDirArray();
      this.myLog = new EzLogger(this.myPanel.myLogFile);
      this.myLog.logLine("Copy To: " + this.myToDir);      
   }

   void stop()
   {
   	this.iStop = true;
   }
   
   // ==========================================================================
   // Runnable Interface
   // ==========================================================================
   public void run()
   {
      this.iStop = false;
      
      Date startDate = new Date();
      this.myLog.logLine("Started" + Jibs.getString("colon") + " " 
      		+ LONG_DATE.format(startDate) + ""); 
      this.myLog.logLine(EzLogger.DIVIDER);
      this.myPanel.myCurrentAction.setText("Started"); 

     	for(int i=0; i<this.myIncludeDirectories.length; i++)
   	{
        	// First, recreate the from directory
         this.myLog.logLine(EzLogger.DIVIDER);
         this.myLog.logLine("Copy From: " + this.myIncludeDirectories[i]);
   		String newDirName = 
   			this.myToDir.getAbsolutePath() + "/" + getFinalDirFromPath(this.myIncludeDirectories[i]);
   		File newToDir = new File(newDirName);
   		if (makeSureDirectoryIsThere(newToDir))
   		{
   			try
   			{	  
   		      // Now recurse
   				processDirectory(this.myIncludeDirectories[i], newToDir);
   			}
   			catch(Exception ex)
   			{
   		      this.myLog.logLine(FAILED + "Could not PROCESS directory: " + this.myIncludeDirectories[i] + "    BECAUSE: " + ex.getMessage());   			   
   		      ex.printStackTrace();
   		   }   				
   		}  	
   	}
       
      this.myLog.logLine(EzLogger.DIVIDER);
      Date stopDate = new Date();
      this.myLog.logLine(
         "Completed" + Jibs.getString("colon") + " " 
         + LONG_DATE.format(stopDate) + ""); 
      this.myPanel.myCurrentAction.setText("Stopped"); 

      try
      {
      	this.myLog.close();
      }
      catch(Exception ex)
      {
         ex.printStackTrace();
      }
   }

   //===========================================================================
   //                              Private Functions
   //===========================================================================
   
   private void processDirectory(File fromDir, File toDir)
   {
   	if (this.iStop)
   	{
   		this.myLog.logLine(ABORTED);
   		return;
   	}
   	
		if (fromDir == null || !fromDir.isDirectory())
		{
		   this.myLog.logLine(FAILED + "Null or not a directory: " + fromDir);   		
		   return;
		}
		
      this.myPanel.myCurrentAction.setText(fromDir.getAbsolutePath()); 
      File[] fromFiles = null;
		try
		{	
			fromFiles = fromDir.listFiles();
		}
		catch(Exception ex)
		{
	      this.myLog.logLine(FAILED + "Could not LOOK UP files in: " + fromDir);   				
		}   	
		
		if (fromFiles == null)
		{
	      this.myLog.logLine(FAILED + "Could not LOOK UP files in: " + fromDir);   	
	      return;			
		}
		
   	for(int i=0; i<fromFiles.length; i++)
   	{
      	if (this.iStop)
      	{
		      this.myLog.logLine(ABORTED);  
      		return;
      	}
      	
   		if (fromFiles[i].isDirectory())
   		{
   			boolean processDirectory = true;
   		  	for(int j=0; j<this.myExcludeDirectories.length; j++)
   		  	{
   		  		if (fromFiles[i].equals(this.myExcludeDirectories[j]))
   		  		{
   				   this.myLog.logLine("EXCLUDED: " + fromFiles[i]);   		
   				   processDirectory = false;
   		  		}
   		  	}
   		  	
   		  	if (processDirectory)
   		  	{
	   			String newDirName = 
	   				toDir.getAbsolutePath() + "/" + getFinalDirFromPath(fromFiles[i]);
	   			File newToDir = new File(newDirName);
	   			if (makeSureDirectoryIsThere(newToDir))
	   			{
	      			try
	      			{	
	      				processDirectory(fromFiles[i], newToDir);
	      			}
	      			catch(Exception ex)
	      			{
	      		      this.myLog.logLine(FAILED + "Could not PROCESS directory: " + fromFiles[i] + "    BECAUSE: " + ex.getMessage());  
	      		      ex.printStackTrace();
	      			}   				
	   			}
   		  	}
   		}
   		else if (fromFiles[i].isFile())
   		{
   			try
   			{
   				if (fromFiles[i].length() > 0)
   				{
	   				FileUtil.copyFileToDir(toDir, fromFiles[i]);
	   		      this.myLog.logLine(SUCCEEDED + fromFiles[i]);    	
   				}
   				else
   				{
      		      this.myLog.logLine("ZERO LENGTH SKIPPED	" + fromFiles[i]);    					
   				}
   			}
   			catch(TargetFileExistsException ex)
   			{
   		      this.myLog.logLine(SKIPPED + fromFiles[i]);    				
   			}
   			catch(Exception ex)
   			{
   		      this.myLog.logLine(FAILED + fromFiles[i]);   				
   			}
   		}
   	}
   }

   public static String getFinalDirFromPath(File dir)
   {
		int length = 0;
		File parent = dir.getParentFile();
		if (parent != null)
		{
			String parentPath = parent.getAbsolutePath();
			//System.out.println(parentPath);
			//System.out.println(dir.getAbsolutePath());
			//System.out.println(length);
			// In case parent is disk drive.
			if (parentPath.endsWith("/") || parentPath.endsWith("\\"))
			{
				length = parentPath.length();
			}
			else
			{
				length = parentPath.length() + 1;
			}
			String rtnMe = dir.getAbsolutePath().substring(length);
	   	return rtnMe;
		}
		// Assume we are on a root drive.
		String rtnMe = dir.getAbsolutePath().substring(0, 1);
		if (rtnMe.endsWith("/"))
		{
			rtnMe = "root";
		}
		return "drive_" + rtnMe;
   }

	private boolean makeSureDirectoryIsThere(File newDir)
	{
		try
		{
			if (newDir.exists())
			{ 
		      this.myLog.logLine(SKIPPED + "Found directory: "  + newDir.getAbsolutePath());     					
			}
			else
			{
				newDir.mkdir();  
		      this.myLog.logLine(SUCCEEDED + "Created directory: "  + newDir.getAbsolutePath());     					
			}  	
			return true;
		}
		catch(Exception ex)
		{
	      this.myLog.logLine(FAILED + "Could not create directory: " + newDir);   	
	      return false;
		}
	}
   
   public static void main(String[] args)
   {
   	System.out.println(getFinalDirFromPath(new File("/")));
   	System.out.println(getFinalDirFromPath(new File("D:")));
   	System.out.println(getFinalDirFromPath(new File("D:\\")));
   	System.out.println(getFinalDirFromPath(new File("D:\\foo")));
   	System.out.println(getFinalDirFromPath(new File("D:\\foo\\bar")));
   }
   
}