/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.diskUsage;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.io.File;
import java.util.Date;
import java.util.prefs.Preferences;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.dirtree.DTNReader;
import org.gerhardb.lib.dirtree.DTNReaderWriter;
import org.gerhardb.lib.dirtree.DirectoryTreeNode;
import org.gerhardb.lib.dirtree.DirectoryTreeRDP;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.scroller.IScroll;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.util.Conversions;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.Loading;

public class ExplorerBox extends JFrame
{
	boolean iExitOnClose;
	boolean iShowLabel;
	BoxPanel myBoxPanel;
	long myTotalDiskSpace;
	
	private static final String PRIOR_DIR = "priorDir"; //$NON-NLS-1$
	private static final String PREF_PATH = 
		"/org/gerhardb/jibs/lib/filetree/ExplorerBox"; //$NON-NLS-1$
	Preferences myPrefs = Preferences.userRoot().node(PREF_PATH);
	
	boolean iCountImages;
	
	JLabel myDirName = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel myDiretoryCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel mySubDiretoryCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel myTotalDiretoryCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	
	JLabel myTotalAllFileCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel myTotalImageFileCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel myTotalAllFileSize = new JLabel("", SwingConstants.CENTER);	 //$NON-NLS-1$
	JLabel myTotalImageFileSize = new JLabel("", SwingConstants.CENTER);	 //$NON-NLS-1$
		
	JLabel myNodeFileCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel myNodeImageFileCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel myNodeFileSize = new JLabel("", SwingConstants.CENTER);	 //$NON-NLS-1$
	JLabel myNodeImageFileSize = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	
	JLabel mySubDirFileCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel mySubDirImageFileCount = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$
	JLabel mySubDirFileSize = new JLabel("", SwingConstants.CENTER);	 //$NON-NLS-1$
	JLabel mySubDirImageFileSize = new JLabel("", SwingConstants.CENTER); //$NON-NLS-1$

	JLabel myDateLabel = new JLabel(AppStarter.getString("date"), SwingConstants.LEFT); //$NON-NLS-1$
	
	DTNReaderWriter myReaderWriter;
	IScroll myScroller;
	boolean iCountHiddenDirectories;
	boolean iShowFreeSpace = false;
	File[] myDrives;
	boolean iSupressRootDirectories;

	public ExplorerBox(boolean standAlone, boolean label, IScroll scroller, DTNReaderWriter dtn, 
			boolean countHiddenDirectories, boolean showFreeSpace)
	{		
		this.iExitOnClose = standAlone;
		this.iShowLabel = label;
		this.myScroller = scroller;
		this.iCountHiddenDirectories = countHiddenDirectories;
		this.iShowFreeSpace = showFreeSpace;
		
		ViewerPreferences.init();		//	If loaded from SortScreen, this has been done already.			
		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
		
		this.myReaderWriter = dtn;
				
		JMenu driveMenu = new JMenu("Open Drive");
		this.myDrives = JibsDiskUsage.listDrives();
		for(int i=0;i<this.myDrives.length;i++)
		{
		    driveMenu.add(new JMenuItem(new OpenRoot(this.myDrives[i])));
		}
		
		JMenu fileMenu = new JMenu(AppStarter.getString("file")); //$NON-NLS-1$
		fileMenu.add(new JMenuItem(new OpenAction()));		
		fileMenu.add(driveMenu);						
		fileMenu.add(new JMenuItem(new ReloadTreeAction()));		
		fileMenu.add(new JMenuItem(new CloseAction()));		
		
		JMenuBar bar = new JMenuBar();
		bar.add(fileMenu);
		super.setJMenuBar(bar);
		
		if (this.iExitOnClose)
		{
			super.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);   
		}
		
		this.myBoxPanel = new BoxPanel(this);
		
		JPanel contentPanel = new JPanel(new BorderLayout());
		contentPanel.add(this.myBoxPanel, BorderLayout.CENTER);
		contentPanel.add(infoPanel(), BorderLayout.SOUTH);
		
		super.getContentPane().add(contentPanel);
		
		int size = 700;
		super.setSize(size, size);
      SwingUtils.centerOnScreen(this);
		super.setVisible(true);		
	}
	
	public void setRootNode(DirectoryTreeNode rootNode)
	{
		this.myBoxPanel.setRootNode(rootNode);
		this.setTitle(rootNode.getAbsolutePath());
	}
	
	public void setRootDirectory(final File rootDirectory, boolean supressRootDirectories)
	throws Exception
	{
		if (!rootDirectory.exists() || !rootDirectory.isDirectory())
		{
			return;
		}
		this.iSupressRootDirectories = supressRootDirectories;
		
		final Loading loader = new Loading(ExplorerBox.this, "Explorer");		
		Runnable runMe = new Runnable()
		{
			public void run()
			{
				DirectoryTreeNode rootNode = null;
				Date lastRead = null;
				try
				{
					if (ExplorerBox.this.iShowFreeSpace)
					{
						ExplorerBox.this.myTotalDiskSpace = rootDirectory.getTotalSpace();
					}		
					
					File jibsTreeFile = new File(rootDirectory + 
							System.getProperty("file.separator")
							+ ExplorerBox.this.myReaderWriter.getFileName());		
					
					DirectoryTreeRDP tree = new DirectoryTreeRDP(ExplorerBox.this, ImageFactory.getImageFactory().getFilter(), null, ExplorerBox.this.myScroller, ExplorerBox.this.iCountHiddenDirectories);
					tree.setCountTargetFiles(ViewerPreferences.countImageFilesInNodes());
					if (ExplorerBox.this.iSupressRootDirectories)
					{
						tree.setExcludedDirectories(ExplorerBox.this.myDrives);
					}
					
					DTNReader reader = ExplorerBox.this.myReaderWriter.getReader(jibsTreeFile, tree, loader);
					ExplorerBox.this.iCountImages = reader.getImagesCounted();
					lastRead = reader.getLastReadDate();
					rootNode = reader.getRootNode();
					if (rootNode == null)
					{
						System.out.println("DISK USAGE STARTUP: MANUAL POPULATION"); //$NON-NLS-1$
						rootNode = new DirectoryTreeNode(tree, rootDirectory, loader);
			
						// The NodeAdded() below will update the loading screen.
						// On startup this is already running in a non-swing thread.
						rootNode.populate(loader);
						lastRead = new Date();						
						try
						{
							ExplorerBox.this.myReaderWriter.getWriter().store(rootNode);
						}
						catch(Exception ex)
						{
							ex.printStackTrace();
						}
					} 
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
				finally
				{
					loader.dispose();
				}
				
				if (rootNode == null)
				{
	            JOptionPane.showMessageDialog(
	            	 ExplorerBox.this,
	                "Unexpected Problem",
	                AppStarter.getString("problem"), //$NON-NLS-1$
	                JOptionPane.ERROR_MESSAGE );
	            return;
				}
				
				if (ExplorerBox.this.iShowFreeSpace)
				{
					long actualUsedDiskSpace = ExplorerBox.this.myTotalDiskSpace - rootDirectory.getFreeSpace();
					long recordedUsedDiskSpace = rootNode.getTreeTotals().aggregateTotalFileSize;
					float percentage = 0.05f;
					long overLimit = (long)(actualUsedDiskSpace * (1 + percentage));
					long underLimit = (long)(actualUsedDiskSpace * (1 - percentage));
					//System.out.println("overLimit: " + overLimit);
					//System.out.println("underLimit: " + underLimit);
					//System.out.println("recordedUsedDiskSpace: " + recordedUsedDiskSpace);
					if (recordedUsedDiskSpace > overLimit || recordedUsedDiskSpace < underLimit)
					{
						int answer = JOptionPane.showConfirmDialog(
								ExplorerBox.this,
								"The recorded size differs by more than 5% from the current size.\n"
								+ "Do you want to rescan the drive?\n",
								"Rescan Drive", JOptionPane.YES_NO_OPTION);
						if (answer == JOptionPane.YES_OPTION) 
						{ 
							rootNode.reloadNodes(ExplorerBox.this, ExplorerBox.this.iSupressRootDirectories, ExplorerBox.this.myReaderWriter.getWriter());
						}						
					}
				}	
				
				setRootNode(rootNode);
				
				if (lastRead == null)
				{
					ExplorerBox.this.myDateLabel.setText(AppStarter.getString("ExplorerBox.24")); //$NON-NLS-1$
					ExplorerBox.this.myDateLabel.setBackground(Color.RED);
				}
				else
				{
					ExplorerBox.this.myDateLabel.setText(AppStarter.getString("ExplorerBox.25") + DTNReaderWriter.DATE_FORMAT.format(lastRead)); //$NON-NLS-1$
					long diff = new Date().getTime() - lastRead.getTime();
					if (diff > Conversions.WEEK)
					{
						ExplorerBox.this.myDateLabel.setBackground(Color.RED);
					}
					else if (diff > Conversions.DAY * 3)
					{
						ExplorerBox.this.myDateLabel.setBackground(Color.YELLOW);
					}
					else
					{
						ExplorerBox.this.myDateLabel.setBackground(Color.GREEN);
					}
					ExplorerBox.this.myDateLabel.invalidate();
				}
			}
		};
		// This starts the real loading process from a new thread inside
		// Loading. That way, messages can show in the AWT thread in Loading.
		loader.start(runMe);
	}
	
	JPanel infoPanel()
	{
		JPanel topPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
		topPanel.add(this.myDirName);		
		
		JPanel gridPanel = new JPanel(new GridLayout(4, 3));
		gridPanel.add(new JLabel(" ")); //$NON-NLS-1$
		gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.27"), SwingConstants.CENTER)); //$NON-NLS-1$
		gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.28"), SwingConstants.CENTER)); //$NON-NLS-1$
		gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.29"), SwingConstants.CENTER)); //$NON-NLS-1$
		
		gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.30"))); //$NON-NLS-1$
		gridPanel.add(this.myTotalDiretoryCount);
		gridPanel.add(this.myDiretoryCount);
		gridPanel.add(this.mySubDiretoryCount);
		
		gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.31"))); //$NON-NLS-1$
		gridPanel.add(this.myTotalAllFileCount);
		gridPanel.add(this.myNodeFileCount);
		gridPanel.add(this.mySubDirFileCount);
		
		gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.32"))); //$NON-NLS-1$
		gridPanel.add(this.myTotalAllFileSize);
		gridPanel.add(this.myNodeFileSize);
		gridPanel.add(this.mySubDirFileSize);
		
		if (this.iCountImages)
		{
			gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.33"))); //$NON-NLS-1$
			gridPanel.add(this.myTotalImageFileCount);
			gridPanel.add(this.myNodeImageFileCount);
			gridPanel.add(this.mySubDirImageFileCount);
			gridPanel.add(new JLabel(AppStarter.getString("ExplorerBox.34"))); //$NON-NLS-1$
			gridPanel.add(this.myTotalImageFileSize);
			gridPanel.add(this.myNodeImageFileSize);
			gridPanel.add(this.mySubDirImageFileSize);
		}
		
		this.myDateLabel.setOpaque(true);
			
		JPanel rtnMe = new JPanel(new BorderLayout());
		rtnMe.add(topPanel, BorderLayout.NORTH);
		rtnMe.add(gridPanel, BorderLayout.CENTER);
		if (this.iShowLabel)
		{
			rtnMe.add(this.myDateLabel, BorderLayout.SOUTH);
		}
		return rtnMe;		
	}
	
	// ===========================================================================
	// Inner Classes
	// ===========================================================================
	class OpenAction extends AbstractAction
	{
		OpenAction()
		{
			super.putValue(Action.NAME, AppStarter.getString("ExplorerBox.23")); //$NON-NLS-1$
		}
		
		public void actionPerformed(ActionEvent e)
		{
			ExplorerBox.this.iShowFreeSpace = false;
			try
			{
				String priorDir = ExplorerBox.this.myPrefs.get(PRIOR_DIR, null);
				String newDirName = FileUtil.lookupDir(AppStarter.getString("ExplorerBox.36"), priorDir, ExplorerBox.this); //$NON-NLS-1$
				if (newDirName == null){return;}
				File newDir = new File(newDirName);
				ExplorerBox.this.setRootDirectory(newDir, false);
				ExplorerBox.this.myPrefs.put(PRIOR_DIR, newDirName);
				ExplorerBox.this.myPrefs.flush();
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	class OpenRoot extends AbstractAction
	{
		File myDrive;
		
		OpenRoot(File drive)
		{
			super.putValue(Action.NAME, drive.toString()); //$NON-NLS-1$
			this.myDrive = drive;
		}
		
		public void actionPerformed(ActionEvent e)
		{
			ExplorerBox.this.iShowFreeSpace = true;
			try
			{
				ExplorerBox.this.setRootDirectory(this.myDrive, true);
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	class CloseAction extends AbstractAction
	{
		CloseAction()
		{
			if (ExplorerBox.this.iExitOnClose)
			{
				super.putValue(Action.NAME, AppStarter.getString("SortScreen.menu.file.exit.label")); //$NON-NLS-1$
			}
			else
			{
				super.putValue(Action.NAME, AppStarter.getString("close"));					 //$NON-NLS-1$
			}
		}
		
		public void actionPerformed(ActionEvent e)
		{
	      if (ExplorerBox.this.iExitOnClose)
	      {
	      	System.exit(0);
	      }
			else
			{
				ExplorerBox.this.dispose();					
			}
		}
	}
	
	class ReloadTreeAction extends AbstractAction
	{
		ReloadTreeAction()
		{
			super.putValue(Action.NAME, AppStarter.getString("FileTreePopUp.4")); //$NON-NLS-1$
		}
		
		public void actionPerformed(ActionEvent e)
		{
			try
			{
				ExplorerBox.this.myBoxPanel.myRootNode.reloadNodes(
						ExplorerBox.this, 
						ExplorerBox.this.iSupressRootDirectories, 
						ExplorerBox.this.myReaderWriter.getWriter());
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	// ===========================================================================
	// Main
	// ===========================================================================
	public static void main(String[] args)
	{
		AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		
		File rootDirectory = new File("d:/testpics"); //$NON-NLS-1$
		//targetType, countTargetFiles, StartingJibsFile
		DTNReaderWriter dtn = new DTNReaderWriter("All", true, PathManager.getStartingJibsFile());
		// standAlone, label, IScroll, dtn
		ExplorerBox box = new ExplorerBox(true, false, new Scroller(), dtn, true, false);
		try
		{
			box.setRootDirectory(rootDirectory, false);
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
	}
}
