/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.editor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.font.FontRenderContext;
import java.awt.font.TextLayout;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.awt.image.SampleModel;
import java.awt.image.WritableRaster;
import java.io.File;

import javax.imageio.ImageIO;
import javax.swing.JPanel;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JViewport;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingUtilities;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.image.ImageFactory;

/**
 * JPanel implementation of showing a single image.
 */
public class PicturePanel extends JPanel implements MouseListener
{
	static protected final Color OK_COLOR = Color.black;
	static protected final Color EMPTY_COLOR = Color.cyan;
	static protected final Color NFG_COLOR = Color.green;

	BufferedImage myImage;
	int myOffsetWidth;
	int myOffsetHeight;
	float myOffsetWidthPercentage;
	float myOffsetHeightPercentage;
	boolean iNeedToRepositionScrollBars = false;
	protected int myShownWidth;
	protected int myShownHeight;

	private JScrollPane myScrollPane = new JScrollPane(this,
			ScrollPaneConstants.VERTICAL_SCROLLBAR_AS_NEEDED, //.VERTICAL_SCROLLBAR_ALWAYS, 
			ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);

	// ==========================================================================
	// Constructor
	// ==========================================================================
	public PicturePanel()
	{
		super.addMouseListener(this);
	}

	public void setPictureFile(File picFile)
	{
		if (picFile != null)
		{
			this.myImage = ImageFactory.getImageFactory().getImageEZ(picFile);
		}
	}

	// IS THIS NEEDED???
	public Component getImageDisplay()
	{
		return this.myScrollPane;
	}

	@Override
	public void paint(Graphics g)
	{
		// Apparently, background color can only be set once per draw without
		// confusing SWING.
		Color backgroundColor = OK_COLOR; // DEFAULT COLOR
		super.setBackground(backgroundColor);
		super.paintComponent(g); // First, fill with background color.

		BufferedImage showMe = this.myImage;
		if (showMe == null)
		{
			showNoPicture(g);
			return;
		}
		backgroundColor = EMPTY_COLOR;

		//System.out.println("======================== SingleNoResizeScrollControl.paint() ============================");
		//System.out.println("Panel w h: " + super.getWidth() + " " + super.getHeight());
		//System.out.println("Raw Picture w h: " + showMe.getWidth() + " " + showMe.getHeight());

		// Remember for pic info dialog
		this.myShownWidth = showMe.getWidth();
		this.myShownHeight = showMe.getHeight();

		Graphics2D g2 = (Graphics2D) g;

		// Initial position for when there are no scroll bars.
		this.myOffsetWidth = (int) ((PicturePanel.this.getWidth() - this.myShownWidth) * this.myOffsetWidthPercentage);
		this.myOffsetHeight = (int) ((PicturePanel.this.getHeight() - this.myShownHeight) * this.myOffsetHeightPercentage);

		Dimension size = new Dimension(this.myShownWidth, this.myShownHeight);
		// Update client's preferred size because the area taken up by the graphics has
		// gotten larger or smaller (if cleared).
		this.setPreferredSize(size);
		// Let the scroll pane know to update itself and its scrollbars.
		this.revalidate();

		// Now the the correct background color has been established,
		// use it.
		super.setBackground(backgroundColor);
		super.paintComponent(g); // First, fill with background color.

		// Draw the new image
		g2.drawImage(showMe, null, this.myOffsetWidth, this.myOffsetHeight);

		if (this.iNeedToRepositionScrollBars)
		{
			SwingUtilities.invokeLater(new Runnable()
			{
				public void run()
				{
					positionPicture();
				}
			});
			this.iNeedToRepositionScrollBars = false;
		}
	}

	protected void showNoPicture(Graphics g)
	{
		super.setBackground(NFG_COLOR);
		super.paintComponent(g); // Fill with background color.

		Graphics2D g2 = (Graphics2D) g;
		Font font = new Font("Default", Font.PLAIN, 24); //$NON-NLS-1$
		FontRenderContext fontRenderContext = g2.getFontRenderContext();

		TextLayout text = new TextLayout(Jibs.getString("PicturePanel.0"), font,  //$NON-NLS-1$
				fontRenderContext);

		// Text is draw up from the Ascent and down from the Descent.
		int xCenter = getWidth() / 2;
		int yCenter = getHeight() / 2;

		int textX = xCenter - (int) text.getBounds().getWidth() / 2;
		// This puts the text at the bottom of the box.
		int textY = (int) (yCenter - text.getDescent());
		//textY = (int) (textY + tlName.getAscent());
		text.draw(g2, textX, textY);
		/*
			if (tlSize != null)
			{
				// Text is draw up from the Ascent and down from the Descent.
				int xCenterPic = (int) ((myPicBoxWidth - tlSize.getBounds()
						.getWidth()) / 2);
				int textX = Xoffset + xCenterPic;
				// This puts the text at the bottom of the box.
				//int textY = (int) (Yoffset + myPicBoxHeight - tLay.getDescent());
				textY = (int) (textY + tlSize.getAscent());
				tlSize.draw(myG2, textX, textY);
			}
			*/

	}

	// ==========================================================================
	// MouseListener
	// ==========================================================================
	public void mouseClicked(MouseEvent event)
	{
		this.getToolkit().beep();

		if (event.getX() < this.myImage.getWidth()
				&& event.getY() < this.myImage.getHeight())
		{
			rasterSwap(event.getX(), event.getY());
			//rasterSwapTest(event.getX(), event.getY());		
		}
	}

	public void mouseEntered(MouseEvent e)
	{
		// We don't care
	}

	public void mouseExited(MouseEvent e)
	{
		// We don't care
	}

	public void mousePressed(MouseEvent e)
	{
		// We don't care
	}

	public void mouseReleased(MouseEvent e)
	{
		// We don't care
	}

	void positionPicture()
	{
		JViewport vp = this.myScrollPane.getViewport();
		Rectangle vpRect = vp.getViewRect();
		Dimension vpExtent = vp.getExtentSize();
		Dimension vpViewSize = vp.getViewSize();

		JScrollBar hBar = this.myScrollPane.getHorizontalScrollBar();
		JScrollBar vBar = this.myScrollPane.getVerticalScrollBar();

		//System.out.println("======================== SingleNoResizeScrollControl.positionPicture() ============================");

		if (vpExtent.width == vpViewSize.width)
		{
			// If there are NO bars
			//System.out.println("NO width bars " + myShownWidth);
			//offsetWidth = (int) ((SingleNoResizeScrollControl.this.getWidth() - myShownWidth) * myOffsetWidthPercentage);
			//myOffsetWidth = (int)((vpRect.getWidth() - myShownWidth) * myOffsetWidthPercentage);
		}
		else
		{
			// If there ARE bars
			//System.out.println("WIDTH bars");
			this.myOffsetWidth = (int) ((hBar.getMaximum() - (int) vpRect.getWidth()) * this.myOffsetWidthPercentage);
		}

		if (vpExtent.height == vpViewSize.height)
		{
			// If there are NO bars
			//System.out.println("NO height bars");
			//myOffsetHeight = (int) ((vpRect.getHeight() - myShownHeight) * myOffsetHeightPercentage);
		}
		else
		{
			// If there ARE bars
			//System.out.println("HEIGHT bars");
			this.myOffsetHeight = (int) ((vBar.getMaximum() - (int) vpRect.getHeight()) * this.myOffsetHeightPercentage);
		}

		Point newPosition = new Point(this.myOffsetWidth, this.myOffsetHeight);
		vp.setViewPosition(newPosition);

		/*
		System.out.println("vpRect: " + vpRect);
		System.out.println("Rect x/w: " + vpRect.x + "/" + vpRect.width + " hBar: " +  hBar.getMaximum() + " offset: " + myOffsetWidth);
		System.out.println("Rect y/h: " + vpRect.y + "/" + vpRect.height + " vBar: " +  vBar.getMaximum() + " offset: " + myOffsetHeight);
		
		System.out.println("Extent: " + vp.getExtentSize());
		System.out.println("View Size: " + vp.getViewSize());
		*/
	}

	void rasterSwapTest(int x, int y)
	{
		System.out.println("---------------------------------"); //$NON-NLS-1$
		BufferedImage pic = this.myImage;		
		WritableRaster alphaRaster = pic.getAlphaRaster();
		if (alphaRaster == null)
		{
			System.out.println("Alpha is null"); //$NON-NLS-1$
		}
		else
		{
			System.out.println("Alpha");			 //$NON-NLS-1$
		}
		
		System.out.println("---------------------------------"); //$NON-NLS-1$
		String[] props = pic.getPropertyNames();
		if (props != null)
		{
			for(int i=0; i<props.length; i++)
			{
				System.out.println("Property: " + props[i]); //$NON-NLS-1$
			}
		}
		
		/*
		// Not available in Java 1.4 
		switch(pic.getTransparency())
		{
		case Transparency.OPAQUE:
			System.out.println("OPAQUE");
			break;
		case Transparency.BITMASK:
			System.out.println("BITMASK");
			break;
		case Transparency.TRANSLUCENT:
			System.out.println("TRANSLUCENT");
			break;
		}
		 */
		
		ColorModel colorModel = pic.getColorModel();
		System.out.println("Color Model: " + colorModel); //$NON-NLS-1$
		
		
		WritableRaster raster = pic.getRaster();
		System.out.println("Number of Bands: " + raster.getNumBands()); //$NON-NLS-1$
		System.out.println("Number of Data Elements: " + raster.getNumDataElements()); //$NON-NLS-1$
		
		int[] pixel =	raster.getPixel(x, y, (int[])null);
		System.out.println("Pixel Length: " + pixel.length); //$NON-NLS-1$
		for(int i=0; i<pixel.length; i++)
		{
			System.out.println(pixel[i]);
		}
		System.out.println("Sample: " + raster.getSample(x, y, 0)); //$NON-NLS-1$
		
		SampleModel model = raster.getSampleModel();
		System.out.println("Sample Model: " + model); //$NON-NLS-1$
		System.out.println("Trasfer Type: " + raster.getTransferType()); //$NON-NLS-1$
		int[] sampleBits = model.getSampleSize();
		for(int i=0; i<sampleBits.length; i++)
		{
			System.out.println(sampleBits[i]);
		}

		int rgb = pic.getRGB(x, y);
		System.out.println("rgb: " + rgb); //$NON-NLS-1$
		System.out.println("alpha: " + colorModel.getAlpha(pixel[0])); //$NON-NLS-1$
		System.out.println("red: " + colorModel.getRed(pixel[0])); //$NON-NLS-1$
		System.out.println("green: " + colorModel.getGreen(pixel[0])); //$NON-NLS-1$
		System.out.println("blue: " + colorModel.getBlue(pixel[0])); //$NON-NLS-1$
		System.out.println("Transparent Pixel: " + ((IndexColorModel)colorModel).getTransparentPixel()); //$NON-NLS-1$
		
		System.out.println("Number of Color Components: " + colorModel.getNumColorComponents()); //$NON-NLS-1$
		System.out.println("Number of Components: " + colorModel.getNumComponents()); //$NON-NLS-1$
		
		int[] sizes = colorModel.getComponentSize();
		System.out.println("Red Component Size: " + sizes[0]); //$NON-NLS-1$
		System.out.println("Green Component Size: " + sizes[1]); //$NON-NLS-1$
		System.out.println("Blue Component Size: " + sizes[2]); //$NON-NLS-1$
		//System.out.println("Alpha Component Size: " + sizes[3]);
		
		IndexColorModel oldICM = (IndexColorModel)colorModel;
		int mapSize = oldICM.getMapSize();
		System.out.println("Map Size: " + mapSize); //$NON-NLS-1$
		
		byte[] r = new byte[mapSize];
		byte[] g = new byte[mapSize];
		byte[] b = new byte[mapSize];
		byte[] a = new byte[mapSize];
		oldICM.getReds(r);
		oldICM.getGreens(g);
		oldICM.getBlues(b);
		oldICM.getAlphas(a);
		
		a[0] = 0;
		
		//for(int i=0; i<mapSize; i++)
		for(int i=0; i<16; i++)
		{
			System.out.println(i + ": " + " r " + r[i] + " b " + b[i] + " g " + g[i] + " a " + a[i]); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}

		IndexColorModel icm = oldICM;
		icm = new IndexColorModel(
					colorModel.getPixelSize(), 
					16, //colorModel.getNumColorComponents(), 
					r, 
					g, 
					b,
					a);
		
		System.out.println("icm Transparent Pixel: " + icm.getTransparentPixel()); //$NON-NLS-1$
		
		// OK SO FAR -- TRANSPARENT PIXEL EITHER WAY
		
		WritableRaster newRaster = raster.createCompatibleWritableRaster();
		//newRaster = raster;
		BufferedImage newPic = new BufferedImage(icm, newRaster, icm.isAlphaPremultiplied(), null);
		//newPic = pic;
		
		//int[] aPixel = new int[1];
		int[] transparent = new int[1];
		transparent[0] = 0;
		for (int xi = 0; xi < pic.getWidth(); xi++)
		{
			for (int yi = 0; yi < pic.getHeight(); yi++)
			{
				/*
				aPixel = raster.getPixel(x, y, aPixel);
				if (aPixelRgb[0] == aPixel[0])
				{
					aPixel[0] = 0;
					newRaster.setPixel(xi, yi, aPixel);					
				}
				else
				{
					newRaster.setPixel(xi, yi, aPixel);					
				}
				*/
				int currentRGB = pic.getRGB(xi, yi);				
				//newPic.setRGB(xi, yi, currentRGB);
				if (currentRGB == rgb)
				{
					newPic.setRGB(xi, yi, 0);
					//newRaster.setPixel(xi, yi, transparent);
				}
				else
				{
					newPic.setRGB(xi, yi, currentRGB);
					//aPixel = raster.getPixel(xi, yi, aPixel);
					//newRaster.setPixel(xi, yi, aPixel);					
				}
			}
		}
		
		try
		{
			ImageIO.write(newPic, "gif", //$NON-NLS-1$
					new File("d:/testpics/grindstoneTEST.gif")); //$NON-NLS-1$
			System.out.println("Wrote GIF"); //$NON-NLS-1$
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		
		try
		{
			ImageIO.write(newPic, "png", new File("d:/testpics/grindstoneTEST.png")); //$NON-NLS-1$ //$NON-NLS-2$
			System.out.println("Wrote PNG"); //$NON-NLS-1$
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}
	
	
	void rasterSwap(int x, int y)
	{
		System.out.println("RasterSwap---------------------------------"); //$NON-NLS-1$
		int rgb = this.myImage.getRGB(x, y);
		System.out.println("rgb: " + rgb); //$NON-NLS-1$

		WritableRaster raster = this.myImage.getRaster();
		IndexColorModel oldICM = (IndexColorModel)this.myImage.getColorModel();
		int mapSize = oldICM.getMapSize();
		System.out.println("Map Size: " + mapSize); //$NON-NLS-1$
		
		byte[] r = new byte[mapSize];
		byte[] g = new byte[mapSize];
		byte[] b = new byte[mapSize];
		byte[] a = new byte[mapSize];
		oldICM.getReds(r);
		oldICM.getGreens(g);
		oldICM.getBlues(b);
		oldICM.getAlphas(a);
		
		// Set the transparent Bit for this index
		a[0] = 0;
		
		IndexColorModel icm = new IndexColorModel(
					oldICM.getPixelSize(), 
					256, 
					r, 
					g, 
					b,
					a);
		
		WritableRaster newRaster = raster.createCompatibleWritableRaster();
		BufferedImage newPic = new BufferedImage(icm, newRaster, icm.isAlphaPremultiplied(), null);
		for (int xi = 0; xi < this.myImage.getWidth(); xi++)
		{
			for (int yi = 0; yi < this.myImage.getHeight(); yi++)
			{
				int currentRGB = this.myImage.getRGB(xi, yi);		
				if (currentRGB == rgb)
				{
					newPic.setRGB(xi, yi, 0); // apparently index works here...
				}
				else
				{
					newPic.setRGB(xi, yi, currentRGB);			
				}
			}
		}
		
		this.myImage = newPic;
		repaint();
		
		/*
		try
		{
			ImageIO.write(newPic, "gif",
					new File("/testpics/grindstoneTEST.gif"));
			System.out.println("Wrote GIF");
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		
		try
		{
			ImageIO.write(newPic, "png", new File("/testpics/grindstoneTEST.png"));
			System.out.println("Wrote PNG");
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		*/
	}
}
