/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.jibs.games.missingTile;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.DisplayMode;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.LayoutManager;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

import javax.swing.JPanel;

import org.gerhardb.lib.image.IOImage;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.image.ImageUtil;

class TilePanel extends JPanel implements DropTargetListener
{
	MissingTile myFrame;
	int myCount = 4;
	Tile[] myTiles;
	BufferedImage[] myTileImages;
	boolean isGameInProgress = false;
	int myOriginalEmptyPlayIndex;
	int myEmptyTile;
	int myMovesMade;
	int myMaxImageSize;
	
	/**
	 * 
	 * @param frame
	 * @param coutn
	 */
	public TilePanel(MissingTile frame, int count)
	{
		this.myFrame = frame;
		// Figure out and set the layout.
		if (count > 2)
		{
			this.myCount = count;
		}
		super.setBackground(Color.green);
		super.setLayout(new TileLayout());
		//super.setLayout(new GridLayout(count, count, hgap, vgap));
		int tileCount = this.myCount * this.myCount;
		this.myTileImages = new BufferedImage[tileCount];
		this.myTiles = new Tile[tileCount];
		for (int i = 0; i < this.myTiles.length; i++)
		{
			this.myTiles[i] = new Tile(i + 1, this);
			super.add(this.myTiles[i]);
		}
		
		// Figure out adjacent
		int tileIndex = 0;
		int lastRow = this.myCount - 1;
		for(int row=0; row<this.myCount; row++)
		{
			for(int col=0; col<this.myCount; col++)
			{
				if ( row == 0 )
				{
					this.myTiles[tileIndex].tileNorth = 0;
				}
				else
				{
					this.myTiles[tileIndex].tileNorth = 
						this.myTiles[tileIndex - this.myCount].myIndex;
				}
				
				if ( row == lastRow )
				{
					this.myTiles[tileIndex].tileSouth = 0;
				}
				else
				{
					this.myTiles[tileIndex].tileSouth = 
						this.myTiles[tileIndex + this.myCount].myIndex;
				}
				
				if ( col == 0 )
				{
					this.myTiles[tileIndex].tileWest = 0;
				}
				else
				{
					this.myTiles[tileIndex].tileWest = 
						this.myTiles[tileIndex - 1].myIndex;
				}
				
				if ( col == lastRow )
				{
					this.myTiles[tileIndex].tileEast = 0;
				}
				else
				{
					this.myTiles[tileIndex].tileEast = 
						this.myTiles[tileIndex + 1].myIndex;
				}
				
				tileIndex++;
			}				
		}	
		
		GraphicsEnvironment ge = 
			GraphicsEnvironment.getLocalGraphicsEnvironment();
		DisplayMode ddm = ge.getDefaultScreenDevice().getDisplayMode();
		int width = ddm.getWidth();
		int height = ddm.getHeight();
		if (width < height)
		{
			this.myMaxImageSize = width;
		}
		else
		{
			this.myMaxImageSize = height;			
		}
		new DropTarget(this, this);				
	}
	
	//===========================================================================
	// DropTargetListener Implementation
	//===========================================================================
	
	public void dragEnter(DropTargetDragEvent event)
	{
		if (event.isDataFlavorSupported(DataFlavor.javaFileListFlavor))
		{
			System.out.println("TilePanel.dragEnter ACCEPTING");
			event.acceptDrag(DnDConstants.ACTION_COPY);
		}
		else if (event.isDataFlavorSupported(DataFlavor.imageFlavor))
		{
			System.out.println("TilePanel.dragEnter ACCEPTING");
			event.acceptDrag(DnDConstants.ACTION_COPY);
		}
		else
		{
			System.out.println("TilePanel.dragEnter REJECTING");
			event.rejectDrag();			
		}
		/*
		switch (event.getDropAction())
		{
		case DnDConstants.ACTION_COPY_OR_MOVE:
		case DnDConstants.ACTION_COPY:
			System.out.println("TilePanel.dragEnter ACCEPTING");
			event.acceptDrag(DnDConstants.ACTION_COPY);
			break;
		default:
			System.out.println("TilePanel.dragEnter REJECTING");
			event.rejectDrag();
		}
		*/
	}

	
	public void dragExit(DropTargetEvent event)
	{
		// We don't care
	}

	
	public void dragOver(DropTargetDragEvent event)
	{
		// We don't care
	}

	
	public void drop(DropTargetDropEvent event)
	{
		// Determine the type of drop.
		/*
		if (event.getDropAction() != DnDConstants.ACTION_COPY)
		{			
			System.out.println("TilePanel REJECTING Drop - not a copy!");
			event.rejectDrop();
			return;
		}
		*/
		if (event.isDataFlavorSupported(DataFlavor.javaFileListFlavor))
		{
			System.out.println("TilePanel Accepting Drop: javaFileListFlavor");
			event.acceptDrop(DnDConstants.ACTION_COPY);
			try
			{
				// For drops from windows to work, you must accept the drop first.
				Transferable transfer = event.getTransferable();
				Object moveObj = transfer.getTransferData(DataFlavor.javaFileListFlavor);
				if (moveObj == null || !(moveObj instanceof List<?>))
				{
					event.dropComplete(false);
					return;
				}
				List<?> fileList = (List<?>)moveObj;
				if (fileList.size() < 1)
				{
					event.dropComplete(false);	
					return;
				}
				File imgFile = (File)fileList.get(0);
				IOImage ioImage = ImageFactory.getImageFactory().makeImage(imgFile);
				setImage(ioImage.getImage());
				event.dropComplete(true);
				MissingTile.clsPrefs.put(MissingTile.LAST_IMAGE, imgFile.getAbsolutePath());
				MissingTile.clsPrefs.put(MissingTile.LAST_DIRECTORY, imgFile.getAbsolutePath());
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
				event.dropComplete(false);
				return;
			}
		}
		else if (event.isDataFlavorSupported(DataFlavor.imageFlavor))
		{
			System.out.println("TilePanel Accepting Drop: imageFlavor");
			event.acceptDrop(DnDConstants.ACTION_COPY);
			try
			{
				// For drops from windows to work, you must accept the drop first.
				Transferable transfer = event.getTransferable();
				Object moveObj = transfer.getTransferData(DataFlavor.imageFlavor);
				if (moveObj == null || !(moveObj instanceof Image))
				{
					event.dropComplete(false);
					return;
				}
				setImage((Image)moveObj);
				event.dropComplete(true);
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
				event.dropComplete(false);
				return;
			}
		}
		else
		{
			System.out.println("TilePanel Rejecting Drop, Bad Flavor.");
			event.rejectDrop();
		}		
	}

	
	public void dropActionChanged(DropTargetDragEvent event)
	{
		// We don't care
	}

	//===========================================================================
	// Package Classes
	//===========================================================================
	void setImage(Image image)
	{
		BufferedImage buffImage = ImageUtil.toBufferedImage(image);
		setImage(buffImage);
	}
	
	void setImage(BufferedImage wholeImage)
	{
		// Make sure we start a new game with a new image.
		this.myFrame.quit();
		
		if (wholeImage == null)
		{
			for (int i = 0; i < this.myTileImages.length; i++)
			{
				this.myTileImages[i] = null;
			}		
			repaint();
			return;
		}
		
		float horizontalXScale = 1.0f;
		float vertcialYScale = 1.0f;
		
		horizontalXScale = this.myMaxImageSize / (float)wholeImage.getWidth();
		vertcialYScale = this.myMaxImageSize / (float)wholeImage.getHeight();
		
		// Default if scaling does not work.
		BufferedImage tileImage = wholeImage;
      try
      {
         AffineTransform at = AffineTransform.getScaleInstance(horizontalXScale, vertcialYScale);
         AffineTransformOp op = new AffineTransformOp(at, null);
         tileImage = op.filter(wholeImage, null);
      }
      catch (Exception ex)
      {
         System.err.println(ex.getMessage());
      }		
		
      /*
      System.out.println("myMaxImageSize: " + myMaxImageSize);
      System.out.println("Start Height: " + wholeImage.getHeight());
      System.out.println("Start Width: " + wholeImage.getWidth());
      System.out.println("Height Scale: " + horizontalXScale);
      System.out.println("Width Scale: " + vertcialYScale);
      System.out.println("Scaled Height: " + tileImage.getHeight());
      System.out.println("Scaled Width: " + tileImage.getWidth());
		*/
      
      int index = 0;
      int x = 0;
      int y = 0;
      int tileSize = this.myMaxImageSize / this.myCount;
		for (int row = 0; row < this.myCount; row++)
		{
			y = tileSize * row;
			for (int col = 0; col < this.myCount; col++)
			{
				x = tileSize * col;
				BufferedImage tile = tileImage.getSubimage(x, y, tileSize, tileSize);
				//System.out.println("index " + index + " row " + row + " col " + col);
				this.myTileImages[index] = tile;
				index++;
			}
		}
		repaint();
	}	
	
	BufferedImage getImage(int index)
	{
		return this.myTileImages[index];
	}
	
	void emptySeed(int emptyTile)
	{
		//System.out.println("emptyTile: " + emptyTile);
		this.myOriginalEmptyPlayIndex = this.myTiles[emptyTile-1].myPlayIndex;
		//System.out.println("emptyTile: " + emptyTile + " index: " + myOriginalEmptyPlayIndex);
   	setEmpty(emptyTile);
   	this.isGameInProgress = true;
	}

	void setEmpty(int emptyTile)
	{
		this.myEmptyTile = emptyTile;
   	for(int i=0; i<this.myTiles.length; i++)
   	{
   		this.myTiles[i].updateTileDisplay(this.myEmptyTile);
   	}
   }
	
	void moveMade(Tile fromTile)
	{
		Tile toTile = this.myTiles[this.myEmptyTile-1];
		toTile.myPlayIndex = fromTile.myPlayIndex;
		fromTile.myPlayIndex = 0;
		setEmpty(fromTile.myIndex);
   	this.myFrame.updateMovesMade(++this.myMovesMade);
   	checkVictory();
	}
	
	void checkVictory()
	{
		//System.out.println("Victory Check myEmptyTile: " + myEmptyTile +  " myOriginalEmptyPlayIndex " + myOriginalEmptyPlayIndex);
		if ( !(this.myOriginalEmptyPlayIndex == this.myEmptyTile))
		{
			return;
		}
		//System.out.println("Victory checking individual tiles");
   	for(int i=0; i<this.myTiles.length; i++)
   	{
   		// skip empty tile
   		if ( this.myOriginalEmptyPlayIndex != i+1 )
   		{
   			// if all non-empty tiles are in place, victory!
   			if (this.myTiles[i].myPlayIndex != this.myTiles[i].myIndex)
   			{
   				return;
   			}
   		}
   	}
   	this.myFrame.victory();
   }
	
	void resetGame()
	{
	  	this.isGameInProgress = false;
   	this.myOriginalEmptyPlayIndex = 0;  	
   	this.myMovesMade = 0;
   	this.myFrame.updateMovesMade(this.myMovesMade);
   	for(int i=0; i<this.myTiles.length; i++)
   	{
   		this.myTiles[i].resetTile();
   	}
   }
	
	  void scramble()
	  {
   	// Play indexes are offset from array by 1.
   	// So play indexes go from 1 to myTiles.length.
   	Random random = new Random();
		for (int i = 0; i < this.myTiles.length; i++)
		{
			// There will likely be duplicates after this...
			int candidate = random.nextInt(this.myTiles.length + 1);
			if ( candidate == 0 )
			{
				candidate = 1;
			}
			this.myTiles[i].myPlayIndex = candidate;
		}
		
		dedupScramble();

		if (this.myEmptyTile > 0)
		{
			this.myTiles[this.myEmptyTile-1].myPlayIndex = 1;
		}
		
		dedupScramble();
		
		repaint();
	}
   
   void dedupScramble()
   {
		boolean[] used = new boolean[this.myTiles.length];
		ArrayList<Tile> needsNewNumbers = new ArrayList<Tile>(this.myTiles.length);
		for (int i = 0; i < this.myTiles.length; i++)
		{
			int usedIndex = this.myTiles[i].myPlayIndex-1;
			if ( used[usedIndex])
			{
				needsNewNumbers.add(this.myTiles[i]);
			}
			else
			{
				used[usedIndex] = true;				
			}
		}
		Iterator<Tile> it = needsNewNumbers.iterator();
		while( it.hasNext())
		{
			Tile theTile = it.next();
			//System.out.println("Tile " + theTile.myIndex + " has duplicate: " + theTile.myPlayIndex );
			for (int i = 0; i < used.length; i++)
			{
				if ( !used[i])
				{
					used[i] = true;
					int newIndex = i+1;
					theTile.myPlayIndex = newIndex;
					//System.out.println("Tile " + theTile.myIndex + " resetting to:  " + newIndex );
					break;
				}
			}
		}
		//System.out.println("-------------------------------------------------" );   	
   }
   
	//	==========================================================================
	// Inner Classes // LayoutManager
	// ==========================================================================
	class TileLayout implements LayoutManager
	{
		public void addLayoutComponent(String name, Component comp)
		{
			// We don't care
		}
	
		public void layoutContainer(Container parent)
		{
			// Where to start drawing from.  
			// Larger deminsion will be adjusted to center tiles.
			int baseX = 0;
			int baseY = 0;

			int parentX = parent.getWidth();
			int parentY = parent.getHeight();
			
			// Pick the smaller
			int size = parentX;
			if ( parentY < parentX)
			{
				size = parentY;
				baseX = ((parentX - size)/2); 
			}
			else
			{
				baseY = ((parentY - size)/2); 
			}
			
			int xOffset = 0;
			int yOffset = 0;
			int tileIndex = 0;
			int tileSize = (size/TilePanel.this.myCount); 
			int lastRow = TilePanel.this.myCount - 1;
			for(int row=0; row<TilePanel.this.myCount; row++)
			{
				yOffset = baseY + (row * tileSize);
				for(int col=0; col<TilePanel.this.myCount; col++)
				{
					xOffset = baseX + (col * tileSize);
					//System.out.println("# " + tileIndex + "  X: " + xOffset + "  Y: " + yOffset );
					if ( row == 0 )
					{
						TilePanel.this.myTiles[tileIndex].outsideNorth = true;
					}
					else if ( row == lastRow )
					{
						TilePanel.this.myTiles[tileIndex].outsideSouth = true;
					}
					if ( col == 0 )
					{
						TilePanel.this.myTiles[tileIndex].outsideWest = true;
					}
					else if ( col == lastRow )
					{
						TilePanel.this.myTiles[tileIndex].outsideEast = true;
					}
					TilePanel.this.myTiles[tileIndex].setBounds(
							xOffset, yOffset, tileSize, tileSize);
					tileIndex++;
				}				
			}
		}
	
		public Dimension minimumLayoutSize(Container parent)
		{
			//System.out.println("layout minimumLayoutSize"); //$NON-NLS-1$
			return new Dimension(100, 100);
		}
	
		public Dimension preferredLayoutSize(Container parent)
		{
			//System.out.println("layout preferredLayoutSize"); //$NON-NLS-1$
			return new Dimension(500, 500);
		}
	
		public void removeLayoutComponent(Component comp)
		{
			// We don't care
		}
	}
}
