/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.textPad.rankPad;

import java.awt.Cursor;
import java.awt.Dimension;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.ArrayList;
import java.util.Enumeration;

import javax.swing.*;
import javax.swing.event.*;

import org.gerhardb.lib.io.FileUtil;

/**
 * The Ranked List Of Files
 */
public class RankList extends JList implements ChangeListener, ListSelectionListener
{
	private static final String VIEW_HISTORY_FILE_NAME = "-ViewHistory.jibs";
	
	RankPad rankPad;
	DefaultListModel listModel = new DefaultListModel();
	boolean iAmAdjustingIndex = false;	
	File rankFile; 
	String header;
	File viewHistoryFile;

	// ==========================================================================
	// Constructor
	// ==========================================================================
	public RankList(RankPad rp)
	{
		super();
		this.rankPad = rp;		
		super.setMinimumSize(new Dimension(100, 100));
		super.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		super.setModel(this.listModel);
		this.rankPad.rankAndFile.addChangeListener(this);
		super.addListSelectionListener(this);
	}
	
	public void stateChanged(ChangeEvent e)
	{
		if (this.rankPad.rankAndFile.getSelectedIndex() == 0)
		{
			RankFile file = (RankFile)this.rankPad.myDirectoryList.getSelectedValue();
			if (file == null)
			{ 
				this.setSelectedValue(null, true);
			}
			//System.out.println("Rank List Selection " + file);
			int index = this.listModel.indexOf(file);
			//System.out.println("Rank List Selection " + index);
			if (index > -1)
			{
				this.iAmAdjustingIndex = true;
				setSelectedIndex(index);
				this.iAmAdjustingIndex = false;
				ensureIndexIsVisible(index);
				this.rankPad.myStatusCount.setText((index + 1) + " of " + this.listModel.size() + " "); //$NON-NLS-1$
			}
		}
	}
	
 	/**
	 * Override so as to not make too wide.
	 */
	@Override
	public Dimension getPreferredSize()
	{
		Dimension oldSize = super.getPreferredSize();
		return new Dimension(100, (int) oldSize.getHeight());
	}

	public File getSelectedFile()
	{
		if (getModel().getSize() < 1)
		{
			return null;
		}
		Object obj = getSelectedValue();
		if (obj == null)
		{
			return null;
		}
		return ((RankFile)obj).getFile();
	}
	
	public void showRecnetFile(RecentFile rfToShow)
	{
		if (rfToShow == null)
		{
			//System.out.println("RankPad scrolling to: First File");
			// Show the first file now that we are visible.
			this.valueChanged(null);
		}
		else
		{
			RankFile ez = new RankFile(new File(rfToShow.filePath));
			int index = this.listModel.indexOf(ez);
			if (index > -1)
			{
				this.iAmAdjustingIndex = true;
				setSelectedIndex(index);
				this.iAmAdjustingIndex = false;
				ensureIndexIsVisible(index);
			}
			this.rankPad.showFile(new File(rfToShow.filePath), rfToShow.y);
			this.rankPad.myStatusCount.setText((index + 1) + " of " + this.listModel.size() + " "); //$NON-NLS-1$

		}	
	}
	
	public void valueChanged(ListSelectionEvent e) 
	{
		if (this.iAmAdjustingIndex)
		{
			return;
		}
		if (e!= null && e.getValueIsAdjusting())
		{
			return;
		}
		if (!this.rankPad.isVisible())
		{
			// Setting text in JTextPane before showing causes random locks on startup.
			return;
		}
		if (getModel().getSize() == 0)
		{
			return;
		}
		int index = getSelectedIndex();
		if (index < 0)
		{
			return;
		}
		//System.out.println("List changed: " + index);
		File ez = ((RankFile)getSelectedValue()).getFile();
		this.rankPad.showFile(ez, 0);
		this.rankPad.myStatusCount.setText((index + 1) + " of " + this.listModel.size() + " "); //$NON-NLS-1$

	}
	
	void populate(String rankFileName)
	{
		this.rankPad.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		if (rankFileName.length() > 0)
		{
			System.out.println("populate RankList: " + rankFileName);
			this.listModel.removeAllElements();
			File candidateFile = new File(rankFileName);
			if (candidateFile.exists())
			{
				this.rankFile = candidateFile;
				this.viewHistoryFile = new File(FileUtil.fileNameNoExtension(this.rankFile.getAbsolutePath()) + VIEW_HISTORY_FILE_NAME);
				System.out.println("viewHistoryFile: " + this.viewHistoryFile);
				try
				{
					BufferedReader buffRead = null;
					buffRead = new BufferedReader(new FileReader(this.rankFile));
					String aLine = buffRead.readLine();
					// Throw away the first line which is just the directory to the tree.
					aLine = buffRead.readLine();
					this.header = aLine;
					// Now that we got header, get the first line of data to process.
					aLine = buffRead.readLine();
					int index = 1;
					while (aLine != null)
					{
						String[] tokens = aLine.split(",");
						RankFile ez = new RankFile(tokens, index++);
						//System.out.println("RankFile: " + ez);
						this.listModel.addElement(ez);
						aLine = buffRead.readLine();
					}
					buffRead.close();
				}
				catch(Exception ex)
				{
					ex.printStackTrace();
				}
			}
		}	
		this.rankPad.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}

	void saveFile()
	{
		System.out.println("SAVING RankList: " + this.rankFile.getAbsolutePath());
		RankFile ez;
		Enumeration<?> ezFiles = this.listModel.elements();
		try
		{
			BufferedWriter buffWrite = null;
			buffWrite = new BufferedWriter(new FileWriter(this.rankFile));
			buffWrite.write(this.header);
			buffWrite.newLine();
			while(ezFiles.hasMoreElements())
			{
				ez = (RankFile)ezFiles.nextElement();
				buffWrite.write(ez.toLine());
				buffWrite.newLine();
			}
			buffWrite.close();
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
	}
	
	void delete(int rank)
	{
		System.out.println("Delete below " + rank);
		RankFile ez;
		ArrayList<RankFile> deleteThese = new ArrayList<RankFile>(100);
		Enumeration<?> ezFiles = this.listModel.elements();
		while(ezFiles.hasMoreElements())
		{
			ez = (RankFile)ezFiles.nextElement();
			if (ez.myRank < rank)
			{
				deleteThese.add(ez);
			}
		}

		if (deleteThese.size() == 0)
		{
			return;
		}
		
		if (this.rankPad.myDeleteDirectory != null && this.rankPad.myDeleteDirectory.exists())
		{
			System.out.println(this.rankPad.myDeleteDirectory);
			int answer = JOptionPane.showConfirmDialog(
					this.rankPad,
					"Move to Trash Directory " + deleteThese.size() + " files below rank of " + rank + "?",
					"Confirm Move To Trash Directory", JOptionPane.YES_NO_OPTION);
			if (answer == JOptionPane.YES_OPTION) 
			{ 
				this.iAmAdjustingIndex = true;
				this.rankPad.iAmAdjustingComboBoxSelection = true;
				this.rankPad.myDirectoryList.iAmAdjustingIndex = true;
				this.rankPad.myEditorPane.setText("");
				for(RankFile deleteThisFile: deleteThese)
				{
					try
					{
						System.out.println("RankList TRASHING: " + deleteThisFile);
						FileUtil.moveFile(this.rankPad.myDeleteDirectory, deleteThisFile.getFile());			
						this.rankPad.myTree.fileDeleted(deleteThisFile.getFile());
						this.listModel.removeElement(deleteThisFile);
						this.rankPad.myRecentFiles.removeElement(deleteThisFile);		
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
					}
				}
				saveFile();
				this.rankPad.myDirectoryList.listModel.removeAllElements();
				populate(RankPad.clsPrefs.get(RankPad.RANK_FILE, ""));	
				this.iAmAdjustingIndex = false;
				this.rankPad.iAmAdjustingComboBoxSelection = false;
				this.rankPad.myDirectoryList.iAmAdjustingIndex = false;
				return;
			}	
		}
		
		int answer = JOptionPane.showConfirmDialog(
				this.rankPad,
				"Permanently delete " + deleteThese.size() + " files below rank of " + rank + "?",
				"Confirm Delete", JOptionPane.YES_NO_OPTION);
		if (answer == JOptionPane.YES_OPTION) 
		{ 
			this.iAmAdjustingIndex = true;
			this.rankPad.iAmAdjustingComboBoxSelection = true;
			this.rankPad.myDirectoryList.iAmAdjustingIndex = true;
			this.rankPad.myEditorPane.setText("");
			for(RankFile deleteThisFile: deleteThese)
			{
				try
				{
					System.out.println("RankList PERMENENTLY deleting: " + deleteThisFile);
					deleteThisFile.getFile().delete();
					this.rankPad.myTree.fileDeleted(deleteThisFile.getFile());
					this.listModel.removeElement(deleteThisFile);
					this.rankPad.myRecentFiles.removeElement(deleteThisFile);		
				}
				catch(Exception ex)
				{
					ex.printStackTrace();
				}
			}
			saveFile();
			this.rankPad.myDirectoryList.listModel.removeAllElements();
			populate(RankPad.clsPrefs.get(RankPad.RANK_FILE, ""));	
			this.iAmAdjustingIndex = false;
			this.rankPad.iAmAdjustingComboBoxSelection = false;
			this.rankPad.myDirectoryList.iAmAdjustingIndex = false;

		}	
	}
}
