/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.textPad.rankPad;

import java.awt.event.*;
import java.io.File;
import java.util.ArrayList;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.LicenseBox;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.print.PrintableText;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.app.AboutBox;
import org.gerhardb.lib.util.app.AppUtils;
import org.gerhardb.lib.util.app.Info;
import org.gerhardb.lib.util.startup.*;

/**
 * Trying to remove clutter from main SortScreen and support
 * internationalized accelerated iconized menus.
 */
public class RankPadActions extends ActiveActions
{
   static String RECENT_TREE = "RECENT_TREE";
	static String RECENT_VIEW = "RECENT_VIEW";
	static int DEFAULT_MAX_COUNT = 9;
   static String RECENT_TREE_COUNT = "RECENT_TREE_COUNT";
	static String RECENT_VIEW_COUNT = "RECENT_VIEW_COUNT";
	
	RankPad myRankPad;  
	JMenu myRecentViewMenu;
	JMenu myRecentTreeMenu;
	
	public RankPadActions(RankPad ss)
	{
		this.myRankPad = ss;
		super.initialize();
	}
	

	@Override
	public Action getAction(String menu, String item)
	{
		// Menus orginally set up this way...
		menu = "SortScreen.menu" + "." + menu; //$NON-NLS-1$
		return super.getAction(menu, item);
	}
	
	/**
	 * Points to class in here with factory default options.
	 * Viewer can change these, then revert to factory defaults.
	 * @see org.gerhardb.lib.util.startup.ActiveActions#getStartupActions()
	 */
	@Override
	protected FactoryDefaultActions getFactoryDefaultActions()
	{
		return new DefaultStartupActions();
	}
	
	// ==========================================================================
	// Starter Actions and Defaults
	// ==========================================================================
	
	/**
	 * Factory Default options.
	 */
	private class DefaultStartupActions extends FactoryDefaultActions
	{
		public DefaultStartupActions()
		{
			addStartupAction("SortScreen.menu","file", "trash", new DeleteAction(), Icons.getIcon(Icons.DELETE)); //$NON-NLS-1$ //$NON-NLS-2$			
			addStartupAction("SortScreen.menu","file", "empty", new EmptyTrashAction(), null); //$NON-NLS-1$ //$NON-NLS-2$					
			addStartupAction("SortScreen.menu","file", "exit", new ExitAction(), null); //$NON-NLS-1$ //$NON-NLS-2$			
			addStartupAction("SortScreen.menu","help", "help", new HelpAction(), Icons.getIcon(Icons.HELP)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","help", "about", new AboutAction(), Icons.getIcon(Icons.ABOUT)); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","help", "license", new LicenseAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
			addStartupAction("SortScreen.menu","help", "info", new InfoAction(), null); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}
	
	class DeleteAction extends AbstractAction
	{
		DeleteAction()
		{
			super(null, Icons.getIcon(Icons.DELETE));
		}

		public void actionPerformed(ActionEvent ae)
		{
			RankPadActions.this.myRankPad.deletePressed();
		}
	}

	// ==========================================================================
	// Helper Methods
	// ==========================================================================
		
	@Override
	public JButton getToolBarButton(String menu, String item)
	{
		JButton rtnMe = getAccelleratedToolBarButton(menu, item);
		return rtnMe;
	}
	
	
	// ==========================================================================
	// FILE Menu
	// ==========================================================================
	JMenu getFileMenu()
	{
		this.myRecentViewMenu = ActionHelpers.makeMenu("recentviews");
		this.myRecentViewMenu.addMenuListener(new ViewMenuListener());

		this.myRecentTreeMenu = ActionHelpers.makeMenu("recenttrees");
		this.myRecentTreeMenu.addMenuListener(new TreeMenuListener());
		
		JMenuItem rankingFileMenu = new JMenuItem("Open Rankings File"); 
		rankingFileMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				RankPadActions.this.myRankPad.openRankings();
			}
		});

		JMenuItem copyMenu = new JMenuItem("Copy All"); //$NON-NLS-1$
		copyMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				new PrintableText(RankPadActions.this.myRankPad.myEditorPane).copy();
			}
		});
		
		JMenuItem rankDelete = new JMenuItem("Delete..."); //$NON-NLS-1$
		rankDelete.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				RankPadActions.this.rankDelete();
			}
		});
				
		JMenuItem openTreeFileMenu = new JMenuItem("Select Tree Directory");
		openTreeFileMenu.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				RankPadActions.this.myRankPad.openTree();
			}
		});
		
		JMenu fileMenu = ActionHelpers.makeMenu("file"); //$NON-NLS-1$
		fileMenu.add(rankingFileMenu);
		fileMenu.add(this.myRecentViewMenu);
		fileMenu.add(openTreeFileMenu); //$NON-NLS-1$ //$NON-NLS-2$
		fileMenu.add(this.myRecentTreeMenu);
		
		// Special because it may not be here for Java 4
		Action action = getAction("file", "defaultFileView");
		if (action != null)
		{
			fileMenu.add(new JMenuItem(action));
		}
		
		fileMenu.addSeparator();
		fileMenu.add(makeMenuItem("file", "trash")); //$NON-NLS-1$ //$NON-NLS-2$);
		fileMenu.add(makeMenuItem("file", "empty")); //$NON-NLS-1$ //$NON-NLS-2$
		fileMenu.add(rankDelete);
		fileMenu.addSeparator();
		
		JMenuItem exit = makeMenuItem("file", "exit"); //$NON-NLS-1$ //$NON-NLS-2$
		String closeOrExit = "Close";
		if (this.myRankPad.iExitOnClose)
		{
			closeOrExit = "Exit";
		}
		exit.setText(closeOrExit);
		fileMenu.add(exit);
		return fileMenu;
	}

	class EmptyTrashAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			File trashDir = RankPadActions.this.myRankPad.getStoredDeleteDirectory();
			if (trashDir != null && trashDir.exists() && trashDir.isDirectory())
			{
				File[] files = trashDir.listFiles();
				for(File deleteMe: files)
				{
					try
					{
						deleteMe.delete();
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
					}
				}
			}
		}
	}

	class ExitAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			RankPadActions.this.myRankPad.exit();
		}
	}	
	
	void rankDelete()
	{
		Object obj = JOptionPane.showInputDialog(
				this.myRankPad, 
				"Delete if rank is below : ", 
				"Delete by Rank",
				JOptionPane.QUESTION_MESSAGE,
				null,
				null,				
				"0");
		if(obj != null && obj instanceof String)
		{
			try
			{
				int rank = Integer.parseInt((String)obj);
				this.myRankPad.rankList.delete(rank);
			}
			catch(Exception ex)
			{
				JOptionPane.showMessageDialog(this.myRankPad, "Not a number: " + ex.getMessage(), "Problem",
						JOptionPane.ERROR_MESSAGE);				
			}
		}
	}

	// ==========================================================================
	// HELP Menu
	// ==========================================================================

	JMenu getHelpMenu()
	{
		//JMenu acceleratorsMenu = ActionHelpers.makeMenu("accelerators"); //$NON-NLS-1$ //$NON-NLS-2$
		//acceleratorsMenu.add(makeMenuItem("accelerators", "focus")); //$NON-NLS-1$ //$NON-NLS-2$

		JMenu menu = ActionHelpers.makeMenu("help"); //$NON-NLS-1$
		menu.add(makeMenuItem("help", "help")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("help", "about")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("help", "license")); //$NON-NLS-1$ //$NON-NLS-2$
		menu.add(makeMenuItem("help", "info")); //$NON-NLS-1$ //$NON-NLS-2$
		return menu;
	}

	class HelpAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			boolean showedDesktopDialog = false;
	      try
	      {
	      	if (AppStarter.isJava6Enabled())
	      	{
	      		showedDesktopDialog = AppUtils.helpToBrowser(RankPadActions.this.myRankPad);
	      	}
	      }
	      catch(NoClassDefFoundError ex)
	      {
	      	// java 4
	   		JOptionPane.showMessageDialog(RankPadActions.this.myRankPad, Jibs.getString("SortScreen.10"), //$NON-NLS-1$
	   				Jibs.getString("SortScreen.51"), JOptionPane.INFORMATION_MESSAGE); //$NON-NLS-1$
	      }
	      if (!showedDesktopDialog)
	      {
	     		JOptionPane.showMessageDialog(RankPadActions.this.myRankPad, Jibs.getString("SortScreen.10"), //$NON-NLS-1$
	   				Jibs.getString("SortScreen.51"), JOptionPane.INFORMATION_MESSAGE); //$NON-NLS-1$      	
	      }
		}
	}

	class AboutAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new AboutBox(RankPadActions.this.myRankPad);  // Info and About could be more generic...  Slightly...
		}
	}

	class InfoAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new Info(RankPadActions.this.myRankPad, RankPadActions.this.myRankPad);  
		}
	}

	class LicenseAction extends AbstractAction
	{
		public void actionPerformed(ActionEvent e)
		{
			new LicenseBox(RankPadActions.this.myRankPad);
		}
	}
	
	class TreeMenuListener implements MenuListener
	{
		public void menuSelected(MenuEvent eA)
		{
			RankPadActions.this.myRecentTreeMenu.removeAll();
			int maxCount = RankPad.clsPrefs.getInt(RECENT_TREE_COUNT, DEFAULT_MAX_COUNT);
			
			ArrayList<String> theList = PathManager.getDirectoryFromDisk(RECENT_TREE, RankPad.clsPrefs);
			for(int i=0; i<=maxCount && i < theList.size(); i++)
			{
				final String dirName = theList.get(i);
				if (dirName != null)
				{
					JMenuItem anItem = new JMenuItem(dirName);
					anItem.addActionListener(new ActionListener()
					{
						public void actionPerformed(ActionEvent eB)
						{
							RankPadActions.this.myRankPad.setTreeDirectory(new File(dirName));
						}
					});
					RankPadActions.this.myRecentTreeMenu.add(anItem);
				}
			}
		}

		public void menuDeselected(MenuEvent e)
		{
			// We don't care
		}
	
		public void menuCanceled(MenuEvent e)
		{
			// We don't care
		}
	}

	class ViewMenuListener implements MenuListener
	{
		public void menuSelected(MenuEvent eA)
		{
			RankPadActions.this.myRecentViewMenu.removeAll();
			int maxCount = RankPad.clsPrefs.getInt(RECENT_VIEW_COUNT, DEFAULT_MAX_COUNT);
			
			ArrayList<String> theList = PathManager.getDirectoryFromDisk(RECENT_VIEW, RankPad.clsPrefs);
			for(int i=0; i<=maxCount && i < theList.size(); i++)
			{
				final String dirName = theList.get(i);
				if (dirName != null)
				{
					JMenuItem anItem = new JMenuItem(dirName);
					anItem.addActionListener(new ActionListener()
					{
						public void actionPerformed(ActionEvent eB)
						{
							 RankPadActions.this.myRankPad.setRankingFile(new File(dirName));
						}
					});
					RankPadActions.this.myRecentViewMenu.add(anItem);
				}
			}
		}

		public void menuDeselected(MenuEvent e)
		{
			// We don't care
		}
	
		public void menuCanceled(MenuEvent e)
		{
			// We don't care
		}
	}
	
}
