/*
 * Copyright (c) 2010 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.textPad.ranker;

import java.io.*;
import java.util.*;

class ScoreARow
{
	RankByWordCounts ranker;
	ScoringFactors[] myFactors;
	ArrayList<ScoringFileName> myFileNameScoring = new ArrayList<ScoringFileName>();

	@SuppressWarnings("null")
	ScoreARow(RankByWordCounts rankerIn, File scoringFile) throws Exception
	{
		this.ranker = rankerIn;
		if (this.ranker.iStop) { return; }
		String errMsg = null;
		RankByWordCounts
				.logln("\n---------------------------------------------------");
		RankByWordCounts.logln("Reading in scoring file: " + scoringFile);

		BufferedReader buffRead = null;
		HashMap<String, ScoringFactors> factors = new HashMap<String, ScoringFactors>(50);
		try
		{
			// First read in the file names.
			buffRead = new BufferedReader(new FileReader(scoringFile));

			String aLine = buffRead.readLine();

			int count = 0;
			while (aLine != null)
			{
				aLine = aLine.trim();
				if (aLine.startsWith("@"))
				{
					this.myFileNameScoring.add(new ScoringFileName(aLine));
				}
				else if (!aLine.startsWith("#") && aLine.length() > 0)
				{
					ScoringFactors ws = new ScoringFactors(this.ranker, count, aLine);
					factors.put(ws.myLabel, ws);
				}
				count++;
				aLine = buffRead.readLine();
			}
			buffRead.close();
		}
		catch (Exception ex)
		{
			errMsg = ex.getMessage();
			ex.printStackTrace();
		}
		finally
		{
			try
			{
				buffRead.close();
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
		if (errMsg != null) { throw new Exception(errMsg); }

		// Now convert HashMap into an ordered array.
		// Words first then simple groups then complex groups.
		// Check for cyclic dependencies along way.
		ArrayList<ScoringFactors> orderedFactors = new ArrayList<ScoringFactors>(factors.size());

		Iterator<ScoringFactors> factorList = factors.values().iterator();
		while (factorList.hasNext())
		{
			ScoringFactors aFactor = factorList.next();
			if (!aFactor.iAmAGroup)
			{
				orderedFactors.add(aFactor);
				aFactor.iNeedArranging = false;
			}
		}

		boolean processMoreGroups = true;
		int count = 0;
		while (processMoreGroups)
		{
			processMoreGroups = processGroups(factors, orderedFactors);
			count++;
			if (count > 20) { throw new Exception(
					"Could not resolve cyclic dependencies in 20 tries."); }
		}

		// Finish populating the array.
		this.myFactors = orderedFactors.toArray(new ScoringFactors[factors.size()]);
	}

	private boolean processGroups(HashMap<String, ScoringFactors> factors, ArrayList<ScoringFactors> orderedFactors)
			throws Exception
	{
		Iterator<ScoringFactors> factorList = factors.values().iterator();
		boolean someGroupsNotAdded = false;
		while (factorList.hasNext())
		{
			ScoringFactors aFactor = factorList.next();
			if (aFactor.iAmAGroup && aFactor.iNeedArranging)
			{
				boolean iCanBeAdded = true;
				for (int i = 0; i < aFactor.myScoringWordsOrGroups.length; i++)
				{
					ScoringFactors calledFactor =  factors
							.get(aFactor.myScoringWordsOrGroups[i]);
					if (calledFactor == null) { throw new Exception("In group "
							+ aFactor.myLabel + ", cound not find label for: "
							+ aFactor.myScoringWordsOrGroups[i]); }
					if (calledFactor.iNeedArranging)
					{
						iCanBeAdded = false;
						someGroupsNotAdded = true;
						break;
					}
				}
				if (iCanBeAdded)
				{
					orderedFactors.add(aFactor);
					aFactor.iNeedArranging = false;
				}
			}
		}
		return someGroupsNotAdded;
	}

	/**
	 * words at this point is:
	 * words[0] = Original index of this row in the file as read in.
	 * The word counts don't have the label row, so at this point row + 1 should = words[0]
	 * The last column of the row is where the score goes for printing.
	 * The a copy of the score goes into IndexedScore just to do the ranking.
	 * words[words.length-1] = score;
	 */
	IndexedScore score(int row, int[] wordCounts, String filePath) throws Exception
	{
		RankByWordCounts.logln("");
		RankByWordCounts
				.logln("---------------------------------------------------");
		RankByWordCounts.logln("Row: " + wordCounts[0] + " "
				+ this.ranker.myFileList.get(row + 1));
		for (int i = 1; i < this.ranker.myTokenList.length; i++)
		{
			RankByWordCounts.logln(this.ranker.myTokenList[i] + ": "
					+ wordCounts[i]);
		}
		RankByWordCounts.logln("");

		int score = 0;

		HashMap<String, LabelAndCount> scoresByFactor = new HashMap<String, LabelAndCount>(this.myFactors.length);

		// myFactors has the list of words which will be scored.
		// myTokenList has the list of words counted in the file.
		// words has the count of the words in myTokenList
		for (int factorCounter = 0; factorCounter < this.myFactors.length; factorCounter++)
		{
			RankByWordCounts.logln("FACTORS: " + this.myFactors[factorCounter]);
			score += this.myFactors[factorCounter].score(wordCounts, scoresByFactor);
			RankByWordCounts.logln("CUMULATIVE SCORE: " + score + "\n");
		}
		
		for(ScoringFileName scorer: this.myFileNameScoring)
		{
			RankByWordCounts.logln("File Name Factor: " + scorer);
			score += scorer.score(filePath);
			RankByWordCounts.logln("CUMULATIVE SCORE: " + score + "\n");
		}

		// Now return the score in two places...
		wordCounts[wordCounts.length - 1] = score;
		return new IndexedScore(wordCounts[0], score);
	}
}
