/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.util;

import java.io.File;
import java.io.FileNotFoundException;
import java.text.DecimalFormat;
import java.util.ArrayList;

import javax.swing.JButton;
import javax.swing.JFormattedTextField;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JProgressBar;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.optimizer.NoDirectories;
import org.gerhardb.lib.io.EzLogger;
import org.gerhardb.lib.io.FileUtil;

/**
 * Some helpfull utility functions
 */
public class FileLoopHelper
{
   public static final String RELABEL_POSTFIX = "-jibs"; //$NON-NLS-1$

   static final DecimalFormat COUNT_FORMAT = new DecimalFormat("000000"); //$NON-NLS-1$


   JFrame myFrame;
   String myBaseTitle;
   JLabel myLabel;
   JProgressBar myBar;
   EzLogger myLog;

   JFormattedTextField myRelabelFld;
   int myRelabelCount = 0;
   int myMaxValue = 0;

   //===========================================================================
   //                              Constructor
   //===========================================================================
   public FileLoopHelper(
		   JFrame frame,
		   String baseTitle,
		   JFormattedTextField relabelFld,
		   JLabel label, 
		   JProgressBar bar, 
		   File logFile)
      throws Exception
   {
	   this.myFrame = frame;
	   this.myBaseTitle = baseTitle;
      this.myRelabelFld = relabelFld;
      this.myLabel = label;
      this.myBar = bar;

      this.myBar.setValue(0);
      this.myBar.setMaximum(1);
      this.myBar.setIndeterminate(false);

      this.myLog = new EzLogger(logFile);

      this.myRelabelCount = getIntFromField(this.myRelabelFld, 0);
   }

   // ==========================================================================
   // Progress Bar Reporting
   // ==========================================================================
   public JProgressBar getBarX()
   {
      return this.myBar;
   }

   public void setValue(int value)
   {
	   this.myBar.setValue(value);
	   if (this.myMaxValue > 0)
	   {
		   float percent = (int)(((float)value/(float)this.myMaxValue) * 100);
		   String msg = this.myBaseTitle + " " + percent + "%"; //$NON-NLS-1$ //$NON-NLS-2$
		   this.myFrame.setTitle(msg);
	   }
	   else
	   {
		   this.myFrame.setTitle(this.myBaseTitle);		   
	   }
   }

   public void setMaximum(int max)
   {
	   this.myMaxValue = max;
	   this.myBar.setMaximum(this.myMaxValue);
   }

   public void setIndeterminate(boolean indeterminate)
   {
	   this.myBar.setIndeterminate(indeterminate);
   }
   
   public void done()
   {
	   this.myBar.setMaximum(1);
	   this.myBar.setValue(1);
	   this.myFrame.setTitle(this.myBaseTitle + " " + Jibs.getString("FileLoopHelper.6")); //$NON-NLS-1$ //$NON-NLS-2$
	}

   // ==========================================================================
   // Other Public Functions
   // ==========================================================================
   public JLabel getLabel()
   {
      return this.myLabel;
   }

    public EzLogger getLog()
   {
      return this.myLog;
   }

   public void updateCountField()
   {
      this.myRelabelFld.setValue(new Integer(this.myRelabelCount));
   }

   public int getRelableCount()
   {
      return this.myRelabelCount;
   }

   public ArrayList<File> getFilesFromDirectories(File[] dirArray)
   {
      NoDirectories noDirectories = new NoDirectories();
      ArrayList<File> outList = new ArrayList<File>();

      this.myBar.setMaximum(dirArray.length - 1);
      // Loop over files shown on the directory panel
      for (int i = 0; i < dirArray.length; i++)
      {
         this.myBar.setValue(i);
         File dir = dirArray[i];
         this.myLabel.setText(
            Jibs.getString("FileLoopHelper.7") + " " + i + ": " + dir); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
         File[] addThese = dir.listFiles(noDirectories);
         if (addThese != null)
         {
            for (int j = 0; j < addThese.length; j++)
            {
               outList.add(addThese[j]);
            }
         }
      }
      return outList;
   }

   /**
    * Checks to see if the file has already been processed so you can skip processing.
    * @param fileName
    * @return
    */
   public static boolean alreadyProcessed(String fileName)
   {
   	return fileName.indexOf(RELABEL_POSTFIX) > -1;
   }
   
   public File renameUnique(File currFile)
   {
      String name = currFile.getName();

      // Fast exit if it already has postfix.
      if (alreadyProcessed(name))
      {
         return currFile;
      }

      String front = ""; //$NON-NLS-1$
      String back = ""; //$NON-NLS-1$
      int lastPeriod = name.lastIndexOf('.');
      if (lastPeriod < 0)
      {
         // if no ending, then use the name as the front.
         front = name;
      }
      else
      {
         if (lastPeriod > 0)
         {
            front = name.substring(0, lastPeriod);
         }
         if (lastPeriod < name.length())
         {
            back = name.substring(lastPeriod);
         }
      }

      String newName = currFile.getParent()
                       + File.separator + front + RELABEL_POSTFIX +
                       COUNT_FORMAT.format(this.myRelabelCount++)
                       + back;

      File newFile = new File(newName);
      if (newFile.exists())
      {
         this.myLog.logLine(Jibs.getString("FileLoopHelper.11") + Jibs.getString("colon") + " " + newFile //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
            + ""); //$NON-NLS-1$
      }
      else
      {
         currFile.renameTo(newFile);
         // Other processes need to know the new file name.
         return newFile;
      }
      // Other processes should continue to use the old file name.
      return currFile;
   }

   public static int getIntFromField(JFormattedTextField field, int theDefault)
   {
      try
      {
         Object nbr = field.getValue();
         if (nbr instanceof Long)
         {
            return ((Long)nbr).intValue();
         }
         else if (nbr instanceof Integer)
         {
            return ((Integer)nbr).intValue();
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      return theDefault;
   }

   public static JFormattedTextField makeRelabelFld()
   {
      return new JFormattedTextField(new Integer(0));
   }

   public static boolean checkDirectory(String dir, JButton saveBtn)
   {
      if (dir == null || dir.trim().length() == 0)
      {
         return false;
      }
      try
      {
         FileUtil.validateDirectory(dir.toString());
      }
      catch (FileNotFoundException ex)
      {
         int create = JOptionPane.showConfirmDialog(
                      saveBtn,
                      dir
                      + " " + Jibs.getString("FileLoopHelper.13"), //$NON-NLS-1$ //$NON-NLS-2$
                      Jibs.getString("FileLoopHelper.14"), //$NON-NLS-1$
                      JOptionPane.YES_NO_OPTION);
         if (create == JOptionPane.YES_OPTION)
         {
            boolean rtnMe = false;
            File file = new File(dir);
            try
            {
               rtnMe = file.mkdir();
            }
            catch (Exception exx)
            {
               JOptionPane.showMessageDialog(
                  saveBtn,
                  exx.getMessage(),
                  Jibs.getString("FileLoopHelper.15"), //$NON-NLS-1$
                  JOptionPane.ERROR_MESSAGE);
               return false;
            }
            return rtnMe;
         }
         return false;
      }
      catch (Exception ex)
      {
         return false;
      }
      return true;
   }

   public static void main(String[] args) 
   {
	   int value = 100;
	   int maxValue = 1000;
	   System.out.println(Jibs.getString("FileLoopHelper.16") + " " + value + " " + Jibs.getString("FileLoopHelper.17") + maxValue); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
	   float amtDone = (float)value/(float)maxValue*100;
	   System.out.println("amtDone  " + amtDone); //$NON-NLS-1$
	   int percent = (int)amtDone;
	   String msg = "JIBS " + percent + "%"; //$NON-NLS-1$ //$NON-NLS-2$
	   System.out.println(msg);
   }
   
}
