/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer;

import java.awt.event.ActionEvent;
import java.io.File;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ButtonGroup;
import javax.swing.JMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.SwingUtilities;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.frame.SortScreen;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.util.startup.AppPreferencesFactory;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * Wraps the preferences used by the viewer and provides an way of editing the
 * preferences.
 * 
 * @author Gerhard Beck
 */
public class ViewerPreferences
{
	public static final long serialVersionUID = 0;

	public static String JIBS_DIR_NAME = "/JIBS"; //$NON-NLS-1$
	
	private static final String PREF_PATH = 
		"/org/gerhardb/jibs/viewer/ViewerPreferences"; //$NON-NLS-1$
	private static Preferences clsPrefs;
	private static Action[] clsViewActions;
	
	// ===========================================================================
	// Constructor
	// ===========================================================================

	private ViewerPreferences()
	{
   	// Don't allow public creation.
	}

	/**
	 * @throws SecurityException
	 */
	public static void init()
	{
		// Initialize only once.
		if (clsPrefs != null) { return; }
		clsPrefs = Preferences.userRoot().node(PREF_PATH);
	}
	
	public static Preferences getPrefs()
	{
		return clsPrefs;
	}
	
	// ===========================================================================
	// Config directory
	// ===========================================================================
	private static final String JIBS_CONFIG_DIR = "jibsConfigDir"; //$NON-NLS-1$

	public static File getKeyChangerFile()
	{
		return new File(getConfigDirThumbed().getAbsoluteFile() + "/JibsMenu.xml");	 //$NON-NLS-1$
	}
	
	/**
	 * Must come before Jibs.getJibsPropFile() test because
	 * his could be set to something bogus for read-only thumbdrives.
	 * Maybe.
	 * @return
	 */
	public static File getConfigDirThumbed()
	{
		if (AppStarter.isThumbdrive())
		{
			// Have to use this because if thumbdrive it might
			// be from a read-only directory where there is 
			// no property file to get the parent directory of.
			return AppStarter.getStartupDirectory();
		}
		return getConfigDir();
	}
	
	
	/**
	 * Get the config dir.
	 * If it does not exist, defaults to
	 * for Windows: /user/Application Data/JIBS
	 * for Linux:   /user home/JIBS.
	 * Automatically initializes first time through.
	 * This is where properties goes and listing of any read-only media go.
	 * @return
	 */
	@SuppressWarnings("null")
	public static File getConfigDir()
	{
		String fileName = System.getProperty(AppPreferencesFactory.PROP_FILE_LOOKUP);
		if (fileName != null)
		{
			File propFile = new File(fileName);
			try
			{
				return propFile.getParentFile();
			}
			catch(Exception ex)
			{
				// Just in case user deleted that directory...
			}
		}
		
		File rtnMe = null;
		String path = clsPrefs.get(JIBS_CONFIG_DIR, null);
		if (path != null)
		{
			rtnMe = new File(path);
			if (rtnMe.exists() && rtnMe.isDirectory())
			{
				return rtnMe;
			}
		}
		
		String os = System.getProperty("os.name").toLowerCase(); //$NON-NLS-1$
		if (os.startsWith("windows")) //$NON-NLS-1$
		{
			File appData = new File(System.getenv("APPDATA")); //$NON-NLS-1$
			if (appData.exists() && appData.isDirectory())
			{
				rtnMe = new File(appData.getAbsolutePath()+ JIBS_DIR_NAME);
			}
		}
		else
		{
			rtnMe = new File(AppStarter.getMyPicturesDir().getAbsolutePath() + JIBS_DIR_NAME);
		}
		
		if (!rtnMe.exists())
		{
			try
			{
				rtnMe.mkdir();
			}
			catch(Exception ex)
			{
				rtnMe = AppStarter.getMyPicturesDir();
				ex.printStackTrace();
			}
		}
		
		try
		{
			setConfigDir(rtnMe);
			flush();
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		return rtnMe;	
	}

	public static void setConfigDir(File dir)
	throws Exception
	{
		if (!dir.exists() && !dir.isDirectory())
		{
			throw new Exception(Jibs.getString("ViewerPreferences.18") + dir.getAbsolutePath()); //$NON-NLS-1$
		}
		clsPrefs.put(JIBS_CONFIG_DIR, dir.getAbsolutePath());
		flush();
	}

	// ===========================================================================
	// Selected User Interface
	// ===========================================================================
	private static final String GD_FULL_SCREEN = "gdFullScreen"; //$NON-NLS-1$

	public static boolean getGDFullScreen()
	{
		return clsPrefs.getBoolean(GD_FULL_SCREEN, true);
	}

	public static void setGDFullScreen(boolean gdfs)
	{
		clsPrefs.putBoolean(GD_FULL_SCREEN, gdfs);
	}

	// ===========================================================================
	// Panel Size On Start
	// ===========================================================================
	private static final String FILE_PANEL_SIZE = "ListPicturePanelDividerLocation";  //$NON-NLS-1$
	private static final String TREE_PANEL_SIZE = "TreePanelSize";  //$NON-NLS-1$
	private static final String EXIF_PANEL_SIZE = "ExifDividerLocation";  //$NON-NLS-1$
	
	public static int getListPicturePanelDividerLocation()
	{
		return clsPrefs.getInt(FILE_PANEL_SIZE, SortScreen.FILE_MIN);
	}

	public static void setListPicturePanelDividerLocation(int size)
	{
		clsPrefs.putInt(FILE_PANEL_SIZE, size);
	}

	public static int getTreePanelSize()
	{
		return clsPrefs.getInt(TREE_PANEL_SIZE, SortScreen.TREE_MIN);
	}

	public static void setExifPanleSize(int size)
	{
		clsPrefs.putInt(EXIF_PANEL_SIZE, size);
	}

	public static int getExifPanelSize()
	{
		return clsPrefs.getInt(EXIF_PANEL_SIZE, SortScreen.EXIF_MIN);
	}

	public static void setTreePanleSize(int size)
	{
		clsPrefs.putInt(TREE_PANEL_SIZE, size);
	}

	// ===========================================================================
	// Save Enabled Screen
	// ===========================================================================
	private static final String SHOW_EXIF_MESSAGE = "showEXIFmessage"; //$NON-NLS-1$

	public static boolean showExifWarning()
	{
		return clsPrefs.getBoolean(SHOW_EXIF_MESSAGE, true);
	}

	public static void setShowExifWarning(boolean showExifEnabled)
	{
		clsPrefs.putBoolean(SHOW_EXIF_MESSAGE, showExifEnabled);
	}
	
	// ===========================================================================
	// Tree Panel
	// ===========================================================================
	private static final String COUNT_IMAGE_FILES_IN_NODES = "countOnlyImageFilesInNodes"; //$NON-NLS-1$

	public static boolean countImageFilesInNodes()
	{
		return clsPrefs.getBoolean(COUNT_IMAGE_FILES_IN_NODES, false);
	}

	public static void setCountImageFilesInNodes(boolean enabled)
	{
		clsPrefs.putBoolean(COUNT_IMAGE_FILES_IN_NODES, enabled);
	}
	
	// ===========================================================================
	// Full Screen
	// ===========================================================================
	private static final String FULL_SCREEN_START = "fullScreenStart"; //$NON-NLS-1$

	private static final String FULL_SCREEN_AUTO = "fullScreenAuto"; //$NON-NLS-1$

	private static final String FULL_SCREEN_TIME = "fullScreenTime"; //$NON-NLS-1$

	private static final String CONTINUOUS_SHOW = "continuousShow"; //$NON-NLS-1$

	// private static final String SHRINK_SHOW = "shrinkUpShow";
	private static final String BLOW_UP_SHOW = "blowUpShow"; //$NON-NLS-1$

	private static final String MAX_BLOW = "maxBlowUpMax"; //$NON-NLS-1$

	public static boolean startFullScreen()
	{
		return clsPrefs.getBoolean(FULL_SCREEN_START, false);
	}

	public static boolean immediateShow()
	{
		return clsPrefs.getBoolean(FULL_SCREEN_AUTO, false);
	}

	public static boolean continuousShow()
	{
		return clsPrefs.getBoolean(CONTINUOUS_SHOW, false);
	}

	/*
	 * public static boolean shrink() { return clsPrefs.getBoolean(SHRINK_SHOW,
	 * true); }
	 */

	public static boolean blowUp()
	{
		return clsPrefs.getBoolean(BLOW_UP_SHOW, true);
	}

	public static int getMaxBlow()
	{
		return clsPrefs.getInt(MAX_BLOW, 3);
	}

	/**
	 * Show delay
	 * 
	 * @return seconds
	 */
	public static int getSlideFlipDelay()
	{
		return clsPrefs.getInt(FULL_SCREEN_TIME, 3000);
	}

	public static void setStartFullScreen(boolean startFullScreen)
	{
		clsPrefs.putBoolean(FULL_SCREEN_START, startFullScreen);
	}

	public static void setImmediateShow(boolean startShow)
	{
		clsPrefs.putBoolean(FULL_SCREEN_AUTO, startShow);
	}

	public static void setContinuousShow(boolean continuousShow)
	{
		clsPrefs.putBoolean(CONTINUOUS_SHOW, continuousShow);
	}

	/*
	 * public static void setShrink(boolean blowUp) {
	 * clsPrefs.putBoolean(SHRINK_SHOW, blowUp); }
	 */

	public static void setBlowUp(boolean blowUp)
	{
		clsPrefs.putBoolean(BLOW_UP_SHOW, blowUp);
	}

	public static void setMaxBlow(int times)
	{
		if (times > 1)
		{
			clsPrefs.putInt(MAX_BLOW, times);
			// ViewerGlobals.getScroller().setSlideFlipDelay(milliseconds);
		}
	}

	public static void setSlideFlipDelay(int milliseconds, Scroller scroller)
	{
		if (milliseconds > 100)
		{
			clsPrefs.putInt(FULL_SCREEN_TIME, milliseconds);
			scroller.setSlideFlipDelay(milliseconds);
		}
	}

	// ===========================================================================
	// Sorter Tile Size
	// ===========================================================================
	private static final String SORTER_TILE_SIZE = "sorterTileSize"; //$NON-NLS-1$

	public static int getSorterTileSize()
	{
		return clsPrefs.getInt(SORTER_TILE_SIZE, 100);
	}

	public static void setSorterTileSize(int size)
	{
		clsPrefs.putInt(SORTER_TILE_SIZE, size);
	}

	// ===========================================================================
	// View Menus and Buttons
	// ===========================================================================

	public static final int VIEW_RESIZE = 0;
	public static final int VIEW_NO_RESIZE_KEY = 1;
	public static final int VIEW_THUMBS = 2;
	public static final int VIEW_NO_RESIZE_SCROLL = 3;
	private static final String SHOW_VIEW = "showView"; //$NON-NLS-1$
	private static final String SHOW_SORTER = "showSorter"; //$NON-NLS-1$

	public static int getView()
	{
		return clsPrefs.getInt(SHOW_VIEW, VIEW_RESIZE);
	}

	public static void setView(int view)
	{
		clsPrefs.putInt(SHOW_VIEW, view);
	}
	
	public static boolean isShowSorterShowing()
	{
		return clsPrefs.getBoolean(SHOW_SORTER, false);
	}

	public static void setShowSorterShowing(boolean view)
	{
		clsPrefs.putBoolean(SHOW_SORTER, view);
	}


	public static void setViewFrame(final IFrame frame)
	{
		frame.setWaitCursor(true);
		
		switch (getView())
		{
		case VIEW_RESIZE:
			//System.out.println("setViewFrame: VIEW_RESIZE");
			frame.setViewer(new org.gerhardb.jibs.viewer.shows.classic.SingleResize(
							frame));
			break;
		case VIEW_NO_RESIZE_KEY:
			//System.out.println("setViewFrame: VIEW_NO_RESIZE_KEY");
			frame.setViewer(
					new org.gerhardb.jibs.viewer.shows.classic.SingleNoResizeKeyControl(
							frame));
			break;
		case VIEW_THUMBS:
			//System.out.println("setViewFrame: VIEW_THUMBS");
			frame.setViewer(new org.gerhardb.jibs.viewer.shows.thumbs.Thumbnails(frame, frame.getScroller().getValueZeroBased()));
			break;
		case VIEW_NO_RESIZE_SCROLL:
			//System.out.println("setViewFrame: VIEW_NO_RESIZE_SCROLL");
			frame.setViewer(
					new org.gerhardb.jibs.viewer.shows.classic.SingleNoResizeScrollControl(
							frame));
			break;
		}
		frame.setWaitCursor(false);
	}

	// Let's the menu go away faster.
	static void setViewLater(final IFrame frame)
	{
		SwingUtilities.invokeLater(new Runnable()
		{
			public void run()
			{
				setViewFrame(frame);
			}
		});
	}

	public static Action[] getViewActions(IFrame frame)
	{
		class ResizeAction extends AbstractAction
		{
			IFrame myFrame;

			ResizeAction(IFrame aFrame)
			{
				super(Jibs.getString("ViewerPreferences.26")); //$NON-NLS-1$
				this.myFrame = aFrame;
			}

			public void actionPerformed(ActionEvent event)
			{
				ViewerPreferences.setView(VIEW_RESIZE);
				setViewLater(this.myFrame);
			}
		}

		class NoResizeKeyAction extends AbstractAction
		{
			IFrame myFrame;

			NoResizeKeyAction(IFrame aFrame)
			{
				super(Jibs.getString("ViewerPreferences.27")); //$NON-NLS-1$
				this.myFrame = aFrame;
			}

			public void actionPerformed(ActionEvent event)
			{
				ViewerPreferences.setView(VIEW_NO_RESIZE_KEY);
				setViewLater(this.myFrame);
			}
		}

		class NoResizeScrollAction extends AbstractAction
		{
			IFrame myFrame;

			NoResizeScrollAction(IFrame aFrame)
			{
				super(Jibs.getString("ViewerPreferences.29")); //$NON-NLS-1$
				this.myFrame = aFrame;
			}

			public void actionPerformed(ActionEvent event)
			{
				ViewerPreferences.setView(VIEW_NO_RESIZE_SCROLL);
				setViewLater(this.myFrame);
			}
		}

		class ThumbnailsAction extends AbstractAction
		{
			IFrame myFrame;

			ThumbnailsAction(IFrame aFrame)
			{
				super(Jibs.getString("ViewerPreferences.28")); //$NON-NLS-1$
				this.myFrame = aFrame;
			}

			public void actionPerformed(ActionEvent event)
			{
				ViewerPreferences.setView(VIEW_THUMBS);
				setViewLater(this.myFrame);
			}
		}

		if (clsViewActions == null)
		{
			Action[] x = { new ResizeAction(frame), new NoResizeScrollAction(frame),
					new NoResizeKeyAction(frame), new ThumbnailsAction(frame), };
			clsViewActions = x;
		}
		return clsViewActions;
	}

	/**
	 * Makes a new set for each UI its used on.
	 * 
	 * @return Sort Menu
	 */
	public static void makeViewMenu(IFrame frame, JMenu viewMenu)
	{
		Action[] viewActions = getViewActions(frame);
		final JRadioButtonMenuItem[] menuItems = new JRadioButtonMenuItem[viewActions.length];
		viewMenu.addMenuListener(new MenuListener()
		{
			public void menuCanceled(MenuEvent e)
			{
				// We don't care
			}

			public void menuDeselected(MenuEvent e)
			{
				// We don't care
			}

			public void menuSelected(MenuEvent e)
			{
				switch (ViewerPreferences.getView())
				{
				case ViewerPreferences.VIEW_RESIZE:
					menuItems[0].setSelected(true);
					break;
				case ViewerPreferences.VIEW_NO_RESIZE_SCROLL:
					menuItems[1].setSelected(true);
					break;
				case ViewerPreferences.VIEW_NO_RESIZE_KEY:
					menuItems[2].setSelected(true);
					break;
				case ViewerPreferences.VIEW_THUMBS:
					menuItems[3].setSelected(true);
					break;
					
				}
			}
		});

		ButtonGroup viewGroup = new ButtonGroup();
		for (int i = 0; i < viewActions.length; i++)
		{
			menuItems[i] = new JRadioButtonMenuItem(viewActions[i]);
			viewMenu.add(menuItems[i]);
			viewGroup.add(menuItems[i]);
		}
	}

	// ===========================================================================
	// Helper Methods
	// ===========================================================================

	/**
	 * Clears all preferences, typically for testing a new user.
	 */
	/*
	public static void clear() throws BackingStoreException
	{
		clsPrefs.clear();
		clsPrefs.flush();
	}
*/
	// ===========================================================================
	// Hide EXIF panel on main viewer
	// ===========================================================================
	private static final String EXIF_PANEL_SHOWING = "exifPanelShowing"; //$NON-NLS-1$

	public static boolean isExifPanelShowing()
	{
		return clsPrefs.getBoolean(EXIF_PANEL_SHOWING, true);
	}

	public static void setExifPanelShowing(boolean enabled)
	{
		clsPrefs.putBoolean(EXIF_PANEL_SHOWING, enabled);
	}

	// ===========================================================================
	// Other Public Functions
	// ===========================================================================
	public static void flush()
	{
		try
		{
			clsPrefs.flush();
		}
		catch (BackingStoreException ex)
		{
			ex.printStackTrace();
		}				
	}

	// ===========================================================================
	// Main
	// ===========================================================================
	public static void main(String[] args)
	{
		try
		{
			System.out.println(System.getProperty("user.home"));  //$NON-NLS-1$
			System.out.println(AppStarter.getMyDocumentsDir());
			System.out.println(AppStarter.getMyPicturesDir());
			System.out.println(System.getenv("APPDATA")); //$NON-NLS-1$

			/*
			ViewerPreferences.init();
			ViewerPreferences.clear();
			System.out.println(ViewerPreferences.getLastViewedDirectory());
			try
			{
				// Windows does not give correct root if you map away.
				java.awt.FileDialog dialog = new java.awt.FileDialog((java.awt.Frame)null);
				System.out.println(dialog.getDirectory());
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			*/
			
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}
}
