/*
 * Copyright (c) 2008 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.jibs.viewer.frame;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.util.*;

import javax.swing.*;
import javax.swing.table.*;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.swing.*;
import org.gerhardb.lib.util.ActionHelpers;

/**
 * Should be in org.gerhardb.jibs.viewer.frame, but uses row sorting from Java 6.
 * @author  Gerhard Beck
 */
public class KeyChangerDialog extends javax.swing.JDialog
{
	private static final int MENU_ITEM_COL = 0;
	private static final int TEXT_COL = 1;
	private static final int TOOL_TIP_COL = 2;
	private static final int MNEMONIC_COL = 3;
	private static final int ACCELERATOR_COL = 4;
	private static Object[] COL_NAMES = { Jibs.getString("KeyChangerDialog.0"), Jibs.getString("KeyChangerDialog.1"), Jibs.getString("KeyChangerDialog.2"), //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			Jibs.getString("KeyChangerDialog.3"), Jibs.getString("KeyChangerDialog.4") }; //$NON-NLS-1$ //$NON-NLS-2$

	JPanel myPanel = new JPanel(new BorderLayout());
	SortScreen mySortScreen;
	KeyTableModel myTableData;
	JTable myTable;
	JButton mySaveBtn = new JButton(Jibs.getString("KeyChangerDialog.5")); //$NON-NLS-1$

	public KeyChangerDialog(SortScreen screen)
	{
		super(screen, Jibs.getString("KeyChangerDialog.6"), true); //$NON-NLS-1$
		this.mySortScreen = screen;
		HashMap<String, Action> currentActionMap = this.mySortScreen.getActions().getCurrentActions();
		this.myTableData = new KeyTableModel(convertDataToShow(currentActionMap),
				COL_NAMES);
		layoutDialog();
	}

	void save()
	{
		HashMap<String, Action> currentActionMap = this.mySortScreen.getActions().getCurrentActions();

		CellEditor editor = this.myTable.getCellEditor();
		if (editor != null)
		{
			editor.stopCellEditing();
		}
		for (int row = 1; row < this.myTableData.getRowCount(); row++)
		{
			Action action = currentActionMap.get(this.myTableData.getValueAt(
					row, 0));

			String text = (String) this.myTableData.getValueAt(row, TEXT_COL);
			if (text != null && text.trim().length() > 0)
			{
				//System.out.println("text: " + text);
				action.putValue(Action.NAME, text);
			}
			
			String tooltip = (String) this.myTableData.getValueAt(row, TOOL_TIP_COL);
			if (tooltip != null && tooltip.trim().length() > 0)
			{
				//System.out.println("tooltip: " + tooltip);
				action.putValue(Action.SHORT_DESCRIPTION, tooltip);
			}

			Object mnemonicObject = this.myTableData.getValueAt(row, MNEMONIC_COL);
			if (mnemonicObject instanceof Integer)
			{
				Integer mnemonicCode = (Integer)this.myTableData.getValueAt(row, MNEMONIC_COL);
				action.putValue(Action.MNEMONIC_KEY, mnemonicCode);
			}
			else if (mnemonicObject instanceof Character)
			{
				Character mnemonicCode = (Character)this.myTableData.getValueAt(row, MNEMONIC_COL);
				int keyCode = ActionHelpers.getKeyCode(mnemonicCode.charValue());
				action.putValue(Action.MNEMONIC_KEY, new Integer(keyCode));
			}

			// First, clear out old values.
			action.putValue(Action.ACCELERATOR_KEY, null);
			// Now get new values.
			Object accelObj = this.myTableData.getValueAt(row, ACCELERATOR_COL);
			if (accelObj != null)
			{
				System.out.println("accelObj.getClass(): " + accelObj.getClass());
				System.out.println("accelObj: " + accelObj);
				if (accelObj instanceof String)
				{
					String accelerator = (String)accelObj;
					if (accelerator.trim().length() > 0)
					{
						System.out.println("accelerator: " + accelerator);
						KeyStroke ks = KeyStroke.getKeyStroke(accelerator);
						if (ks != null)
						{
							action.putValue(Action.ACCELERATOR_KEY, ks);
						}
						else
						{
							System.out.println("NOT A VALID ACCELERATOR"); //$NON-NLS-1$
						}
					}
				}
				else if  (accelObj instanceof KeyStroke)
				{
					action.putValue(Action.ACCELERATOR_KEY, accelObj);				
				}
				else if  (accelObj instanceof KeyStrokeDisplay)
				{
					action.putValue(Action.ACCELERATOR_KEY, ((KeyStrokeDisplay)accelObj).keyStroke);				
				}
			}
		}

		/*
		Iterator it = currentActionMap.entrySet().iterator();
		while(it.hasNext())
		{
			Map.Entry item = (Map.Entry)it.next();
			AbstractAction anAction = (AbstractAction)item.getValue();
			System.out.print(item.getKey() + ": ");
			System.out.println(anAction.toString());
			System.out.println(anAction.getValue(Action.ACCELERATOR_KEY));
			System.out.println(anAction.getValue(Action.MNEMONIC_KEY));
			System.out.println(anAction.getValue(Action.NAME));
			System.out.println(anAction.getValue(Action.SHORT_DESCRIPTION));
			System.out.println("");
		}
		*/
		File writeFile = ViewerPreferences.getKeyChangerFile();
		KeyChangerXML kc = new KeyChangerXML();
		kc.writeActionsToXML(writeFile, currentActionMap);

		setVisible(false);
		dispose();
	}

	void restore()
	{
		setWaitCursor(true);
		HashMap<String, Action> defaultActionMap = this.mySortScreen.getActions().getDefaultActions();
		this.myTableData = new KeyTableModel(convertDataToShow(defaultActionMap),
				COL_NAMES);
		this.myTable.setModel(this.myTableData);
		// Setting model makes table forget editors.
		addCellEditors(this.myTable);
		this.mySaveBtn.setEnabled(true);
		setWaitCursor(false);
	}

	private void layoutDialog()
	{
		addWindowListener(new WindowAdapter()
		{
			@Override
			public void windowClosing(WindowEvent e)
			{
				cancel();
			}
		});

		this.mySaveBtn.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				save();
			}
		});

		JButton cancelBtn = new JButton(Jibs.getString("cancel")); //$NON-NLS-1$
		cancelBtn.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				cancel();
			}
		});

		JButton restoreBtn = new JButton(Jibs.getString("KeyChangerDialog.8")); //$NON-NLS-1$
		restoreBtn.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				restore();
			}
		});

		this.myTable = new KeyChangerJTable(this.myTableData);
		this.myTable.setAutoCreateRowSorter(true);
		//table.putClientProperty("terminateEditOnFocusLost", Boolean.TRUE);
		
		addCellEditors(this.myTable);
		
		// Sort on the initial column, "menu item"
		DefaultRowSorter<?,?> sorter = (DefaultRowSorter<?,?>) this.myTable.getRowSorter();
		ArrayList<RowSorter.SortKey> keys = new ArrayList<RowSorter.SortKey>();
		keys.add(new RowSorter.SortKey(0, SortOrder.ASCENDING));
		sorter.setSortKeys(keys);
		sorter.sort();

		JScrollPane tblScroll = new JScrollPane(this.myTable,
				ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_AS_NEEDED);

		JPanel myBottomPanel = new JPanel(new FlowLayout(FlowLayout.CENTER));
		myBottomPanel.add(this.mySaveBtn);
		myBottomPanel.add(cancelBtn);
		myBottomPanel.add(restoreBtn);

		this.myPanel.add(tblScroll, BorderLayout.CENTER);
		this.myPanel.add(myBottomPanel, BorderLayout.SOUTH);

		addAccelerators(this.mySaveBtn);
		this.getContentPane().add(this.myPanel);
		this.setSize(new Dimension(700, 550));
		//pack();
		SwingUtils.centerOnScreen(this);

		this.setVisible(true);
	}

	private void addAccelerators(JButton saveBtn)
	{
		InputMap map = null;

		// Accelerator - ok
		// Turned off because this does not consume the Enter key which then
		// goes on to bring up the move dialog.  From line 429 in Scroller.
		this.getRootPane().setDefaultButton(saveBtn);
		
		// 429 solution
		// This works - ALT K is how to activate.
		saveBtn.setMnemonic(KeyEvent.VK_S);

		// Accelerator - cancel
		class Dismiss extends AbstractAction
		{
			Dismiss()
			{
				super("dismiss"); //$NON-NLS-1$
			}

			public void actionPerformed(ActionEvent e)
			{
				cancel();
			}
		}
		Action dismiss = new Dismiss();
		String cancelID = "dismiss"; //$NON-NLS-1$
		this.myPanel.getActionMap().put(cancelID, dismiss);
		map = this.myPanel.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
		map.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), cancelID);
	}

	void cancel()
	{
		setVisible(false);
		dispose();
	}

	private void setWaitCursor(boolean wait)
	{
		if (wait)
		{
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		}
		else
		{
			setCursor(Cursor.getDefaultCursor());
		}
	}

	private Object[][] convertDataToShow(HashMap<String, Action> map)
	{
		Object[][] rtnMe = new Object[map.entrySet().size()][5];
		int i = 0;
		//Object[] items = map.entrySet().toArray();
		//Arrays.sort(items);
		Iterator<Map.Entry<String, Action>> it = map.entrySet().iterator();
		while (it.hasNext())
		{
			Map.Entry<String, Action> item = it.next();
			AbstractAction anAction = (AbstractAction) item.getValue();
			rtnMe[i][MENU_ITEM_COL] = item.getKey();
			rtnMe[i][TEXT_COL] = anAction.getValue(Action.NAME);
			rtnMe[i][TOOL_TIP_COL] = anAction.getValue(Action.SHORT_DESCRIPTION);
			rtnMe[i][MNEMONIC_COL] = anAction.getValue(Action.MNEMONIC_KEY);
			rtnMe[i][ACCELERATOR_COL] = anAction.getValue(Action.ACCELERATOR_KEY);
			if (rtnMe[i][ACCELERATOR_COL] != null)
			{
				if (rtnMe[i][ACCELERATOR_COL] instanceof String)
				{
					KeyStrokeDisplay ksd = new KeyStrokeDisplay();
					ksd.keyStroke = KeyStroke.getKeyStroke((String)rtnMe[i][ACCELERATOR_COL]);
					rtnMe[i][ACCELERATOR_COL] = ksd;
				}
				if (rtnMe[i][ACCELERATOR_COL] instanceof KeyStroke)
				{
					KeyStrokeDisplay ksd = new KeyStrokeDisplay();
					ksd.keyStroke = (KeyStroke)rtnMe[i][ACCELERATOR_COL];
					rtnMe[i][ACCELERATOR_COL] = ksd;
				}
			}
			i++;
		}
		return rtnMe;
	}

	class KeyTableModel extends javax.swing.table.DefaultTableModel
	{
		KeyTableModel(Object[][] data, Object[] columnNames)
		{
			super(data, columnNames);
		}

		//@Override
		@Override
		public boolean isCellEditable(int rowIndex, int columnIndex)
		{
			switch (columnIndex)
			{
			case MENU_ITEM_COL:
				return false;
			case TEXT_COL:
				return true;
			case TOOL_TIP_COL:
				return true;
			case MNEMONIC_COL:
				return true;
			case ACCELERATOR_COL:
				return true;
			default:
				return false;
			}
		}
	}

	// ==========================================================================
	// Cell rendering and editing
	// ==========================================================================
	
	TableCellRenderer mnemonicRenderer = new MnemonicRenderer();
	TableCellRenderer validatingRenderer = new ValidatingRenderer();
	
	class KeyChangerJTable extends JTable
	{
		KeyChangerJTable(TableModel tableModel)
		{
			super(tableModel);
		}
		
	    @Override
		public TableCellRenderer getCellRenderer(int row, int column) 
	    {
	        if (column == MNEMONIC_COL)
	        {
	            return KeyChangerDialog.this.mnemonicRenderer;
	        }
	        else if (column == ACCELERATOR_COL)
	        {
	            return KeyChangerDialog.this.validatingRenderer;
	        }	        
	        else if (column == TEXT_COL)
	        {
	            return KeyChangerDialog.this.validatingRenderer;
	        }	        
	        return super.getCellRenderer(row, column);
	    }
	}
	
	class ValidatingRenderer extends DefaultTableCellRenderer
	{
		public ValidatingRenderer()
		{
			super();
		}

		@Override
		public void setValue(Object value)
		{
			super.setValue(value);
		}
		
		@Override
		public Component getTableCellRendererComponent(JTable table,
            Object value,
            boolean isSelected,
            boolean hasFocus,
            int row,
            int column)
		{
			//System.out.println("getTableCellRendererComponent Called  cell: " + value + " r: " + row + " c: " + column);
			if (isCellValid(row, column, value))
			{
				this.setBackground(Color.WHITE);
				//this.setForeground(Color.WHITE);			
			}
			else
			{
				this.setBackground(Color.RED);			
				//this.setForeground(Color.BLACK);			
			}
			return super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
		}
	}
	
	private void setOKStatus(int changedRow)
	{
		boolean okStatus = true;
		int rowCount = this.myTableData.getRowCount();
		
		// Text check
		for(int row=0; row<rowCount; row++)
		{
			//System.out.println(myTableData.getValueAt(row, TEXT_COL) + "    "
			//		+ myTableData.getValueAt(row, ACCELERATOR_COL));
			
			Object data = this.myTableData.getValueAt(row, TEXT_COL);
			if (!goodTextCol(data))
			{
				okStatus = false;
				break;			
			}
		}
		
		// Accelerator check
		KeyStrokeDisplay[] accelerators = new KeyStrokeDisplay[rowCount];
		for(int row=0; row<rowCount; row++)
		{
			KeyStrokeDisplay ksd = null;
			Object value = this.myTableData.getValueAt(row, ACCELERATOR_COL);
			if (value == null)
			{
				// Everything is OK!
			}
			else if (value instanceof KeyStrokeDisplay)
			{
				//System.out.println("KeyStrokeDisplay found");
				ksd = (KeyStrokeDisplay)value;
			}
			else if (value instanceof String)
			{
				//System.out.println("string found");
				String ksText = ((String)value).trim();
				if (ksText.length() > 0)
				{
					KeyStroke ks = KeyStroke.getKeyStroke(ksText);
					if (ks == null)
					{			
						// Everything is OK!
					}
					else
					{
						ksd = new KeyStrokeDisplay();
						ksd.keyStroke = ks;
						this.myTableData.setValueAt(ksd, row, ACCELERATOR_COL);
					}
				}
			}
			else
			{
				okStatus = false;
			}
			if (ksd != null)
			{
				ksd.duplicate = false; // Set default value
				accelerators[row] = ksd;
				for(int i=0; i<row; i++)
				{
					if (ksd.equals(accelerators[i])) 
					{
						accelerators[i].duplicate = true;					
						accelerators[row].duplicate = true;
						if (i != changedRow)
						{
							this.myTableData.fireTableRowsUpdated(i, i);
						}
						if (row != changedRow)
						{
							this.myTableData.fireTableRowsUpdated(row, row);							
						}
						okStatus = false;
					}
				}
			}
		}
		this.mySaveBtn.setEnabled(okStatus);
	}
	
	private boolean goodAcceleratorCol(Object value)
	{
		if (value == null)
		{
			return true;
		}
		
		if (value instanceof KeyStrokeDisplay)
		{
			return !((KeyStrokeDisplay)value).duplicate;
		}
		if (value instanceof String)
		{
			String strValue = (String)value;
			if (strValue.trim().length() == 0)
			{
				return true;
			}			
			KeyStroke ks = KeyStroke.getKeyStroke(strValue);
			if (ks == null)
			{
				return false;
			}
			return true;
		}
		return false;
	}
		
	boolean isCellValid(int row, int column, Object value)
	{
		setOKStatus(row);
		if (column == TEXT_COL)
		{
			return goodTextCol(value);
		}
		else if (column == ACCELERATOR_COL)
		{
			return goodAcceleratorCol(value);
		}
		return true;
	}
	
	boolean goodTextCol(Object value)
	{
		if (value == null)
		{
			return false;
		}
		else if (!(value instanceof String))
		{
			return false;
		}
		else if (((String)value).trim().length() == 0)
		{
			return false;
		}
		else
		{
			return true;
		}		
	}
	
	class MnemonicRenderer extends DefaultTableCellRenderer
	{
		public MnemonicRenderer()
		{
			super();
		}

		@Override
		public void setValue(Object value)
		{
			if (value instanceof Integer)
			{
				int keyCode = ((Integer)value).intValue();
				String keyText = Character.toString((char)keyCode);
				super.setText(keyText);
				return;
			}
			super.setValue(value);
		}
	}
	
	private void addCellEditors(JTable table)
	{
		TableColumn mnemonicColumn = table.getColumnModel().getColumn(MNEMONIC_COL);
		JComboBox comboBox = new JComboBox();
		mnemonicColumn.setCellEditor(new MnemonicEditor(comboBox));
 		
		/*
		TableColumn acceleratorColumn = table.getColumnModel().getColumn(ACCELERATOR_COL);
		final JTextField acceleratorTextField = new JTextField();	
		acceleratorColumn.setCellEditor(new DefaultCellEditor(acceleratorTextField));
		
		class AcceleratorListener extends KeyAdapter
		{
			public void keyTyped(KeyEvent event) 
			{
				System.out.println(event);
				//acceleratorTextField.transferFocus();
			}
		}		
		acceleratorTextField.addKeyListener(new AcceleratorListener());
		*/
	}

	class KeyStrokeDisplay
	{
		KeyStroke keyStroke;
		boolean duplicate = false;
		
		@Override
		public String toString()
		{
			return this.keyStroke.toString();
		}
		
		@Override
		public boolean equals(Object value)
		{
			if (value == null || !(value instanceof KeyStrokeDisplay))
			{
				return false;
			}
			KeyStrokeDisplay that = (KeyStrokeDisplay)value;
			if (this.keyStroke == null && that.keyStroke == null)
			{
				return true;
			}
			if (this.keyStroke == null && that.keyStroke != null)
			{
				return false;
			}
			if (this.keyStroke != null && that.keyStroke == null)
			{
				return false;
			}
			return this.keyStroke.equals(that.keyStroke);
		}
		
		@Override
		public int hashCode()
		{
			return this.keyStroke.hashCode();
		}
	}
	
	class MnemonicEditor extends DefaultCellEditor
	{
		JComboBox myComboBox;
		
		MnemonicEditor(JComboBox comboBox)
		{
			super(comboBox);
			this.myComboBox = comboBox;
			this.myComboBox.addItem(Jibs.getString("KeyChangerDialog.9")); //$NON-NLS-1$
			for(int i = 65; i < 91; i++) 
			{
	         this.myComboBox.addItem(new Character((char)i));
	      }
		}
		
		@Override
		public Component getTableCellEditorComponent(JTable table, Object value, boolean isSelected, int row, int column)
		{
			System.out.println("cell: " + value + " r: " + row + " c: " + column); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			char selectedValue = 0;
			if (value != null && value instanceof Integer)
			{
				selectedValue = ActionHelpers.getKeyChar(((Integer)value).intValue());
			}
			this.myComboBox.removeAllItems();
			this.myComboBox.addItem(Jibs.getString("KeyChangerDialog.13")); //$NON-NLS-1$
			
			HashSet<Character> showThese = new HashSet<Character>(50);
			String text = ((String)KeyChangerDialog.this.myTable.getValueAt(row, TEXT_COL)).toUpperCase();
			if (text != null)
			{
				char[] chars = text.toCharArray();
				for(int i = 0; i < chars.length; i++) 
				{
					if (chars[i] > 60 && chars[i] < 91)
					{
						showThese.add(new Character(chars[i]));
					}
		      }
				Object[] objArray = new Object[showThese.size()];
				showThese.toArray(objArray);
				Arrays.sort(objArray);
				for(int i=0; i<objArray.length; i++)
				{
					this.myComboBox.addItem(objArray[i]);
					if (value != null)
					{
						System.out.println("kcd objArray[i]: " + objArray[i] + " class: " + objArray[i].getClass().getName() + " value: " + value + " class: " + value.getClass().getName()); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
					}
					char aChar = ((Character)objArray[i]).charValue();
					int itemValue = Character.getNumericValue(((Character)objArray[i]).charValue());
					System.out.println("kcd item: " + itemValue + " selectedValue: " + selectedValue + " aChar: " + aChar);					 //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
					if (aChar == selectedValue)
					{
						this.myComboBox.setSelectedItem(objArray[i]);						
					}
				}
			}
			return super.getTableCellEditorComponent(table, value, isSelected, row, column);
		}
	}
	

	// ==========================================================================
	// Main
	// ==========================================================================
	public static void main(String[] args)
	{
		try
		{
			/*
			JFrame f = new JFrame("Key Changer Test"); //$NON-NLS-1$
			f.getContentPane().add(new JLabel("test")); //$NON-NLS-1$
			f.setSize(300, 300);
			f.addWindowListener(new WindowAdapter()
			{
				public void windowClosing(WindowEvent evt)
				{
					System.exit(0);
				}
			});
			f.setVisible(true);
			*/
			SortScreen s = new SortScreen();
			Thread.sleep(2000);
			new KeyChangerDialog(s);
		}
		catch (Exception ex)
		{
			System.out.println(ex.getMessage());
		}

		System.out.println("Main Exiting Normally"); //$NON-NLS-1$
	}

}
