/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.frame;

import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.io.File;
import java.util.ArrayList;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.PicInfoDialogBase;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.image.ImageUtilDrew;
import org.gerhardb.lib.swing.JPanelRows;

/**
 * Saves the currently selected file.
 */
public class SaverActions
{
	SortScreen mySortScreen;

	private Action mySavePictureAction = new SavePictureAction();
	private Action mySaveAsPictureAction = new SaveAsPictureAction();
	private Action myRotateRightAction = new RotateRightAction();
	private Action myRotateLeftAction = new RotateLeftAction();
	private Action myShrinkAction = new ShrinkAction();
	private Action myGrowAction = new GrowAction();

	// =========================================================================
	// Constructor
	// =========================================================================
	SaverActions(SortScreen ss)
	{
		this.mySortScreen = ss;
	}

	// ==========================================================================
	// Public Methods
	// ==========================================================================
	
	public void enableSave(boolean enabled)
	{
		this.mySavePictureAction.setEnabled(enabled);	
	}
	
	public void enableSaveAs(boolean enabled)
	{
		this.mySaveAsPictureAction.setEnabled(enabled);	
	}
	
	public void enableRotation(boolean enabled)
	{
		this.myRotateRightAction.setEnabled(enabled);	
		this.myRotateLeftAction.setEnabled(enabled);	
	}
	
	public void enableShrinkGrow(boolean enabled)
	{
		this.myShrinkAction.setEnabled(enabled);	
		this.myGrowAction.setEnabled(enabled);	
	}
	
	public boolean cancelSaveForExif(File sampleFile)
	{
		if (ViewerPreferences.showExifWarning())
		{
			// Only warn if there is EXIF data.
			ArrayList<?> exifTags = ImageUtilDrew.getExifTags(this.mySortScreen.getScroller().getCurrentFile());
			if (exifTags.size() > 0)
			{
				return cancelSaveForExifDialog(sampleFile);
			}
		}
		// EXIF warning turned off OR no exif tags.
		return false;
	}
	
	public boolean cancelSaveForExifDialog(File sampleFile)
	{
		final JCheckBox showAgainChkBox = new JCheckBox(Jibs
				.getString("ViewerPreferencesDialog.93")); //$NON-NLS-1$
		JOptionPane pane = new JOptionPane(warningPanel(showAgainChkBox),
				JOptionPane.WARNING_MESSAGE, JOptionPane.YES_NO_OPTION);

		String[] options =
		{
				Jibs.getString("SaverActions.0"), Jibs.getString("SaverActions.1"), Jibs.getString("SaverActions.2") //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		};
		pane.setOptions(options);
		pane.setInitialSelectionValue(options[1]);
		pane.setInitialValue(options[1]);

		boolean reshow = true;
		Object selectedValue = null;
		
		while(reshow)
		{
			reshow = false;
			JDialog dialog = pane.createDialog(this.mySortScreen, Jibs.getString("SaverActions.3")); //$NON-NLS-1$
			dialog.setVisible(true);
			selectedValue = pane.getValue();
			if (selectedValue == null)
			{
				return true;
			}
			if (options[2].equals(selectedValue))
			{
				// "See EXIF data" picked.
				PicInfoDialogBase info = new PicInfoDialogBase(this.mySortScreen, this.mySortScreen.getScroller());
			   info.display(sampleFile, null, -1, -1, false); 
			   reshow = true;
			}
			dialog.setVisible(false);
			dialog.dispose();
		}
		
		// Save change
		boolean showAgain = showAgainChkBox.isSelected();
		ViewerPreferences.setShowExifWarning(showAgain);

		if (options[1].equals(selectedValue))
		{
			// "Cancel" picked.
			return true;
		}
		
		// "Save" picked.
		return false;		
	}

	public Action getSavePictureAction()
	{
		return this.mySavePictureAction;
	}

	public Action getSaveAsPictureAction()
	{
		return this.mySaveAsPictureAction;
	}

	public Action getRotateRightAction()
	{
		return this.myRotateRightAction;
	}

	public Action getRotateLeftAction()
	{
		return this.myRotateLeftAction;
	}

	public Action getShrinkAction()
	{
		return this.myShrinkAction;
	}

	public Action getGrowAction()
	{
		return this.myGrowAction;
	}

	// =========================================================================
	// Save Items
	// =========================================================================

	private class SavePictureAction extends AbstractAction
	{
		protected SavePictureAction() { /* just to get rid of synthetic warning...*/ } 
		
		public void actionPerformed(ActionEvent e)
		{
			SaverActions.this.mySortScreen.myShow.getSaver().save(false);
		}

		@Override
		public void setEnabled(boolean setEnabled)
		{
			super.setEnabled(setEnabled);
		}
	}

	private class SaveAsPictureAction extends AbstractAction
	{
		protected SaveAsPictureAction() { /* just to get rid of synthetic warning...*/ } 
		
		public void actionPerformed(ActionEvent e)
		{
			SaverActions.this.mySortScreen.myShow.getSaver().save(true);
		}

		@Override
		public void setEnabled(boolean setEnabled)
		{
			super.setEnabled(setEnabled);
		}
	}

	// =========================================================================
	// Tool Bar Buttons
	// Save is not here because of the ViewerPreference to turn save off
	// globably, and ViewerPreferences is not part of the library.
	// =========================================================================

	private class RotateRightAction extends AbstractAction
	{
		protected RotateRightAction() { /* just to get rid of synthetic warning...*/ } 
		
		public void actionPerformed(ActionEvent e)
		{
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ));
			SaverActions.this.mySortScreen.myShow.getSaver().rotateRight();
			SaverActions.this.mySortScreen.getScroller().editedImage();
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
		}
	}

	private class RotateLeftAction extends AbstractAction
	{
		protected RotateLeftAction() { /* just to get rid of synthetic warning...*/ } 
		
		public void actionPerformed(ActionEvent e)
		{
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ));
			SaverActions.this.mySortScreen.myShow.getSaver().rotateLeft();
			SaverActions.this.mySortScreen.getScroller().editedImage();
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
		}
	}

	private class ShrinkAction extends AbstractAction
	{
		protected ShrinkAction() { /* just to get rid of synthetic warning...*/ } 
		
		public void actionPerformed(ActionEvent e)
		{
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ));
			SaverActions.this.mySortScreen.myShow.getSaver().shrink();
			SaverActions.this.mySortScreen.getScroller().editedImage();
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
		}
	}

	private class GrowAction extends AbstractAction
	{
		protected GrowAction() { /* just to get rid of synthetic warning...*/ } 
		
		public void actionPerformed(ActionEvent e)
		{
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ));
			SaverActions.this.mySortScreen.myShow.getSaver().grow();
			SaverActions.this.mySortScreen.getScroller().editedImage();
			SaverActions.this.mySortScreen.setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
		}
	}

	private JPanel warningPanel(JCheckBox showAgainChkBox)
	{
		showAgainChkBox.setSelected(ViewerPreferences.showExifWarning());

		JPanelRows rtnMe = new JPanelRows();
		JPanel aRow = rtnMe.topRow();
		aRow.add(new JLabel(Jibs.getString("SaverActions.4"))); //$NON-NLS-1$

		aRow = rtnMe.nextRow();
		aRow.add(new JLabel(Jibs.getString("SaverActions.5"))); //$NON-NLS-1$

		aRow = rtnMe.nextRow();
		aRow.add(new JLabel(Jibs.getString("SaverActions.6"))); //$NON-NLS-1$

		aRow = rtnMe.nextRow();
		aRow.add(showAgainChkBox);

		return rtnMe;
	}

}
