/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.frame;

import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;

import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JToolBar;
import javax.swing.SwingConstants;
import javax.swing.tree.TreePath;
import javax.swing.undo.UndoableEdit;
import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.IShow;
import org.gerhardb.jibs.viewer.PicInfoScrollPane;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.dirtree.filelist.FileList;
import org.gerhardb.jibs.viewer.shows.comic.ComicScreen;
import org.gerhardb.jibs.viewer.shows.full.FullScreen;
import org.gerhardb.jibs.viewer.sorter.SorterPanel;
import org.gerhardb.jibs.viewer.tree.FileTreePopUp;
import org.gerhardb.lib.dirtree.DirectoryTreeNode;
import org.gerhardb.lib.dirtree.ExtendedDirectoryTree;
import org.gerhardb.lib.dirtree.IScrollDirTree;
import org.gerhardb.lib.dirtree.IUndo;
import org.gerhardb.lib.dirtree.filelist.FileListManager;
import org.gerhardb.lib.dirtree.filelist.FileListPlugins;
import org.gerhardb.lib.dirtree.rdp.BasicOptionsManager;
import org.gerhardb.lib.dirtree.rdp.IListShowTree;
import org.gerhardb.lib.dirtree.rdp.ILoad;
import org.gerhardb.lib.dirtree.rdp.ListShowTreeCoordinator;
import org.gerhardb.lib.dirtree.rdp.MoveManager;
import org.gerhardb.lib.dirtree.rdp.PathManager;
import org.gerhardb.lib.dirtree.rdp.RDPmanager;
import org.gerhardb.lib.dirtree.rdp.RDPplugins;
import org.gerhardb.lib.dirtree.rdp.TreeManagerRDP;
import org.gerhardb.lib.dirtree.rdp.TreeManagerSet;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.io.FilenameFileFilter;
import org.gerhardb.lib.scroller.KeypadOps;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.scroller.ScrollerListener;
import org.gerhardb.lib.scroller.ScrollerSlider;
import org.gerhardb.lib.swing.SwingUtils;
import org.gerhardb.lib.swing.UIMenu;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.Range;
import org.gerhardb.lib.util.app.AppUtils;
import org.gerhardb.lib.util.startup.AppPluginStartup;
import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.Loading;

// import javax.swing.plaf.synth.*;

/**
 * MAIN WINDOW FOR APPLICATION. Jibs starts Viewer which starts this JFrame.
 * This JFrame has all of the sorting features and does the initialization.
 * <p>
 */
public class SortScreen extends JFrame implements IListShowTree, IFrame, IUndo, IScrollDirTree, TreeManagerSet, FileListPlugins, ILoad
{
	public static final int TREE_MIN = 100;
	public static final int FILE_MIN = 100;
	public static final int EXIF_MIN = 0;
	
	boolean iNeedToFinalize = true;
	boolean iWasMaximized = false;
	boolean iNeedToInitializeShow = true;
	
	// KEY UI PARTS - before scrolling and paneling
	// There are four parts from left to right:
	// 1. The File List   myFileList                myListComponent
	// 2. The EXIF Info   myExifInfo
	// 3. The Picture     myShow.getImageDisplay()
	// 4. The Tree        myDirectoryTree           myTreePanel
	
	// Let's set up the base panels these go in.		
	private JSplitPane myExifPicturePanel;
	JSplitPane myListPicturePanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);	
	JSplitPane myListPictureTreePanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
	
	IShow myShow = new LoadingShow(); 
	
   PicInfoScrollPane myExifInfo;	
	
	JMenuBar myMenuBar = new JMenuBar();
	ScrollerSlider mySlider;

	JToolBar myToolBar = new JToolBar();
	
	// Managers - some of which need the UI stuff above.
	ViewerActions myActions;   
	UIMenu myUImenu = new UIMenu(this, getAppIDandTargetType());
	Popups myPopups = new Popups(this);
	public SlideShowManager mySlideShowManager;
	public SorterPanel mySorterPanel; // after ViewerPreferences initializes.
	
	JMenu myMenuView;
	JMenuItem myItemViewOptions;
	private Loading myLoading;
	private boolean iAmFullScreen = false;
	JPanel myCenterPanel;	
	private JFrame myCurrentTopFrame = this;
	
	ListShowTreeCoordinator myLST;
	
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public SortScreen()
	{
		super("JIBS"); //$NON-NLS-1$
		try
		{
			ViewerPreferences.init();
		}
		catch (Exception ex)
		{
			// Catch SecurityException and anything else that can go wrong.
			JOptionPane.showMessageDialog(this,
					Jibs.getString("SortScreen.2") + ex.getMessage(), //$NON-NLS-1$
					Jibs.getString("SortScreen.3"), JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
			ex.printStackTrace();
		}
		this.myLST = new LSTX(this, this);		
		
		// -----------------------------------------------------------------------
		// Set up the coordinator
		// -----------------------------------------------------------------------

		this.myActions = new ViewerActions(this);
		try
		{
			this.myActions.new MenuOverrides();	
		}
		catch(NoClassDefFoundError ex)
      {
      	ex.printStackTrace();
      }
		catch(UnsupportedClassVersionError ex)
      {
      	ex.printStackTrace();
      }
		
		this.myLST.addActions(this.myActions);
			
		this.mySlideShowManager = new SlideShowManager(this);
		
		// -----------------------------------------------------------------------
		// Make sure we know how to exit!! 
		//(Still falling through the AWT event thread.)
		// -----------------------------------------------------------------------
		super.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		addWindowListener(
			new WindowAdapter()
			{
				@Override
				public void windowClosing(WindowEvent e)
				{
					exit();
				}
			});		
		Runtime.getRuntime().addShutdownHook(new Thread()
		{
			@Override
			public void run()
			{
				exit();
			}
		});
		// AWT event thread comes to rest while thread in loading does real work.

		// Done in separate thread so you can see directories scroll by.
		// Loading comes up in the AWT thread.
		this.myLoading = new Loading(this, "JIBS");
		
		Runnable runMe = new Runnable()
		{
			public void run()
			{
				try
				{
					build();
					/*
					java.util.Date stopTime = new Date();
					long elapsed = (stopTime.getTime() - 
						Jibs.TIME_STARTED.getTime())/1000;
					System.out.println("Overall Startup Time: " + elapsed); //$NON-NLS-1$
					*/
				}
				catch (Exception ex)
				{
					JOptionPane.showMessageDialog(SortScreen.this,
							Jibs.getString("SortScreen.5") + ex.getMessage() //$NON-NLS-1$
									+ "\n" + Jibs.getString("SortScreen.7") //$NON-NLS-1$ //$NON-NLS-2$
									+ "gerhardb@users.sourceforge.net", //$NON-NLS-1$
							Jibs.getString("SortScreen.9"), JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
					ex.printStackTrace();
				}
			}
		};
		// This starts the real loading process from a new thread inside
		// Loading. That way, messages can show in the AWT thread in Loading.
		this.myLoading.start(runMe);
	}

	
	// ==========================================================================
	// IListShowTree
	// ==========================================================================
	public String getAppIDandTargetType()
	{
		return "Images";
	}
	
	public JFrame getTopFrame()
	{
		return this;
	}

	public void showFile(File file)
	{
		this.myShow.showImage();
	}
	
	public ScrollerListener getScrollerListener()
	{
		// Manually set by application depending on circumstances.
		// Different for different show types.
		return null;
	}
	
	public FilenameFileFilter getFilenameFileFilter()
	{
		return ImageFactory.getImageFactory().getFilter();
	}
	
	public boolean getCountTargetFiles()
	{
		return ViewerPreferences.countImageFilesInNodes();
	}
	
	public JPopupMenu getPopupMenu(TreePath path)
	{
		return new FileTreePopUp(this.myLST.getTree(), path, ViewerPreferences.countImageFilesInNodes());
	}

	// ==========================================================================
	// Public
	// ==========================================================================
	
	public ListShowTreeCoordinator getLST()
	{
		return this.myLST;
	}
	
	public ExtendedDirectoryTree getTree()
	{
		return this.myLST.getTree();
	}
	
	public FileListManager getFileListManager()
	{
		return this.myLST.getFileListManager();
	}
	
	public TreeManagerRDP getTreeManager()
	{
		return this.myLST.getTreeManager();
	}

	// ==========================================================================
	// FileListPlugins
	// ==========================================================================
	public Range showingIndexes()
	{
		return this.myShow.showingIndexes();
	}
	
	// ==========================================================================
	// TreeManagerSet
	// ==========================================================================
	public Scroller getScroller(){ return this.myLST.getScroller(); }
	public RDPplugins getRDPplugins(){ return this.myLST; }
	public PathManager getPathManager()
	{ 
		return this.myLST.getPathManager();
	}
	public MoveManager getMoveManager(){ return getRDPmanager().getMoveManager(); }
	
	// ==========================================================================
	// FileListPlugins
	// ==========================================================================
	public void reloadScroller(int scrollBackTo, boolean reloadCache)
	{
		this.myShow.reloadScroller(scrollBackTo, reloadCache);
	}
	
	public void reloadScroller()
	{
		this.myShow.reloadScroller();
	}
	
   
	// ==========================================================================
	// LSTX
	// ==========================================================================
   public boolean isFullScreen()
   {
   	return this.iAmFullScreen;
   }   
   
	// ==========================================================================
	// IScrollDirTree
	// ==========================================================================
	public void setCurrentDirectory(DirectoryTreeNode[] nodes, boolean nonTarget)
	{
		getFileListManager().setTargetDir(nodes, nonTarget);
	}

	public void setCurrentDirectory(DirectoryTreeNode node, boolean nonTarget)
	{
		getFileListManager().setTargetDir(node, nonTarget);
	}
	
	public void selectFiles(File[] filesToSelect)
	{
		getFileList().selectFiles(filesToSelect);
	}
	
	public void fileListUpdated()
	{
		this.mySlideShowManager.setSlideShowButtons(true);
	}
	
	// ==========================================================================
	// IUndo
	// ==========================================================================
	public void addUndoable(UndoableEdit edit){getRDPmanager().getMoveManager().addUndoable(edit);}
	public void redid(){	this.myShow.reloadScroller(); } 	
	public void undid(File file){ this.myShow.jumpTo(file); }

	// ==========================================================================
	// IFrame
	// ==========================================================================
	public void setWaitCursor(boolean wait)
	{
		if (wait)
		{
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		}
		else
		{
			setCursor(Cursor.getDefaultCursor());
		}
	}
	
	public void gotoFullScreen(boolean startNow)
	{
		/*
		// bogus results in linux.
		GraphicsEnvironment ge = 
			GraphicsEnvironment.getLocalGraphicsEnvironment();
		if (!ge.getDefaultScreenDevice().isFullScreenSupported())
		{
			JOptionPane.showMessageDialog(this, 
					"Full Screen not supported by default graphics device", 
					"System Issue", JOptionPane.INFORMATION_MESSAGE);
			return;
		}
		*/
		if (super.getExtendedState() == Frame.MAXIMIZED_BOTH)
		{
			this.iWasMaximized = true;
		}
		else
		{
			this.iWasMaximized = false;
		}
		FullScreen fs = new FullScreen(SortScreen.this, startNow);
		this.myCurrentTopFrame = fs.getFrame();
		super.setVisible(false);
		this.iAmFullScreen = true;
	}

	public void gotoComicScreen()
	{
		if (super.getExtendedState() == Frame.MAXIMIZED_BOTH)
		{
			this.iWasMaximized = true;
		}
		else
		{
			this.iWasMaximized = false;
		}

		ComicScreen comics = new ComicScreen(SortScreen.this);
		if (comics.isFullScreen())
		{
			this.iAmFullScreen = true;
			this.myCurrentTopFrame = comics.getFrame();
			super.setVisible(false);
		}
	}	
	
	public ViewerActions getActions(){return this.myActions;}	
	public JFrame getFrame(){return this;}
	public ScrollerSlider getSlider(){return this.mySlider;}
	public FileList getFileList()
	{
		return this.myLST.getFileList();
	}
	
	public JFrame getCurrentTopFrame()
	{
		return this.myCurrentTopFrame;
	}
	
	/**
	 * Reshows the main frame. Typically called when returning from full screen.
	 */
	public void gotoRegularScreen()
	{
		this.iAmFullScreen = false;
		this.myCurrentTopFrame = this;
		this.myShow.resetFocus();
		this.myShow.showImage();
		if (this.iWasMaximized)
		{
			super.setExtendedState(Frame.MAXIMIZED_BOTH);
		}
		initializeShow();
		super.setVisible(true);
	}

	public void setViewer(IShow show)
	{
		super.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				
		statusCurrentPage();
		if (this.myShow != null)
		{
			this.myShow.disconnect();
		}
		this.myShow = show;

		show.getSaver().resetToolbar();
		
		if (this.myItemViewOptions != null)
		{
			this.myMenuView.remove(this.myItemViewOptions);
		}
		this.myItemViewOptions = this.myShow.makePreferencesItem();
		if (this.myItemViewOptions != null)
		{
			this.myMenuView.add(this.myItemViewOptions);
		}

		this.myShow.connect();
		JComponent picturePanel = this.myShow.getImageDisplay();

		if (this.myExifPicturePanel != null)
		{
			int exifDividerLocation = this.myExifPicturePanel.getDividerLocation();
			this.myExifInfo.setPicturePanel(picturePanel);
			this.myExifPicturePanel.setRightComponent(picturePanel);
			this.myExifPicturePanel.setDividerLocation(exifDividerLocation);
		}
		else
		{
			this.myListPicturePanel.setRightComponent(picturePanel);
		}
		
		this.myLST.getScroller().requestFocus();

		super.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}

	public void showHelp()
	{
		boolean showedDesktopDialog = false;
      try
      {
      	if (AppStarter.isJava6Enabled())
      	{
      		showedDesktopDialog = AppUtils.helpToBrowser(this);
      	}
      }
      catch(NoClassDefFoundError ex)
      {
      	// java 4
   		JOptionPane.showMessageDialog(this, Jibs.getString("SortScreen.10"), //$NON-NLS-1$
   				Jibs.getString("SortScreen.51"), JOptionPane.INFORMATION_MESSAGE); //$NON-NLS-1$
      }
      if (!showedDesktopDialog)
      {
     		JOptionPane.showMessageDialog(this, Jibs.getString("SortScreen.10"), //$NON-NLS-1$
   				Jibs.getString("SortScreen.51"), JOptionPane.INFORMATION_MESSAGE); //$NON-NLS-1$      	
      }
	}

	/**
	 * Set status bar to current page.
	 */ 
	public void statusCurrentPage()
	{
		this.myLST.getStatusBarManager().statusCurrentPage();
	}

   public KeypadOps getKeypadOps(){return this.getRDPmanager();}
	
	public ExtendedDirectoryTree getExtendedDirectoryTree(){return getTree();}
	
	// ==========================================================================
	// OTHER PUBLIC FUNCTIONS
	// ==========================================================================	
   
	public IShow getShow()
	{
		return this.myShow;
	}

	public RDPmanager getRDPmanager()
	{
		//return myRDPmanagerXXX;
		return this.myLST.getRDPmanager();
	}
	
	
	public Popups getPopups(){return this.myPopups;}
	
	// ==========================================================================
	// BUILD
	// ==========================================================================
	/**
	 * Called from separate thread in Loading.
	 * Only one thread going at this point (beside AWT thread).
	 */
	void build()
	{
		loadingMsg("Waking Up!"); //$NON-NLS-1$

		// Set the three key elements.
		this.mySlider = new ScrollerSlider(SwingConstants.VERTICAL, this.myLST.getScroller());
		this.mySlider.setInverted(true);
		this.mySlider.applyComponentOrientation(ComponentOrientation.RIGHT_TO_LEFT);
		
		getFileList().setModel(this.myLST.getScroller());
		this.myLST.getScroller().getScrollerKeyListener().setKeypadOps(getRDPmanager());

		File loadFile = new File(this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_TREE));
		this.myLST.load(loadFile, this.myLoading, this, Thread.currentThread());
				
		loadingMsg(Jibs.getString("SortScreen.13")); //$NON-NLS-1$

		//myLastViewedDir = myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_LAST_VIEWED);

		enableEvents(AWTEvent.WINDOW_EVENT_MASK);
		try
		{
			initMenuAndToolbar();
			layoutComponents();
			getRDPmanager().getMoveManager().updateUndoRedoButtons();
			this.mySlideShowManager.setupSlideShowListener();
		}
		catch (Exception e)
		{
			e.printStackTrace();
			System.out.println("*** Unkonwn abnormal termination ***"); //$NON-NLS-1$
			System.exit(0);
		}

		this.myLST.getScroller().setSlideFlipDelay(ViewerPreferences
				.getSlideFlipDelay());

		this.setIconImage(Icons.getIcon(Icons.JIBS_16).getImage());
		
		this.myUImenu.initLookAndFeel();
		
		// Size to 75% of screen.
		SwingUtils.sizeScreen(this, .75f);
		
		// Do after sizing (pack or validate)...
		SortScreen.this.validate();
		// Had to use center panel because myListPictureTreePanel has a width at
		// startup different than at shutdown.
		this.myListPictureTreePanel.setDividerLocation(this.myCenterPanel.getWidth() - ViewerPreferences.getTreePanelSize());
		
		SwingUtils.centerOnScreen(SortScreen.this); // seems to need to be here also
		
		//showDefaultDirectory();
		
		// Move to bottom of menu.
		if (this.myItemViewOptions != null)
		{
			this.myMenuView.remove(this.myItemViewOptions);
			this.myMenuView.add(this.myItemViewOptions);
		}		
		System.out.println("Loading Thread Complete.  Now joining awtComplete()");
	}
	
	boolean shouldBeFullScreen()
	{
		return ViewerPreferences.startFullScreen()
		&& this.mySlideShowManager.myFullScreenBtn.isEnabled();
	}
	
	/**
	 * Put setVisable on the AWT Event Queue to avoid following error message:
	 * Exception in thread "AWT-EventQueue-0" java.lang.NullPointerException
	 * at javax.swing.plaf.basic.BasicSliderUI.calculateTrackBuffer(Unknown Source)
	 *
	 */
	public void awtComplete()
	{
		showDefaultDirectory();
		
		// Show the Full Screen if requested AND we have a directory to show!
		if (shouldBeFullScreen())
		{
			// Get rid of the intro box
			this.myLoading.dispose();
			this.myLoading = null;
			gotoFullScreen(ViewerPreferences.immediateShow());
		}
		else
		{
			gotoRegularScreen();
			initializeShow();			

			this.myLST.getScroller().requestFocus();

			// Get rid of the intro box
			this.myLoading.dispose();
			this.myLoading = null;

			/*
			String SHOW_BAD_NEWS = "SortScreen.ShowBadNews"; //$NON-NLS-1$
			boolean showBadNews = ViewerPreferences.getPrefs().getBoolean(
					SHOW_BAD_NEWS, true);
			if (showBadNews)
			{
				String msg = Jibs.getString("SortScreen.16") //$NON-NLS-1$
						+ Jibs.getString("SortScreen.17") //$NON-NLS-1$
						+ Jibs.getString("SortScreen.18") //$NON-NLS-1$
						+ Jibs.getString("SortScreen.19") //$NON-NLS-1$
						+ Jibs.getString("SortScreen.20"); //$NON-NLS-1$

				JOptionPane.showMessageDialog(this, msg, Jibs.getString("SortScreen.21"), //$NON-NLS-1$
						JOptionPane.INFORMATION_MESSAGE);
				ViewerPreferences.getPrefs().putBoolean(SHOW_BAD_NEWS, false);
				try
				{
					ViewerPreferences.getPrefs().flush();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
			*/
		}

		System.out.println("\n" //$NON-NLS-1$
				+ "**************************************************** \n" //$NON-NLS-1$
				+ "*         JIBS Background Population Ended         * \n" //$NON-NLS-1$
				+ "**************************************************** \n"); //$NON-NLS-1$
		
		String JIBS_2_8_0_NEEDS_INIT = "JIBS_2_8_0_NEEDS_INIT";
		if (ViewerPreferences.getPrefs().getBoolean(JIBS_2_8_0_NEEDS_INIT, true))
		{
         JOptionPane.showMessageDialog(
            	this,
               "Settings may have been reset",
               "Note",
               JOptionPane.PLAIN_MESSAGE);   		
			
         // Show with directory tab showing.
         this.myPopups.showSetup(1);

			ViewerPreferences.getPrefs().putBoolean(JIBS_2_8_0_NEEDS_INIT, false);
		}
	}
	
	synchronized void initializeShow()
	{
		if (this.iNeedToInitializeShow)
		{
			this.iNeedToInitializeShow = false;
			ViewerPreferences.setViewFrame(SortScreen.this);
			SwingUtils.centerOnScreen(SortScreen.this);
		}		
	}
	
	void showDefaultDirectory()
	{
		// Decide what directory to start with
		// AFTER we have decided the size so the picture does not influences
		// size.

		String dir = null;
		if (PathManager.overrideStartPassedIn())
		{		
			dir = this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_START);
		}
		else
		{
			BasicOptionsManager bom = this.myLST.getBasicOptionsManager();
			switch (bom.getStartType())
			{
			case BasicOptionsManager.START_PARTICULAR_DIRECTORY:
				dir = this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_START);
				try
				{
					getFileListManager().setTargetDir(this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_START));
				}
				catch (Exception ex)
		      {
		      	ex.printStackTrace();
		      }
				break;
			case BasicOptionsManager.START_LAST_VIEWED_DIRECTORY:
				dir = this.myLST.getPathManager().getDirectoryAbsolute(PathManager.DIR_LAST_VIEWED);
				break;
			case BasicOptionsManager.START_NO_DIRECTORY:
			default:
				break;
			}
		}
		
		try
		{
			if (dir != null)
			{
				getFileListManager().setTargetDir(dir);
			}
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		this.mySlideShowManager.setSlideShowButtons(true);
	}
	
	// ==========================================================================
	// Component Layout
	// ==========================================================================
	
	private void layoutComponents() throws Exception
	{
		// Set up application
		this.setSize(new Dimension(600, 600));
		this.setJMenuBar(this.myMenuBar);

		JComponent listComponent = new JScrollPane(getFileList());
		JPanel treePanel = new JPanel(false); 

		// Provide minimum sizes for the two components in the split pane
		// Picture is done in coordinator
		listComponent.setMinimumSize(new Dimension(FILE_MIN, 50));
		treePanel.setMinimumSize(new Dimension(TREE_MIN, 50));

		
		if ( ViewerPreferences.isExifPanelShowing())
		{
			this.myExifInfo = new PicInfoScrollPane(true);	
			this.myExifPicturePanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);

			ScrollerListener picChangeListener = new ScrollerListener()
			{
			   public void scrollerChanged(ScrollerChangeEvent ce)
			   {
			   	File currentFile = SortScreen.this.myLST.getScroller().getCurrentFile();
					// 31 May 2009 Coordinated Fix
			   	// Without this fix, the threading caused this code to run when
			   	// there was no currentFile (the file had been moved so you could
			   	// not retrieve it from the file system until you wait for the 
			   	// scroller to reset itself).  The fix causes this to not be called
			   	// until the scroller has reloaded itself.  The fix was to add
			   	// AWT waits
			   	SortScreen.this.myExifInfo.update(currentFile, 
				   		SortScreen.this.myShow.getImageDisplay().getWidth(), 
				   			SortScreen.this.myShow.getImageDisplay().getHeight());
			   }
			};
			this.myLST.getScroller().addScrollerListener(picChangeListener);
			
			// Adjustments for the EXIF pane.
			this.myExifInfo.setPicturePanel(this.myShow.getImageDisplay());
			this.myExifPicturePanel.setLeftComponent(this.myExifInfo);
			this.myExifPicturePanel.setRightComponent(this.myShow.getImageDisplay());
			this.myExifPicturePanel.setOneTouchExpandable(true);
			this.myExifPicturePanel.setDividerLocation(ViewerPreferences.getExifPanelSize());
			
			// Put it where we can see it.
			this.myListPicturePanel.setRightComponent(this.myExifPicturePanel);
		}
		else
		{
			// Just use the picture
			this.myListPicturePanel.setRightComponent(this.myShow.getImageDisplay());
		}
		
		// Create a split pane with the two scroll panes in it.
		this.myListPicturePanel.setLeftComponent(listComponent);
		//myListPicturePanel.setRightComponent(myExifPicturePanel);
		this.myListPicturePanel.setOneTouchExpandable(true);
		this.myListPicturePanel.setDividerLocation(ViewerPreferences.getListPicturePanelDividerLocation());

		//  Create a split pane with the two scroll panes in it.
		this.myListPictureTreePanel.setLeftComponent(this.myListPicturePanel);
		this.myListPictureTreePanel.setRightComponent(treePanel);		
		this.myListPictureTreePanel.setOneTouchExpandable(true);
		this.myListPictureTreePanel.setResizeWeight(1.0);
		
		this.mySorterPanel = new SorterPanel(this);
		
		this.myCenterPanel = new JPanel(new BorderLayout());
		this.myCenterPanel.add(this.mySlider, BorderLayout.WEST);
		this.myCenterPanel.add(this.myListPictureTreePanel, BorderLayout.CENTER); 
		this.myCenterPanel.add(this.mySorterPanel, BorderLayout.SOUTH);		
		
		JPanel mainPanel = new JPanel(new BorderLayout());
		mainPanel.add(this.myToolBar, BorderLayout.NORTH);
		mainPanel.add(this.myCenterPanel, BorderLayout.CENTER);
		mainPanel.add(this.myLST.getStatusBarManager().getStatusPanel(), BorderLayout.SOUTH);		
		
		// Set up content pane
		this.getContentPane().add(mainPanel);

		// myStatusPanel.add(myLoadedPanel, BorderLayout.EAST);
		// myLoadedPanel.setPreferredSize(new Dimension(120, 5));

		// Set up list panel
		// myListComponent.setVisible( false );

		// Set up Tree panel
		/*
		JPanel treeScrollPanel = new JPanel();	
		treeScrollPanel.add(new JScrollPane(myDirectoryTree));
		// Force the tree inside to blow up to fill the entire space.
		treeScrollPanel.setLayout(new GridLayout());		
		myTreePanel.setLayout(new BorderLayout());		
		myTreePanel.add(treeScrollPanel, BorderLayout.CENTER);
      */
		treePanel.setLayout(new GridLayout());		
		treePanel.add(new JScrollPane(getTree()));

	}

	// ==========================================================================
	// Menu
	// ==========================================================================
	private void initMenuAndToolbar() throws Exception
	{
		this.myLST.getScroller().setFrame(this);

		// Set up menu
		this.myMenuView = this.myActions.getViewMenu(this);
		
		this.myMenuBar.add(this.myActions.getFileMenu());
		this.myMenuBar.add(this.myActions.getEditMenu());
		this.myMenuBar.add(this.myLST.getSortManager().getSortMenu());
		this.myMenuBar.add(this.myMenuView);
		this.myMenuBar.add(this.myActions.getToolsMenu());
		this.myMenuBar.add(this.myActions.getToysMenu());
		this.myMenuBar.add(this.myUImenu.populateUIMenu(ActionHelpers.makeMenu("ui"))); //$NON-NLS-1$
		this.myMenuBar.add( Box.createHorizontalGlue() );
		this.myMenuBar.add(this.myActions.getHelpMenu());

		// TOOLBAR SETUP
		this.myToolBar.add(this.myActions.makeSaveButton());
		this.myToolBar.add(this.myActions.getToolBarButton("file", "saveas")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.myActions.getToolBarButton("file", "rename")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.addSeparator();

		this.myToolBar.add(getRDPmanager().getMoveManager().getUndoButton()); 
		this.myToolBar.add(getRDPmanager().getMoveManager().getRedoButton());
		this.myToolBar.addSeparator();

		getFileListManager().addButtonsToToolBar(this.myToolBar, this.myActions);
		this.myToolBar.add(this.myActions.getToolBarButton("view", "reload")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.addSeparator();

		this.myToolBar.add(this.myActions.getToolBarButton("edit", "left")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.myActions.getToolBarButton("edit", "right")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.myActions.getToolBarButton("edit", "shrink")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.myActions.getToolBarButton("edit", "grow")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.addSeparator();

		this.myToolBar.add(this.mySlideShowManager.getFullScreenButton());
		this.myToolBar.add(this.myActions.getToolBarButton("view", "comic")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.mySlideShowManager.getFullScreenSlideShowButton());
		this.myToolBar.add(this.mySlideShowManager.getSlideShowButton());
		this.myToolBar.addSeparator();

		JButton clearBtn = this.myActions.getToolBarButton("edit", "clear"); //$NON-NLS-1$ //$NON-NLS-2$
		SelectedItemCountListener selectionCouter = 
			new SelectedItemCountListener(getFileList(), clearBtn);
		this.myToolBar.add(selectionCouter.getToolBarLabel());
		this.myToolBar.add(clearBtn);
		
		// BIG GAP - to stuff on the right
		// Separator is needed when the toolbar is undocked.
		this.myToolBar.addSeparator();
		this.myToolBar.add(javax.swing.Box.createHorizontalGlue());

		getRDPmanager().addButtonsToToolbar(this.myToolBar);
		this.myToolBar.addSeparator();

		this.myToolBar.add(this.myActions.getToolBarButton("help", "about")); //$NON-NLS-1$ //$NON-NLS-2$
		this.myToolBar.add(this.myActions.getToolBarButton("help", "help")); //$NON-NLS-1$ //$NON-NLS-2$
	}

	// ===========================================================================
	// ===========================================================================
	// WORK FUNCTIONS
	// ===========================================================================
	// ===========================================================================
	// ===========================================================================
	// NonPublic Methods
	// ===========================================================================

	void loadingMsg(final String msg)
	{
		if (this.myLoading != null)
		{
			this.myLoading.setMessage(msg);
		}
	}
	
	// ---------------------------------------------------------------------------


	void exit()
	{
 		if (this.iNeedToFinalize)
		{
 			setWaitCursor(true);
			this.myShow.disconnect();
			this.iNeedToFinalize = false;
			if (this.myExifPicturePanel != null)
			{
				ViewerPreferences.setExifPanleSize(this.myExifPicturePanel.getDividerLocation());
			}
			ViewerPreferences.setListPicturePanelDividerLocation(this.myListPicturePanel.getDividerLocation());
			// Had to use center panel because myListPictureTreePanel has a width at
			// startup different than at shutdown.
			int w = this.myCenterPanel.getWidth()- this.myListPictureTreePanel.getDividerLocation();
			ViewerPreferences.setTreePanleSize(w);
			this.myLST.storeDirectories();
			this.myLST.getPathManager().flush();
			ViewerPreferences.flush();
			System.out.println("" //$NON-NLS-1$
					+ "**************************************************** \n" //$NON-NLS-1$
					+ "*               JIBS Exited Normally               * \n" //$NON-NLS-1$
					+ "**************************************************** \n"); //$NON-NLS-1$
			AppStarter.closeDownApp();
			setWaitCursor(false);
			System.out.println(AppPluginStartup.JIBS_OUT); //$NON-NLS-1$
			System.exit(0);
		}
	}
}

/*
 * EventQueue.invokeLater(new Runnable() { public void run() {
 * SortScreen.this.myLoading.myCurrentLbl.setText(msg); } });
 */