/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.classic;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.datatransfer.DataFlavor;
import java.awt.dnd.*;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.font.FontRenderContext;
import java.awt.font.TextLayout;
import java.io.File;

import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ScrollPaneConstants;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.IRevert;
import org.gerhardb.jibs.viewer.ISave;
import org.gerhardb.jibs.viewer.IShow;
import org.gerhardb.jibs.viewer.PicInfoDialog;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.dirtree.filelist.FileList;
import org.gerhardb.jibs.viewer.sorter.TileConstants;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.util.Range;

/**
 * JPanel implementation of showing a single image.
 */
public abstract class Single extends JPanel implements IShow, IRevert, DropTargetListener
{
	static protected final Color	OK_COLOR					= Color.black;	
	static protected final Color	EMPTY_COLOR				= Color.cyan;
	static protected final Color	NFG_COLOR				= Color.green;

	// -------------------- Things passed into me -------------------------------
	protected IFrame					myViewerFrame;
	protected FileList				myFileList;

	// -------------------- Things I create ---------------------
	protected int myShownWidth;
	protected int myShownHeight;
	protected SingleSaver mySaver;
	private boolean iShowText = false;
	private JComponent myTextPane;
	private JEditorPane myEditorPane;

	// ==========================================================================
	// Constructor
	// ==========================================================================
	public Single(IFrame mf)
	{
		// Set up double buffered
		super(true);
		
		this.myViewerFrame = mf;
		this.myFileList = this.myViewerFrame.getFileList();
		this.mySaver = new SingleSaver(
				this.myViewerFrame.getActions().getSaverActions(),
				mf.getFrame(),
				this,
				this.myViewerFrame.getScroller());
		
		this.myViewerFrame.getFileList().getFileListSelectionModel().setMouseAlwaysAdds(false);
		this.myViewerFrame.getFileList().getFileListSelectionModel().clearSelection();
		
		super.setFocusable(true);
		resetFocus();

		super.addMouseListener(new PictureMouseListener());
		
		// All drag support is in here!
		new SingleDrag(this, mf);
		new DropTarget(this, this);
		
		// Setup for Showing Text
		this.myEditorPane = new JEditorPane();
		this.myEditorPane.setEditable(false);
		this.myEditorPane.setDisabledTextColor(Color.black);
		this.myEditorPane.setEditorKit(JEditorPane
					.createEditorKitForContentType("text/html")); //$NON-NLS-1$
		this.myTextPane = new JScrollPane(this.myEditorPane,
				ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS,
				ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
  		this.setLayout(new BorderLayout());

	}
   
	/**
	 * All important paint method.  This is where image gets shown.
	 * Actually images shown by sub classes showImage().
	 * This allows overriding to show text.
	 */
   @Override
	public void paint(Graphics g)
   {  
   	/*
   	 * Show Text kinda works, but not reliably enough.  
   	 * You can get to it when non-image files are shown.  
   	 * To make reliable, would need to update switch the Single panel at 
   	 * display time vice mucking with paint in such a weird way.
   	 * ViewerPreferencesDialog has aRow.add(myShowText); commented out.
   	 * Commenting back in would cause text files to always be shown.
   	 * Text files are read in from the ImageFactory.
   	 */
   	this.iShowText = showText();
    	if (!this.iShowText)
   	{
     		this.removeAll();
   		showImage(g);
   	}
   }
   
   abstract void showImage(Graphics g);
	
	// ==========================================================================
	// IRevert
	// ==========================================================================
	public void revert()
	{
		this.myViewerFrame.gotoFullScreen(false);
	}

	public boolean isFullScreen()
	{
		return false;
	}
	
   // ==========================================================================
	// IShow Implementation
	// ==========================================================================	
	public ISave getSaver()
	{
		return this.mySaver;
	}
	
	public JComponent getImageDisplay()
	{
		return this;
	}

	public void connect()
	{
		this.myViewerFrame.getScroller().addScrollerListener(this);
		super.addKeyListener(this.myViewerFrame.getScroller().getScrollerKeyListener());		
	}

	public void disconnect()
	{
		this.myViewerFrame.getScroller().removeScrollerListener(this);
		super.removeKeyListener(this.myViewerFrame.getScroller().getScrollerKeyListener());				
	}

	public void jumpTo(File file)
	{
		reloadScroller();
		this.myViewerFrame.getScroller().jumpTo(file);
		showPageFromScroller();
	}
	
	public void reloadScroller()
	{
		this.myViewerFrame.getScroller().reloadScroller();
	}

	public void reloadScroller(int index, boolean reloadCache)
	{
		this.myViewerFrame.getScroller().reloadScroller(index, reloadCache);
	}

	public void resetFocus()
	{
		this.myViewerFrame.getScroller().setAutoFocus(this);
	}
	
	public void swapPictureFocus()
	{
		if (this.myFileList.hasFocus())
		{
			this.myViewerFrame.getScroller().forceFocus();
		}
		else
		{
			this.myFileList.grabFocus();
		}
	}

	public void showPageFromScroller()
	{
		showImage();
      this.myViewerFrame.statusCurrentPage();
      if (this.myFileList.isVisible())
      {
         this.myFileList.sliderAdjusted();
      }		
	}

	public void startSlideShow()
	{
		this.myViewerFrame.getScroller().startSlideShow(ViewerPreferences
				.continuousShow());
	}

	public void info(StringBuffer sb)
	{
		// We don't care		
	}

	public Range showingIndexes()
	{
		int index = this.myViewerFrame.getScroller().getValueZeroBased();
		return new Range(index, index);
	}
	
	// ==========================================================================
	// ScrollerListener Implemenation
	// ==========================================================================
	public void scrollerChanged(ScrollerChangeEvent ce)
	{
		switch (ce.getEventType())
		{
		case ScrollerChangeEvent.CURRENT_IMAGE_CHANGED:
			if (!this.myViewerFrame.getScroller().getValueIsAdjusting())
			{
				showImage();
			}
			break;
		case ScrollerChangeEvent.CURRENT_PAGE_CHANGED:
			if (!this.myViewerFrame.getScroller().getValueIsAdjusting())
			{
				showPageFromScroller();
			}
			break;
		case ScrollerChangeEvent.ONE_FILE_REMOVED:
      case ScrollerChangeEvent.LIST_RELOADED:
			if (this.myViewerFrame.getScroller().getMaximumZeroBased() < 0)
			{
				repaint();
			}
			showPageFromScroller();
			break;
		}
	}

	// ==========================================================================
	// Protected
	// ==========================================================================
	
	private boolean showText()
	{
   	boolean foundTextFile = false;
   	File currentFile = this.myViewerFrame.getScroller().getCurrentFile();
		if (currentFile != null && currentFile.exists())
		{
	   	String lowerCaseName = currentFile.getName().toLowerCase();
	   	for(int i=0;i<ImageFactory.TEXT_ENDINGS.length;i++)
	   	{
		   	if (lowerCaseName.endsWith(ImageFactory.TEXT_ENDINGS[i]))
		   	{
		   		foundTextFile = true;
		   		break;
		   	}
	   	}
		}
   	if (foundTextFile)
   	{ 
   		if (!this.iShowText)
   		{
   			this.add(this.myTextPane, BorderLayout.CENTER);
   		}
			String url = "file:///" + currentFile;
			//System.out.println("Single SHOWING TEXT URL: " + url);
			try
			{
				this.myEditorPane.setPage(url);
				this.myEditorPane.setCaretPosition(0);
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}			
			this.myEditorPane.repaint();
   		return true;
   	}
		return false;
	}
	
	protected boolean iCanResize()
	{
		return true;
	}
	
	protected void showNoPicture(Graphics g)
	{
		super.setBackground(NFG_COLOR);
		super.paintComponent(g); // Fill with background color.
		
		Graphics2D g2 = (Graphics2D)g;
		Font font = new Font("Default", Font.PLAIN, 24); //$NON-NLS-1$
		FontRenderContext fontRenderContext = g2.getFontRenderContext();

		TextLayout text = new TextLayout(Jibs.getString("Single.1"), font, fontRenderContext); //$NON-NLS-1$
		
		// Text is draw up from the Ascent and down from the Descent.
		int xCenter = getWidth() / 2;
		int yCenter = getHeight() / 2;
		
		
		int textX = xCenter - (int)text.getBounds().getWidth()/2;
		// This puts the text at the bottom of the box.
		int textY = (int) (yCenter - text.getDescent());
		//textY = (int) (textY + tlName.getAscent());
		text.draw(g2, textX, textY);
/*
	if (tlSize != null)
	{
		// Text is draw up from the Ascent and down from the Descent.
		int xCenterPic = (int) ((myPicBoxWidth - tlSize.getBounds()
				.getWidth()) / 2);
		int textX = Xoffset + xCenterPic;
		// This puts the text at the bottom of the box.
		//int textY = (int) (Yoffset + myPicBoxHeight - tLay.getDescent());
		textY = (int) (textY + tlSize.getAscent());
		tlSize.draw(myG2, textX, textY);
	}
	*/

	}

	
	
	// ==========================================================================
	// Private
	// ==========================================================================
	private class PictureMouseListener extends MouseAdapter
	{
		protected PictureMouseListener() { /* just to get rid of synthetic warning...*/ } 
		
		@Override
		public void mouseClicked(MouseEvent event)
		{
			if (event.getClickCount() == 2)
			{
				// Don't revert on click on full screen since that advances the
				// screen.
				//if (!myViewerFrame.isFullScreen()) 
				{
					Single.this.myViewerFrame.gotoFullScreen(false);
				}
			}
			else if (MouseEvent.BUTTON2 == event.getButton()
					|| MouseEvent.BUTTON3 == event.getButton()) // two works in
			// windows
			{
				if (Single.this.myViewerFrame.getScroller().isBeyond())
				{
					return;
				}
				new PicInfoDialog(
						Single.this.myViewerFrame.getScroller().getCurrentFile(),
				      Single.this.myViewerFrame.getFrame(),
				      Single.this,
				      Single.this,
						Single.this.myViewerFrame.getActions().getToolBarButton("file","trash"), //$NON-NLS-1$ //$NON-NLS-2$
						Single.this.myViewerFrame.getActions().getToolBarButton("file","park"), //$NON-NLS-1$ //$NON-NLS-2$
				      false,
				      Single.this.myShownWidth,
				      Single.this.myShownHeight,
				      true, Single.this.myViewerFrame.getScroller() 
				      );
			}
			else
			{
				Single.this.myViewerFrame.getScroller().requestFocus();
			}
			// if ( event.isPopupTrigger() )
		}
	}
	
	/*
	 * ========================================================================
	 * DropTargetListener Implementation
	 * Only used to turn off pictures dragged from sorter.
	 * ========================================================================
	 */

	private void evaluateDrag(DropTargetDragEvent event)
	{
		DataFlavor[] flavors  = event.getCurrentDataFlavors();
		if (flavors == TileConstants.CLS_DATA_FLAVORS)
		{
			event.acceptDrag(DnDConstants.ACTION_MOVE);
		}
		else
		{
			event.rejectDrag();
		}
	}

	public void dragEnter(DropTargetDragEvent event)
	{
		// System.out.println("DragDropList.dragEnter");
		evaluateDrag(event);
	}

	public void dropActionChanged(DropTargetDragEvent event)
	{
		// System.out.println( "DragDropList.dropActionChanged" );
		evaluateDrag(event);
	}

	public void dragExit(DropTargetEvent dte)
	{
		// System.out.println("DragDropList.dragExit");
	}

	public void dragOver(DropTargetDragEvent dtde)
	{
		// System.out.println( "DragDropList.dragOver" );
	}

	public void drop(DropTargetDropEvent event)
	{
		// Fine example of how to do the local drop if you need tighter
		// communications.
		if (event.isLocalTransfer())
		{
			DataFlavor[] flavors  = event.getCurrentDataFlavors();
			if (flavors == TileConstants.CLS_DATA_FLAVORS)
			{
				event.dropComplete(true);
			}
			else
			{
				event.dropComplete(false);
			}
		}
	}
	
}
