/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.classic;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.image.BufferedImage;
import java.io.File;

import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.sorter.TilePanel;

/**
 * JPanel implementation of showing a single image.
 */
public class SingleNoResizeKeyControl extends SingleNoResize
{	
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public SingleNoResizeKeyControl(IFrame mf)
	{
		super(mf);
	}

	// ==========================================================================
	// Overridden methods
	// ==========================================================================	
	//public void paint(Graphics g)
	@Override
	public void showImage(Graphics g)
 	{
   	// Apparently, background color can only be set once per draw without
		// confusing SWING.
		Color backgroundColor = OK_COLOR; // DEFAULT COLOR

		BufferedImage showMe = this.mySaver.getCurrentImage();
		if (showMe == null)
		{
			showMe = this.myViewerFrame.getScroller().getBeyondBounds();
			if (showMe == null)
			{
				super.showNoPicture(g);
				return;
			}
			backgroundColor = EMPTY_COLOR;
		}

		//System.out.println("======================== Paint ============================");
		//System.out.println("Panel w h: " + super.getWidth() + " " + super.getHeight());
		//System.out.println("Raw Picture w h: " + showMe.getWidth() + " " + showMe.getHeight());

		// Remember for pic info dialog
		this.myShownWidth = showMe.getWidth();
		this.myShownHeight = showMe.getHeight();

		// Now the the correct background color has been established,
		// use it.
		super.setBackground(backgroundColor);
		super.paintComponent(g); // First, fill with background color.

		Graphics2D g2 = (Graphics2D) g;

		// Position picture
		int offsetWidth = (int) ((SingleNoResizeKeyControl.this.getWidth() - this.myShownWidth) * this.myOffsetWidthPercentage);
		int offsetHeight = (int) ((SingleNoResizeKeyControl.this.getHeight() - this.myShownHeight) * this.myOffsetHeightPercentage);

		// Draw the new image
		g2.drawImage(showMe, null, offsetWidth, offsetHeight);

		if (this.myViewerFrame.getScroller().getCurrentIOImage() != null)
		{
	      File file = this.myViewerFrame.getScroller().getCurrentIOImage().getFile();
			TilePanel.clsTilePanel.clip(file, g2, super.getWidth(), super.getHeight());
		}

		this.myViewerFrame.statusCurrentPage();
	}

	// ==========================================================================
	// Inner Classes
	// ==========================================================================

	@Override
	KeyListener getKeyListener()
	{
		return new KeyActions();
	}
	
	private class KeyActions implements KeyListener
	{
		boolean	iAmScrolling	= false;

		protected KeyActions() { /* just to get rid of synthetic warning...*/ } 
		
		public void keyTyped(KeyEvent event)
		{
			//System.out.println("keyTyped");
		}

		public void keyPressed(KeyEvent event)
		{
			//System.out.println("keyPressed");
			if (!event.isConsumed())
			{
				switch (event.getKeyCode())
				{
				case KeyEvent.VK_ENTER:
					if ((event.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) == InputEvent.CTRL_DOWN_MASK)
					{
						revert();
					}
					if ((event.getModifiersEx() & InputEvent.ALT_DOWN_MASK) == InputEvent.ALT_DOWN_MASK)
					{
						revert();
					}
					event.consume();
					return;
				}
				
				if (event.getKeyCode() == KeyEvent.VK_ALT)
				{
					this.iAmScrolling = true;
				}
				else if (event.isAltDown())
				{
					processScroll(event);
				}
			}
		}

		public void keyReleased(KeyEvent event)
		{
			//System.out.println("keyReleased");
			if (!event.isConsumed())
			{
				if (event.getKeyCode() == KeyEvent.VK_ALT)
				{
					this.iAmScrolling = false;
				}
				else if (event.isAltDown())
				{
					processNumPad(event);
				}
				else if (!this.iAmScrolling)
				{
					processFuncitionKeys(event);
				}
				if ( this.iAmScrolling )
				{
					event.consume();
				}
			}
		}

		private void processFuncitionKeys(KeyEvent event)
		{
			switch (event.getKeyCode())
			{
			case KeyEvent.VK_F2:
				SingleNoResizeKeyControl.this.myViewerFrame.gotoFullScreen(false);
				event.consume();
				return;
			case KeyEvent.VK_F4:
				SingleNoResizeKeyControl.this.myOptions.setFullSizeTileTo(NoResizeOptions.FULL_SIZE_CENTER_CENTER);
				showImage();
				event.consume();
				return;
			}
		}

		private void processScroll(KeyEvent event)
		{
			switch (event.getKeyCode())
			{
			case KeyEvent.VK_UP:
			{
				event.consume();
				float pixels = (SingleNoResizeKeyControl.this.getHeight() - SingleNoResizeKeyControl.this.myShownHeight)
						* SingleNoResizeKeyControl.this.myOffsetHeightPercentage;
				if (pixels > -SingleNoResizeKeyControl.this.myShownHeight)
				{
					if (SingleNoResizeKeyControl.this.myShownHeight < SingleNoResizeKeyControl.this.getHeight())
					{
						SingleNoResizeKeyControl.this.myOffsetHeightPercentage = SingleNoResizeKeyControl.this.myOffsetHeightPercentage - 0.1f;
					}
					else
					{
						SingleNoResizeKeyControl.this.myOffsetHeightPercentage = SingleNoResizeKeyControl.this.myOffsetHeightPercentage + 0.1f;
					}
				}
				repaint();
				return;
			}
			case KeyEvent.VK_DOWN:
			{
				event.consume();
				float pixels = (SingleNoResizeKeyControl.this.getHeight() - SingleNoResizeKeyControl.this.myShownHeight)
						* SingleNoResizeKeyControl.this.myOffsetHeightPercentage;
				if (pixels < SingleNoResizeKeyControl.this.getHeight())
				{
					if (SingleNoResizeKeyControl.this.myShownHeight < SingleNoResizeKeyControl.this.getHeight())
					{
						SingleNoResizeKeyControl.this.myOffsetHeightPercentage = SingleNoResizeKeyControl.this.myOffsetHeightPercentage + 0.1f;
					}
					else
					{
						SingleNoResizeKeyControl.this.myOffsetHeightPercentage = SingleNoResizeKeyControl.this.myOffsetHeightPercentage - 0.1f;
					}
				}
				repaint();
				return;
			}
			case KeyEvent.VK_LEFT:
			{
				event.consume();
				float pixels = (SingleNoResizeKeyControl.this.getWidth() - SingleNoResizeKeyControl.this.myShownWidth)
						* SingleNoResizeKeyControl.this.myOffsetWidthPercentage;
				if (pixels > -SingleNoResizeKeyControl.this.myShownWidth)
				{
					if (SingleNoResizeKeyControl.this.myShownWidth < SingleNoResizeKeyControl.this.getWidth())
					{
						SingleNoResizeKeyControl.this.myOffsetWidthPercentage = SingleNoResizeKeyControl.this.myOffsetWidthPercentage - 0.1f;
					}
					else
					{
						SingleNoResizeKeyControl.this.myOffsetWidthPercentage = SingleNoResizeKeyControl.this.myOffsetWidthPercentage + 0.1f;
					}
				}
				repaint();
				return;
			}
			case KeyEvent.VK_RIGHT:
			{
				event.consume();
				float pixels = (SingleNoResizeKeyControl.this.getWidth() - SingleNoResizeKeyControl.this.myShownWidth)
						* SingleNoResizeKeyControl.this.myOffsetWidthPercentage;
				if (pixels < SingleNoResizeKeyControl.this.getWidth())
				{
					if (SingleNoResizeKeyControl.this.myShownWidth < SingleNoResizeKeyControl.this.getWidth())
					{
						SingleNoResizeKeyControl.this.myOffsetWidthPercentage = SingleNoResizeKeyControl.this.myOffsetWidthPercentage + 0.1f;
					}
					else
					{
						SingleNoResizeKeyControl.this.myOffsetWidthPercentage = SingleNoResizeKeyControl.this.myOffsetWidthPercentage - 0.1f;
					}
				}
				repaint();
				return;
			}
			}
		}
	}
}
