/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.classic;

import java.awt.Cursor;
import java.awt.datatransfer.Clipboard;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JOptionPane;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.ISave;
import org.gerhardb.jibs.viewer.contact.ContactSheetDisplay;
import org.gerhardb.jibs.viewer.frame.SaverActions;
import org.gerhardb.lib.image.IOImage;
import org.gerhardb.lib.image.ImageChangeUtil;
import org.gerhardb.lib.io.FileSaveAs;
import org.gerhardb.lib.scroller.Scroller;

import com.saic.isd.printscreen.ImageSelection;

/**
 * Saves the currently selected file.
 */
public class SingleSaver implements ISave
{
	BufferedImage myOriginalImage;
	BufferedImage myAdjustedImage;
	IOImage myIOImage;	
	SaverActions mySaverActions;
	JFrame myJFrame;
	Single myShow;
	Scroller myScroller;

	private int myShrinkGrow = 10;
	private int mySizedShrinkGrow = 0;
	private int myRotation = 0;

	// =========================================================================
	// Constructor
	// =========================================================================
	SingleSaver(SaverActions sa, JFrame f, Single show, Scroller scroller)
	{
		this.mySaverActions = sa;
		this.myJFrame = f;
		this.myShow = show;
		this.myScroller = scroller;
	}
	
	// ==========================================================================
	// ISave Interface
	// ==========================================================================
	public void rotateRight()
	{
		this.myRotation = this.myRotation + 90;
		if (this.myRotation == 360)
		{
			this.myRotation = 0;
		}
		this.mySaverActions.enableSave(true);
		adjustImage(this.myRotation, this.mySizedShrinkGrow);
	}

	public void rotateLeft()
	{
		this.myRotation = this.myRotation - 90;
		if (this.myRotation == -360)
		{
			this.myRotation = 0;
		}
		this.mySaverActions.enableSave(true);
		adjustImage(this.myRotation, this.mySizedShrinkGrow);
	}

	public void shrink()
	{
		if (this.mySizedShrinkGrow <= -90)
		{
			return;
		}
		this.mySizedShrinkGrow = this.mySizedShrinkGrow - this.myShrinkGrow;
		this.mySaverActions.enableSave(true);
		adjustImage(this.myRotation, this.mySizedShrinkGrow);
	}

	public void grow()
	{
		this.mySizedShrinkGrow = this.mySizedShrinkGrow + this.myShrinkGrow;
		this.mySaverActions.enableSave(true);
		adjustImage(this.myRotation, this.mySizedShrinkGrow);
	}
	
	public void resetToolbar()
	{
		resetToolbar(true);
	}
	
	public void save(boolean saveAs)
	{
		if (this.mySaverActions.cancelSaveForExif(this.myScroller.getCurrentFile()))
		{
			return;
		}

		this.myJFrame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try
		{
			// float quality = 1.0f;
			if (saveAs)
			{
				JFileChooser chooser = new JFileChooser();
				chooser.setCurrentDirectory(getCurrentFile());
				chooser.setDialogTitle(Jibs.getString("SingleSaver.1")); //$NON-NLS-1$
				chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
				int returnVal = chooser.showSaveDialog(this.myScroller
						.getFocusComponent());
				if (returnVal == JFileChooser.APPROVE_OPTION)
				{
					try
					{
						File oldFile = getCurrentFile();
						File saveAsFile = chooser.getSelectedFile();
						if (FileSaveAs.PossibleDuplicateNameOK(saveAsFile,
								this.myJFrame))
						{
							File oldDir = oldFile.getParentFile();
							String saveAsPath = saveAsFile.getAbsolutePath();
							this.myScroller.getCurrentIOImage().save(
									this.myAdjustedImage, saveAsPath);

							// Get File in case name changed with addition of ending.
							File newFile = getCurrentFile();
							File newDir = newFile.getParentFile();
							if (newDir.equals(oldDir))
							{
								this.myScroller.reloadScroller();
								this.myScroller.jumpTo(newFile);
							}
						}
					}
					catch (Exception ex)
					{
						// So remote as to ignore
						ex.printStackTrace();
					}
				}
			}
			else
			{
				this.myScroller.getCurrentIOImage().save(this.myAdjustedImage);
			}
			// Adjust the user interface
			resetToolbar(true);
			this.myScroller.requestFocus();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			JOptionPane.showMessageDialog(
					this.myScroller.getFocusComponent(),
					Jibs.getString("SingleSaver.2") + Jibs.getString("colon") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
							+ ex.getMessage(),
					Jibs.getString("SingleSaver.3"), JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
		}
		this.myJFrame.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}
	
	public void copy()
	{
		IOImage image = this.myScroller.getCurrentIOImage();
      ImageSelection clip = new ImageSelection( image );
      Clipboard clipboard = this.myJFrame.getToolkit().getSystemClipboard();
      clipboard.setContents( clip, clip );		
	}
	
	public void pageableAction(int action)
	{
		if (this.myScroller.getCurrentIOImage() == null)
		{
			return;
		}		
		File[] showThisFile = new File[]{this.myScroller.getCurrentIOImage().getFile()};		
		new ContactSheetDisplay(this.myJFrame, showThisFile, true, ContactSheetDisplay.SHOW_DIALOG, this.myScroller);
	}
	
	/*
	public void preview()
	{
		if (this.myScroller.getCurrentIOImage() == null)
		{
			return;
		}
		
		this.myJFrame.setCursor(Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ));
		
		BufferedImage image = getCurrentImage();
		String description = 
			this.myScroller.getCurrentIOImage().getFile().toString();
		System.out.println("SingleSaver Preview ====> " + description);
      PageableImage pi = new PageableImage(description, image, PrinterJob.getPrinterJob());
   	new PageablePreviewer(pi, null, null, false);   
		
		this.myJFrame.setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
	}
	
	public void print(boolean highResolution)
	{
		if (this.myScroller.getCurrentIOImage() == null)
		{
			return;
		}

		this.myJFrame.setCursor(Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ));

		BufferedImage image = getCurrentImage();
		String description = 
			this.myScroller.getCurrentIOImage().getFile().toString();
		if (highResolution)
		{
			PrintUtils.printPicHighResolution(description, image, this.myJFrame);
		}
		else
		{
			PrintUtils.printPicLowResolution(description, image, this.myJFrame);			
		}
		
		this.myJFrame.setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
	}
	*/
	
	// ==========================================================================
	// Private Methods
	// ==========================================================================
	/**
	 * Make changes to the image.
	 * @param rotation One of: +/- 0, 90, 180, 270, 360
	 * @param growShrink Zero to leave alone.  Else percentage +/-.
	 * @return
	 */
	private void adjustImage(int rotation, int shrinkGrow)
	{
		if (this.myScroller.getCurrentIOImage() == null)
		{
			return;
		}
		this.myAdjustedImage = this.myOriginalImage;
		switch (rotation)
		{
			case 0:
				break;
			case 90:
				this.myAdjustedImage = ImageChangeUtil.rotateRight(this.myAdjustedImage);
				break;
			case -90:
				this.myAdjustedImage = ImageChangeUtil.rotateLeft(this.myAdjustedImage);
				break;
			case 180:
			case -180:
				this.myAdjustedImage = ImageChangeUtil.flip(this.myAdjustedImage);
				break;
			case 270:
			{
				this.myAdjustedImage = ImageChangeUtil.rotateLeft(this.myAdjustedImage);
				break;
			}
			case -270:
			{
				this.myAdjustedImage = ImageChangeUtil.rotateRight(this.myAdjustedImage);
				break;
			}
		}
		if (shrinkGrow != 0)
		{
			float scale = (shrinkGrow / 100.0F) + 1.0F;
			if (scale <= 0.0F)
			{
				scale = 0.1F;
			}
			this.myAdjustedImage = ImageChangeUtil.scale(this.myAdjustedImage, scale);
		}
		return;
	}
	
	private void resetToolbar(boolean resetRotationShrinkGrow)
	{
		if (resetRotationShrinkGrow)
		{
			this.mySizedShrinkGrow = 0;
			this.myRotation = 0;
		}
		this.mySaverActions.enableSave(false);	
		
		boolean showing = true;		
		// Valid range is 0 to getSize() - 1
		if (this.myScroller.getValueZeroBased() < 0)
		{	
			showing = false;
		}
		else if (this.myScroller.getValueZeroBased() >= this.myScroller
				.getSize())
		{	
			showing = false;
		}

		this.mySaverActions.enableSaveAs(showing);
		this.mySaverActions.enableRotation(showing);
		this.mySaverActions.enableShrinkGrow(showing);
		if (showing && this.myShow.iCanResize())
		{
			this.mySaverActions.enableShrinkGrow(true);
		}
		else
		{
			this.mySaverActions.enableShrinkGrow(false);			
		}		
	}
	
	private File getCurrentFile()
	{
		IOImage image = this.myScroller.getCurrentIOImage();
		if (image == null)
		{
			return null;
		}
		return image.getFile();
	}

	protected BufferedImage getCurrentImage()
	{
		int index = this.myScroller.getValueZeroBased();
		if (index < 0 || index >= this.myScroller.getSize())
		{
			this.myOriginalImage = null;
			return this.myOriginalImage;
		}

		if (this.myScroller.getCurrentIOImage() == this.myIOImage)
		{
			// System.out.println("getCurrentImage(CACHED): " + index);
			return this.myAdjustedImage;
		}

		// System.out.println("getCurrentImage(LOAD): " + index);

		this.myIOImage = null;
		this.myOriginalImage = null;
		this.myAdjustedImage = null;
		resetToolbar(false);

		if (index > -1)
		{
			IOImage current = null;
			try
			{
				current = this.myScroller.getCurrentIOImage();
				if (current != null)
				{
					this.myIOImage = current;
					this.myOriginalImage = current.getImage();
					this.myAdjustedImage = this.myOriginalImage;
				}
				else
				{
					System.out.println("myScroller.getCurrentIOImage() returned null "); //$NON-NLS-1$					
				}
			}
			catch (Exception ex)
			{
				System.out.println("Scroller.getCurrentImage FAILED"); //$NON-NLS-1$
				if (current != null)
				{
					if (current.getFile().exists())
					{
						System.out.println("Problem loading image for: " + current.getFile().getAbsolutePath()); //$NON-NLS-1$
					}
					else
					{
						System.out.println("Missing File: " + current.getFile().getAbsolutePath()); //$NON-NLS-1$						
					}
				}
				System.out.println(ex.getMessage());
				ex.printStackTrace();
			}
		}

		return this.myAdjustedImage;
	}
}
