/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.comic;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.EventQueue;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Rectangle;
import java.awt.event.InputEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.*;
import java.util.prefs.Preferences;

import javax.swing.*;

import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.IRevert;
import org.gerhardb.jibs.viewer.PicInfoDialog;
import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.scroller.ScrollerListener;
import org.gerhardb.lib.scroller.ScrollerSlider;


/**
 * Full Screen Display!
 * ALWAYS SHRINKS!  Tried to do otherwise, but failed.
 * Don't think you EVER need full screen with scroll bars.
 */
public class ComicScreen implements ScrollerListener, IRevert
{
	static final Preferences clsPrefs = Preferences.userRoot().node(
	"/org/gerhardb/jibs/viewer/shows/comic"); //$NON-NLS-1$
	static final String DIVIDER_LOCATION = "DIVIDER_LOCATION";
	static final String DISPLAY_SLIDER = "DISPLAY_SLIDER";
	static final Color BACKGROUND_COLOR = Color.black;
	
	// --------------- Things passed into me ---------------------------
	IFrame myParentFrame;
	private boolean iShowFullScreen;

	// --------------- Things I create ---------------------------
	JFrame mySurfaceFrame;

	// This panel becomes the JFrame Surface
	private JPanel mySurfacePanel = new JPanel(new BorderLayout());
	// This shows the magnified image on the right side
	MagnifiedPanel myMagnifiedPanel = new MagnifiedPanel();
	// This shows the whole thumbnail image on the left side
	private ThumbnailImage myThumbnailImage = new ThumbnailImage();
	// This panel show the controls at the bottom of the screen
	private JPanel myBottomPanel = new JPanel(new BorderLayout());

	private GraphicsEnvironment myGE = GraphicsEnvironment
			.getLocalGraphicsEnvironment();
	GraphicsDevice myGD = this.myGE.getDefaultScreenDevice();
	JSplitPane mySplitPane;
	ComicMouseAdapter myComicMouseAdapter = new ComicMouseAdapter();

	//==========================================================================
	//                         Constructor
	//==========================================================================
	public ComicScreen(IFrame mf)
	{
		this(mf, true);
	}
	
	public ComicScreen(IFrame mf, boolean showFullScreen)
	{
		this.myParentFrame = mf;
		this.iShowFullScreen = showFullScreen;
		this.myParentFrame.setWaitCursor(true);
		this.myParentFrame.getScroller().setEndMessages(true);

		// Must do before creating frame to keep the keys working.
		this.mySplitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,
				this.myThumbnailImage, this.myMagnifiedPanel);

		if (this.iShowFullScreen)
		{
			this.mySurfaceFrame = new JFrame(this.myGD.getDefaultConfiguration());
			this.mySurfaceFrame.setUndecorated(true);
		}
		else
		{
			this.mySurfaceFrame = new JFrame("JIBS Comic Viewer");
			this.mySurfaceFrame.setUndecorated(false);			
		}
		this.mySurfaceFrame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		
		// Set slider for initial show to user
		this.myBottomPanel.setVisible(clsPrefs.getBoolean(DISPLAY_SLIDER, true));
		
		//myMagnifiedImage.setFocusable(true);
		//myThumbnailImage.setFocusable(true);		
		//mySurfaceFrame.setFocusableWindowState(true);
		//mySurfaceFrame.setFocusable(true);

		//JToolBar bottomRight = new JToolBar();
		//bottomRight.setFloatable(false);
		//bottomRight.add(new JLabel("temp"));

		ScrollerSlider slider = new ScrollerSlider(SwingConstants.HORIZONTAL, this.myParentFrame.getScroller());
		this.myBottomPanel.add(slider, BorderLayout.CENTER);
		//myBottomPanel.add(bottomRight, BorderLayout.EAST);

		// Set the surface
		this.mySurfacePanel.add(this.mySplitPane, BorderLayout.CENTER);
		this.mySurfacePanel.add(this.myBottomPanel, BorderLayout.SOUTH);
		this.mySurfaceFrame.setContentPane(this.mySurfacePanel);

		this.mySurfaceFrame.addKeyListener(new CompicKeyAdapater());
		this.mySurfaceFrame.addKeyListener(this.myParentFrame.getScroller().getScrollerKeyListener());
		this.myThumbnailImage.addMouseListener(this.myComicMouseAdapter);
		this.myMagnifiedPanel.myMagnifiedImage.addMouseListener(this.myComicMouseAdapter);
		
		// Actually show the current picture
		scrollerChanged(null);
		this.myParentFrame.getScroller().addScrollerListener(this);
		this.mySplitPane.setDividerLocation(clsPrefs.getInt(DIVIDER_LOCATION, 200));
		
		try
		{
			if (this.iShowFullScreen)
			{
				this.myGD.setFullScreenWindow(this.mySurfaceFrame);
				this.myParentFrame.getScroller().setAutoFocus(this.mySurfaceFrame);
			}
			else
			{
				this.mySurfaceFrame.setBounds(new Rectangle(10, 10, 1200,600));
				this.mySurfaceFrame.setVisible(true);
				this.mySurfaceFrame.addWindowListener(new WindowAdapter()
				{
					@Override
					public void windowClosing(WindowEvent evt)
					{
						System.out.println("Comic setting prefs");
						try
						{
							clsPrefs.putInt(DIVIDER_LOCATION, ComicScreen.this.mySplitPane.getDividerLocation());
							clsPrefs.flush();
						}
						catch(Exception ex)
				      {
				      	ex.printStackTrace();
				      }
					}
				});
			}
			this.mySurfaceFrame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			
			this.myMagnifiedPanel.myMagnifiedImage.requestFocus(true);
			this.mySurfaceFrame.requestFocus();
			
			this.mySurfaceFrame.setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
			this.myParentFrame.setWaitCursor(false);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}

	public JFrame getFrame()
	{
		return this.mySurfaceFrame;
	}
	
	//==========================================================================
	//                         ScrollerListener Implemenattion
	//==========================================================================
	public void scrollerChanged(ScrollerChangeEvent ce)
	{
		if (!this.myParentFrame.getScroller().getValueIsAdjusting())
		{
			BufferedImage image = null;
			try
			{
				image = this.myParentFrame.getScroller().getCurrentImage();
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}

			if (image == null)
			{
				// If there is no image to show, try to show the past the end
				// picture.  In any event, return since there is no picture.
				image = this.myParentFrame.getScroller().getBeyondBounds();
				if (image != null)
				{
					this.myMagnifiedPanel.myMagnifiedImage.setImage(image);
					this.myThumbnailImage.setImage(image);
				}
				return;
			}

			//System.out.println("Image Adjusted: w: " + image.getWidth() + " h: " + image.getHeight() );
			this.myMagnifiedPanel.myMagnifiedImage.setImage(image);
			this.myThumbnailImage.setImage(image);
		}
	}

	//==========================================================================
	//                        IRevert
	//==========================================================================
	public void revert()
	{
		if (!this.iShowFullScreen)
		{
			return;
		}
		this.myParentFrame.getScroller().stopSlideShow();
		this.myParentFrame.getScroller().removeScrollerListener(ComicScreen.this);
		
		System.out.println("Comic setting prefs");
		try
		{
			clsPrefs.putInt(DIVIDER_LOCATION, this.mySplitPane.getDividerLocation());
			clsPrefs.flush();
		}
		catch(Exception ex)
      {
      	ex.printStackTrace();
      }

		// Clear screen, set to gray, and start wait for slow switch in Linux.
		this.mySurfaceFrame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		// Need to set color before changing image.
		// (In case we add color changes to the background some other time.)
		this.myMagnifiedPanel.myMagnifiedImage.setBackground(ComicScreen.BACKGROUND_COLOR);
		this.myMagnifiedPanel.myMagnifiedImage.setImage(null);
		this.myThumbnailImage.setImage(null);

		// If you run this without putting on thread,
		// the myGD.setFullScreenWindow executes before clear starts.
		Runnable runSwitch = new Runnable()
		{
			public void run()
			{
				//try{Thread.sleep(500);}catch(Exception ex){}
				ComicScreen.this.myGD.setFullScreenWindow(null);
				ComicScreen.this.mySurfaceFrame.setVisible(false);
				ComicScreen.this.mySurfaceFrame.dispose();
				ComicScreen.this.myParentFrame.setWaitCursor(true);
				ComicScreen.this.myParentFrame.gotoRegularScreen();
				ComicScreen.this.myParentFrame.setWaitCursor(false);
			}
		};
		EventQueue.invokeLater(runSwitch);
	}

	public boolean isFullScreen()
	{
		return this.iShowFullScreen;
	}

	//==========================================================================
	//                        Methods
	//==========================================================================

	void toggleBottom()
	{
		if (this.myBottomPanel.isShowing())
		{
			this.myBottomPanel.setVisible(false);
			this.mySurfacePanel.revalidate();
			try
			{
				clsPrefs.putBoolean(DISPLAY_SLIDER, false);
				clsPrefs.flush();
			}
			catch(Exception ex)
	      {
	      	ex.printStackTrace();
	      }
		}
		else
		{
			//Robot robot = new Robot();
			//robot.mouseMove(x, y);
			this.myBottomPanel.setVisible(true);
			this.mySurfacePanel.revalidate();
			try
			{
				clsPrefs.putBoolean(DISPLAY_SLIDER, true);
				clsPrefs.flush();
			}
			catch(Exception ex)
	      {
	      	ex.printStackTrace();
	      }
		}
	}
	
	private class CompicKeyAdapater extends KeyAdapter
	{

		protected CompicKeyAdapater() { /* just to get rid of synthetic warning...*/ } 
		
		@Override
		public void keyPressed(KeyEvent event)
		{
			//System.out.println("ComicScreen keyPressed: event.getKeyCode(): " + event.getKeyCode() + "  KEY: " + KeyEvent.getKeyText(event.getKeyCode()));
			switch (event.getKeyCode())
			{
				case KeyEvent.VK_ENTER:
					// Alt did work here, but not on full screen...
					if ((event.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) == InputEvent.CTRL_DOWN_MASK)
					{
						revert();
					}
					if ((event.getModifiersEx() & InputEvent.ALT_DOWN_MASK) == InputEvent.ALT_DOWN_MASK)
					{
						revert();
					}
					event.consume();
					return;
				case KeyEvent.VK_UP:
					ComicScreen.this.myMagnifiedPanel.myMagnifiedImage.up();
					event.consume();
					break;
				case KeyEvent.VK_DOWN:
					ComicScreen.this.myMagnifiedPanel.myMagnifiedImage.down();
					event.consume();
					break;
			}
		}

		@Override
		public void keyReleased(KeyEvent event)
		{
			//System.out.println("ComicScreen keyReleased: event.getKeyCode(): " + event.getKeyCode() + "  KEY: " + KeyEvent.getKeyText(event.getKeyCode()));
			switch (event.getKeyCode())
			{
				case KeyEvent.VK_F1:
					revert();
					ComicScreen.this.myParentFrame.showHelp();
					event.consume();
					return;
				case KeyEvent.VK_ESCAPE:
				case KeyEvent.VK_F2:
					revert();
					event.consume();
					return;
				case KeyEvent.VK_F3:
					toggleBottom();
					event.consume();
					return;
				case KeyEvent.VK_UP:
				case KeyEvent.VK_DOWN:
					event.consume();
					break;
				case KeyEvent.VK_PAGE_UP:
					ComicScreen.this.myParentFrame.getScroller().up();
					event.consume();
					break;
				case KeyEvent.VK_PAGE_DOWN:
					ComicScreen.this.myParentFrame.getScroller().down();
					event.consume();
					break;
				case KeyEvent.VK_HOME:
					ComicScreen.this.myMagnifiedPanel.myMagnifiedImage.allUp();
					event.consume();
					break;
				case KeyEvent.VK_END:
					ComicScreen.this.myMagnifiedPanel.myMagnifiedImage.allDown();
					event.consume();
					break;
			}
		}
	}	
	
	private class ComicMouseAdapter extends MouseAdapter
	{

		protected ComicMouseAdapter() { /* just to get rid of synthetic warning...*/ } 
		
		@Override
		public void mousePressed(MouseEvent event)
		{
			System.out.println("COMIC MOUSE PRESSED");
			if (event.isPopupTrigger()
					|| event.getButton() == MouseEvent.BUTTON2
					|| event.getButton() == MouseEvent.BUTTON3)
			{
				if (ComicScreen.this.myParentFrame.getScroller().isBeyond())
				{
					return;
				}
				int width = -1;
				int height = -1;
				try
				{
					BufferedImage image = ComicScreen.this.myParentFrame.getScroller()
							.getCurrentImage();
					width = image.getWidth();
					height = image.getHeight();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
				new PicInfoDialog(
						ComicScreen.this.myParentFrame.getScroller().getCurrentFile(),
						ComicScreen.this.mySurfaceFrame, 
						ComicScreen.this.myMagnifiedPanel.myMagnifiedImage, 
						ComicScreen.this, 
						ComicScreen.this.myParentFrame.getActions().getToolBarButton("file", "trash"), //$NON-NLS-1$ //$NON-NLS-2$
						ComicScreen.this.myParentFrame.getActions().getToolBarButton("file", "park"), //$NON-NLS-1$ //$NON-NLS-2$
						true, width, height, true, ComicScreen.this.myParentFrame.getScroller());
			}
			else
			{
				ComicScreen.this.myParentFrame.getScroller().down();
			}
		}
	}
}
