/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.thumbs;

import java.awt.image.BufferedImage;
import java.io.File;

import org.gerhardb.lib.image.IOImage;
import org.gerhardb.lib.image.ImageChangeUtil;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.util.*;


/**
 * Caches Thumbnail images in their full resolution.
 * Use JUST for the images currently shown on screen for grow/shrink situations.
 */
class ThumbnailCache extends Thread
{
	/**
	 * Used to kill the thread.
	 */
	private boolean iKeepRunning = true;
	
	boolean iAmWaitingForMyThreadToStart = true;
	private boolean iPauseRunning = false;
	private int mySetSize = 1;	
	private SoftKeyHashMap myMap = new SoftKeyHashMap();
	private int myStartLoadingCacheAtIndex = 0;
	private int mySmallWidth;
	private int mySmallHeight;
	private int myMaxPreload = 100;
	private Scroller myScroller;
	private ThumbnailOptions myOptions;
	private Thumbnails myParent;
	private String myMapSync = "Sync for Map";
	
	/**
	 * Creates the cache and starts it running.
	 */
	ThumbnailCache(int rows, int cols, Scroller scroller, ThumbnailOptions options, Thumbnails parent)
	{
		//System.out.println("NEW THUMBNAIL CACHE");
      this.myScroller = scroller;
      this.myParent = parent;
		this.mySmallWidth = org.gerhardb.jibs.optimizer.OptimizerPreferences.getFullScreenWidth() / cols;
		this.mySmallHeight = org.gerhardb.jibs.optimizer.OptimizerPreferences.getFullScreenHeight() / rows;
		this.mySetSize = rows * cols;		
      this.myOptions = options;
      setMaxPreload(this.myOptions.getPreloadCount());
		super.setName("ThumbnailCache");
		super.setPriority(Thread.MIN_PRIORITY);
		super.start();
		// Thread starts whenever it is ready to start.
		// This causes random problems on first screen where other threads go ahead.
		// This maker sure thread has started and that its is nicely asleep
		// waiting to show some pictures from the empty cache.
		while(this.iAmWaitingForMyThreadToStart)
		{
			try{Thread.sleep(100);}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	// ==========================================================================
	// Partially Synchronized
	// ==========================================================================
	/**
	 * This does a fast return from the cache.
	 * It does not get image from disk if it is not in the cache.
	 * Second look should get image from disk if not in the cache.
	 * @param index
	 * @return
	 */
	BufferedImage getPreviouslyCachedImage(int index)
	{
		File file = this.myScroller.getFile(index);	
		if (file == null)
		{
			return null;
		}
		synchronized(this.myMapSync)
		{
			return (BufferedImage)this.myMap.get(file.getName());
		}
	}

	void remove(File file)
	{
		// System.out.println("ThumbnailCache.remove()");
		String fileName = file.getName();
		synchronized(this.myMapSync)
		{
			this.myMap.remove(fileName);
		}
	}
	
	void resetAndReshow()
	{
	   //System.out.println("################################ cache.resetAndReshow() #####################################"); //$NON-NLS-1$
		new Throwable("RELOADING THUMBNAIL CACHE").printStackTrace();		
		pause();
		synchronized(this)
		{
			synchronized(this.myMapSync)
			{
				this.myMap = new SoftKeyHashMap();
				this.myParent.myRotations.resetRotations(this.myScroller.getSize());
			}
		}
		Runtime.getRuntime().gc();	
		this.myParent.showPageFromScroller();
	   this.myParent.myViewerFrame.getFileList().clearSelection();
	}
	
	BufferedImage grabImage(File file)
	{
		//System.out.println("Grabbing Image: " + file.getName());
		BufferedImage smallSize = null;
		try
		{
			IOImage io = ImageFactory.getImageFactory().makeImage(file);
			BufferedImage fullSize = io.getImage();
			smallSize = ImageChangeUtil.fitAspectDown(fullSize,
					this.mySmallWidth, this.mySmallHeight); 
			synchronized(this.myMapSync)
			{
				this.myMap.put(file.getName(), smallSize);
			}
			//System.out.println("Grabbed: " + file.getName());
		}
		catch (Exception ex)
		{
			// System.out.println(ex.getMessage());
			ex.printStackTrace();
		}
		// Avoids null pointer check for myThumbnails[]
		return smallSize;
	}
		
	private void kickThread()
	{
		//new Exception("kickThread()").printStackTrace();
		//System.out.println("Kicking Thread!");	
		try
		{
			synchronized(this)
			{
				this.iPauseRunning = false;
				super.notify();
			}
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		//System.out.println("Kicked Thread!");	
	}
	
	@Override
	public void run()
	{
		synchronized(this)
		{
			this.iAmWaitingForMyThreadToStart = false;
			while (this.iKeepRunning)
			{
				try
				{
					//System.out.println("ZZZZZZZZZZZZ  THREAD GOING TO SLEEP.  Size of cache: " + this.myMap.size());
					super.wait();                                    // kickThread();
					//System.out.println(">>>>>>>>>> CACHE SHOW AND LOAD THREAD JUST WOKE UP <<<<<<<<<<<  " + Thread.currentThread().getId() + ": " + Thread.currentThread());
				}
				catch (InterruptedException e)
				{
					e.printStackTrace();
					continue;
				}
				if (keepLoading())
				{
					this.myParent.showCurrentPictures();
				}
				if (keepLoading())
				{
					loadAllImages();	
				}			
				// System.out.println(">>>>>>>>>> CACHE LOOPING BACK  ");
			}
			// System.out.println("THUMNAIL CACHE THREAD LOOP TERMINATED");
		}
	}
	
	// ==========================================================================
	// NOT Synchronized
	// ==========================================================================
	void pause()
	{
		this.iPauseRunning = true;
	}

	/**
	 * Pause thread & then start work at top of loop again.
	 */
	void showImages(int startingIndexToLoad)
	{
		//System.out.println("ThumbnailCache.showImages: " + startingIndexToLoad + " thread: " + Thread.currentThread().getId() + ": " + Thread.currentThread());
		//new Throwable("RELOADING THUMBNAIL CACHE").printStackTrace();		
		this.myStartLoadingCacheAtIndex = startingIndexToLoad;
		if (this.iPauseRunning)
		{
			kickThread();
		}
	}
	
	/**
	 * Kills thread permanently and stops work.
	 */
	void stopCache()
	{
		// System.out.println("Thumbnail Cahce exiting");
		this.iPauseRunning = true;
		this.iKeepRunning = false;
		kickThread();
	}
	
	boolean keepLoading()
	{
		if (this.iPauseRunning)
		{
			//System.out.println("********************** DON'T KEEP LOADING **********************************");
			return false;
		}
		return this.iKeepRunning;
	}
	
	void setMaxPreload(int preload)
	{
		if (preload > 20)
		{
			this.myMaxPreload = preload;
		}
		// Update will occur when user next initiates an change of page.
		// Otherwise, this update could interfere with change of page updating.
	}
		
	private void loadAllImages()
	{
		int startAt = this.myStartLoadingCacheAtIndex;
		int firstTwoSets = startAt + (this.mySetSize * 2);
		// Load next two pages
		if (keepLoading())
		{
			loadSet(startAt, firstTwoSets);
		}
		// Load prior page
		if (keepLoading())
		{
			loadSet(startAt - this.mySetSize, startAt);
		}
		// Load rest
		int restStartAt = startAt + firstTwoSets;
		if (keepLoading())
		{
			loadSet(restStartAt, restStartAt + this.myMaxPreload);
		}
	} 

	private void loadSet(int startAt, int stopAt)
	{
		if (startAt < 0)
		{
			startAt = 0;
		}
		
		if (stopAt > this.myScroller.getSize())
		{
			stopAt = this.myScroller.getSize();
		}
		
		// Fail safe with myThumbnails.length if another thread starts...
		// Problem once in dec 2010
		while (startAt < stopAt && keepLoading())
		{
			File file = this.myScroller.getFile(startAt);
			if(file != null && this.myMap.get(file.getName()) == null )
			{
				// System.out.println("BACKGROUND THREAD LOADING (" + startAt + " to " + stopAt + "): " + file.getName());
				grabImage(file);
				Thread.yield();
			}
			startAt++;
		}
	}
} 