/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.thumbs;

import java.awt.FlowLayout;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.SwingConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.IShow;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.SwingUtils;

/**
 * ThumbnailOptionsDialog
 */
public class ThumbnailOptionsDialog extends JDialog
{
   JSlider rowSlider = new JSlider(SwingConstants.HORIZONTAL, 1, 8, 4);
   JSlider colSlider = new JSlider(SwingConstants.HORIZONTAL, 1, 6, 3);
   JSlider preloadSlider  = new JSlider(SwingConstants.HORIZONTAL, 20, 2020, 820);
   JSlider picSlider = new JSlider(SwingConstants.HORIZONTAL, 1, 6, 3);
	JLabel myPicsPerPageLabel = new JLabel();
	JLabel mySelectedPicLabel = new JLabel();
	ThumbnailOptions myOptions;
	IFrame myFrame;
	
   /**
    * @throws SecurityException
    */
   public ThumbnailOptionsDialog(IFrame frame, ThumbnailOptions options) 
   {
   	super(frame.getFrame(), Jibs.getString("ThumbnailOptionsDialog.0"), false); //$NON-NLS-1$
   	this.myOptions = options;
   	this.myFrame = frame;
   	initLayout();
    }


	// ===========================================================================
	// Menu item to display the dialog
	// ===========================================================================
	/**
	 * Makes a new set for each UI its used on.
	 */
	public static JMenuItem makeThumbnailPreferenceMenu(final IFrame frame, final ThumbnailOptions options)
	{
		class ThumbnailPreferencesAction extends AbstractAction
		{
			ThumbnailPreferencesAction()
			{
				super(Jibs.getString("ThumbnailOptionsDialog.1")); //$NON-NLS-1$
			}

			public void actionPerformed(ActionEvent event)
			{
				new ThumbnailOptionsDialog(frame, options);
			}
		}	
		
		return new JMenuItem(new ThumbnailPreferencesAction());
	}
	
	// ===========================================================================
	// The dialog
	// ===========================================================================
	private void initLayout()
   {
      this.rowSlider.addChangeListener(
         new ChangeListener()
      {
         public void stateChanged(ChangeEvent e)
         {
            update();
         }
      });

      this.colSlider.addChangeListener(
         new ChangeListener()
      {
         public void stateChanged(ChangeEvent e)
         {
            update();
         }
      });

      this.preloadSlider.addChangeListener(
            new ChangeListener()
         {
            public void stateChanged(ChangeEvent e)
            {
               update();
            }
         });

     this.picSlider.addChangeListener(
            new ChangeListener()
         {
            public void stateChanged(ChangeEvent e)
            {
               update();
            }
         });
      
      JButton saveBtn = new JButton(Jibs.getString("save")); //$NON-NLS-1$
      saveBtn.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
            saveOptions();
            ThumbnailOptionsDialog.this.dispose();
         }
      });

      JButton cancelBtn = new JButton(Jibs.getString("cancel")); //$NON-NLS-1$
      cancelBtn.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent ae)
         {
            ThumbnailOptionsDialog.this.dispose();
         }
      });

      this.rowSlider.setSnapToTicks(true);
      this.rowSlider.setPaintLabels(true);
      this.rowSlider.setPaintTicks(true);
      this.rowSlider.setMajorTickSpacing(1);

      this.colSlider.setSnapToTicks(true);
      this.colSlider.setPaintLabels(true);
      this.colSlider.setPaintTicks(true);
      this.colSlider.setMajorTickSpacing(1);

      this.preloadSlider.setSnapToTicks(true);
      this.preloadSlider.setPaintLabels(true);
      this.preloadSlider.setPaintTicks(true);
      this.preloadSlider.setMinorTickSpacing(20);   
      this.preloadSlider.setMajorTickSpacing(400);      
      
      this.picSlider.setSnapToTicks(true);
      this.picSlider.setPaintTicks(true);

      JPanelRows basePanel = new JPanelRows();

      JPanel row = basePanel.topRow();
      row.add(new JLabel(Jibs.getString("ThumbnailOptionsDialog.4"))); //$NON-NLS-1$
      row.add(this.rowSlider);

      row = basePanel.nextRow();
      row.add(new JLabel(Jibs.getString("ThumbnailOptionsDialog.5"))); //$NON-NLS-1$
      row.add(this.colSlider);

      row = basePanel.nextRow();
      row.add(new JLabel(Jibs.getString("ThumbnailOptionsDialog.6") + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      row.add(this.myPicsPerPageLabel);

      row = basePanel.nextRow();
      row.add(new JLabel(Jibs.getString("ThumbnailOptionsDialog.7"))); //$NON-NLS-1$
      row.add(this.picSlider);

      row = basePanel.nextRow();
      row.add(new JLabel(Jibs.getString("ThumbnailOptionsDialog.8") + Jibs.getString("colon") + " ")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      row.add(this.mySelectedPicLabel);  

      row = basePanel.nextRow();
      row.add(new JLabel("Number of pictures to preload: ")); //$NON-NLS-1$
      row.add(this.preloadSlider);

      row = basePanel.nextRow(FlowLayout.CENTER);
      row.add(saveBtn);
      row.add(cancelBtn);

     	getOptions();      
     	update();

      super.setContentPane(basePanel);
      super.pack();
      SwingUtils.centerOnScreen(this);
      super.setVisible(true);
      saveBtn.requestFocus();
}

   void update()
   {
   	int total = this.rowSlider.getValue() * this.colSlider.getValue();
   	this.myPicsPerPageLabel.setText(Integer.toString(total));
   	this.picSlider.setMaximum(total);
   	this.mySelectedPicLabel.setText(Integer.toString(this.picSlider.getValue()));
   	if ( total > 10)
   	{
         this.picSlider.setPaintLabels(false);
   		this.picSlider.setMajorTickSpacing(5);
   		this.picSlider.setMinorTickSpacing(1);
   	}
   	else
   	{
         this.picSlider.setPaintLabels(false);
   		this.picSlider.setMajorTickSpacing(1);
   		this.picSlider.setMinorTickSpacing(1);
   	}
   	//this.preloadSlider.getValue();asdf
   }

 
   // **************************************************************************
   // PREFERENCES
   // **************************************************************************
   void getOptions()
   {
      this.rowSlider.setValue(this.myOptions.getRows());
      this.colSlider.setValue(this.myOptions.getCols());
      this.preloadSlider.setValue(this.myOptions.getPreloadCount());
      this.picSlider.setValue(this.myOptions.getSelectedPicCell());
   }

   void saveOptions()
   {
      this.myOptions.setRows(this.rowSlider.getValue());
      this.myOptions.setCols(this.colSlider.getValue());
      this.myOptions.setPreloadCount(this.preloadSlider.getValue());
      this.myOptions.setSelectedPicCell(this.picSlider.getValue());
      IShow show = this.myFrame.getShow();
      if ( show != null && show instanceof Thumbnails)
      {
      	Thumbnails thumbs = (Thumbnails)show;
      	thumbs.setRowCol(this.myOptions.getRows(), 
      			this.myOptions.getCols(), this.myOptions.getSelectedPicCell());
      	show.showImage();
      }
   }
}
