/*
 * // * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.jibs.viewer.shows.thumbs;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.GridLayout;
import java.awt.dnd.DragSource;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.Date;

import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JPanel;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.jibs.viewer.IFrame;
import org.gerhardb.jibs.viewer.ISave;
import org.gerhardb.jibs.viewer.IShow;
import org.gerhardb.jibs.viewer.ViewerPreferences;
import org.gerhardb.lib.scroller.IScroll;
import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.swing.GlassPane2;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.Range;
import org.gerhardb.jibs.viewer.IRevert;
import org.gerhardb.lib.dirtree.filelist.FileList;
import org.gerhardb.lib.image.ImageUtil;

/**
 * Thumbnail show.
 * Lots of images gives:
 * Exception in thread "AWT-EventQueue-0" java.lang.OutOfMemoryError: Java heap space
 * -Joption
 * Pass option to the Java runtime environment, where option is one of the options 
 * described on the reference page for the java application launcher. 
 * For example, -J-Xmx48M sets the maximum memory to 48 megabytes. 
 * NOTE: PASSING TO JAR FILE JUST ADJUSTS HOW JAR FILE IS COMPILED BY JAVA.
 * DOES NOT PASS INTO THE JAR FILE!!!!
 * It is a common convention for -J to pass options to the underlying runtime environment.
 * -Xms Minimum memory.  The default value is 2MB.
 * -Xmx Maximum memory.  The default value is 64MB.
 * -Xms500M -Xmx1000M 
 * -J-Xms500M -J-Xmx1000M 
 */
public class Thumbnails extends JPanel implements IShow, IRevert
{
	Color mySelectedColor;

	// -------------------- Things passed into me -------------------------------
	IFrame myViewerFrame = null;
	FileList myFileList;

	// -------------------- Other Things  -------------------------
	int myPicsPerPage;
	int myRows;
	int myCols;
	int mySelectedPicCell; // start at zero.
	ThumbnailCache myCache;
	private Picture[] myPictures;
	int myShowingScrollerValue;
	private GlassPane2 myGlassPane;

	DragSource myDragSource = new DragSource();
	boolean iAmShowing = false;
	ThumbnailSaver myThumbnailSaver;
	BufferedImage myJIBS;
	ThumbnailRotations myRotations;
	ThumbnailOptions myOptions;
	KeyListener myThumbnailKeyListener;

	// ==========================================================================
	// Constructor
	// ==========================================================================

	/**
	 * myThumbnails.length is to make sure current picture remains displayed when switching from resize to thumbnail view.
	 */
	public Thumbnails(IFrame mf, int zeroBasedIndex)
	{
		// Set up double buffered
		super(true);

		this.myViewerFrame = mf;
		this.myFileList = this.myViewerFrame.getFileList();
		this.myViewerFrame.getFileList().getFileListSelectionModel().clearSelection();

		// needed before setRowCol()
		this.myGlassPane = new GlassPane2(this.myViewerFrame.getFrame());

		this.myJIBS = ImageUtil.toBufferedImage(Icons.getIcon(Icons.THUMBNAIL).getImage());
		this.myOptions = new ThumbnailOptions(this);

		// Needed by ThumbnailCache
		this.myRows = this.myOptions.getRows();
		this.myCols = this.myOptions.getCols();
		this.mySelectedPicCell = this.myOptions.getSelectedPicCell();

		// Set the pictures to view
		this.myCache = new ThumbnailCache(this.myRows, this.myCols,
				this.myViewerFrame.getScroller(), this.myOptions, this);
		this.myThumbnailSaver = new ThumbnailSaver(this.myViewerFrame, this,
				this.myCache);
		this.myThumbnailSaver.resetToolbar();
		this.myRotations = new ThumbnailRotations(this.myViewerFrame
				.getScroller());

		// Must be AFTER cache is set up.
		setRowCol(this.myRows, this.myCols, this.mySelectedPicCell);

		this.mySelectedColor = this.myFileList.getSelectionBackground();

		// Make sure we can get focus back here.
		super.setFocusable(true);
		resetFocus();

		this.myThumbnailKeyListener = new KeyActions();
		super.addKeyListener(this.myThumbnailKeyListener);

		super.setBackground(Color.black);

		this.myViewerFrame.getFileList().getFileListSelectionModel().setMouseAlwaysAdds(true);

		// make sure current picture remains displayed when switching from resize to thumbnail view.
		this.myViewerFrame.getScroller().setValue(zeroBasedIndex);

		this.iAmShowing = true;
		//showImage();		Feb 20, 2011
	}

	@Override
	public void paint(Graphics g)
	{
		this.myViewerFrame.getFrame().setCursor(
				Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		super.paint(g);
		//System.out.println("Thumbnails.painting()");
		this.myViewerFrame.getFrame().setCursor(
				Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
	}

	public ISave getSaver()
	{
		return this.myThumbnailSaver;
	}

	public void setRowCol(int rows, int cols, int selectedPicCell)
	{
		this.myRows = rows;
		this.myCols = cols;
		// Adjust to be count from zero
		this.mySelectedPicCell = selectedPicCell - 1;

		// Out with the old.
		if (this.myPictures != null)
		{
			super.removeAll();
		}

		// Figure out and set the layout.
		int hgap = 2;
		int vgap = 2;
		if (this.myRows < 1)
		{
			this.myRows = 1;
		}
		if (this.myCols < 1)
		{
			this.myCols = 1;
		}
		this.myPicsPerPage = this.myRows * this.myCols;
		if (selectedPicCell < 2) // ready for count from zero adjustment
		{
			selectedPicCell = 2; // ready for count from zero adjustment
		}
		else if (selectedPicCell > this.myPicsPerPage)
		{
			selectedPicCell = this.myPicsPerPage;
		}
		super.setLayout(new GridLayout(this.myRows, this.myCols, hgap, vgap));

		// Add the picture components for the new layout
		this.myPictures = new Picture[this.myPicsPerPage];
		for (int i = 0; i < this.myPictures.length; i++)
		{
			this.myPictures[i] = new Picture(this, i + 1);
			this.myPictures[i].setCurrentPicture(false);
			super.add(this.myPictures[i]);
			if (i == this.mySelectedPicCell)
			{
				this.myPictures[i].setCurrentPicture(true);
			}
		}

		// Show everything
		//reloadScroller(); // Added Aug 1, 2007 // Removed 20 Mar 2011
		showImage();// Removed Aug 1, 2007 // Added back 20 Mar 2011
	}

	private void invalidatePicPointer()
	{
		// Cancel current showing stuff to force re-evaluation and loading of all images.
		this.myShowingScrollerValue = -1;
	}

	public int getScrollBarIndex(int cellIndex)
	{
		return this.myViewerFrame.getScroller().getValueZeroBased() + cellIndex
				- this.mySelectedPicCell;
	}

	// ==========================================================================
	// IRevert
	// ==========================================================================
	public void revert()
	{
		this.myViewerFrame.gotoFullScreen(false);
	}

	public boolean isFullScreen()
	{
		return false;
	}

	// ==========================================================================
	// Show Pictures
	// ==========================================================================

	/**
	 * Designed to be called FROM the UI thread.
	 */
	private void quicklyShowPictures()
	{
		if (this.myShowingScrollerValue > 0
				&& this.myShowingScrollerValue == this.myViewerFrame.getScroller().getValueZeroBased())
		{
			// Don't need to do anything if you have not scrolled.
			return;
		}
		//System.out.println("Thumbnails.quicklyShowPictures()");

		setWaitCursor(true);
		this.myShowingScrollerValue = this.myViewerFrame.getScroller().getValueZeroBased();
		
		// subtract one to get the difference between the two.
		int offset = this.myViewerFrame.getScroller().getValueZeroBased() - this.mySelectedPicCell;
		//System.out.println("Thumbnails.quicklyShowPictures().offset: " + offset);
		
		// This is guaranteed to display almost instantly.
		// Then we need to kick thread to populate from disk.
		for (int i = 0; i < this.myPicsPerPage; i++)
		{
			int adjustedIndex = i + offset;
			this.myPictures[i].setImage(adjustedIndex);
		}

		// Starting offset
		int startingIndexToLoad = this.myViewerFrame.getScroller().getValueZeroBased() - this.mySelectedPicCell;

		// Fire up cache starting with current picture set.
		this.myCache.showImages(startingIndexToLoad);

		// System.out.println("Thumbnails.quicklyShowPictures() DONE DONE DONE");
		setWaitCursor(false);
	}

	/**
	 * Designed to be called from something other than the UI thread.
	 */
	void showCurrentPictures()
	{
		//new Throwable("RELOADING THUMBNAIL CACHE").printStackTrace();		
		//System.out.println("showCurrentPictures().START");
		for (int i = 0; i < this.myPicsPerPage && this.myCache.keepLoading(); i++)
		{
			//System.out.println("showCurrentPictures(): " + i   );
			this.myPictures[i].grabImage();
			Thread.yield();
		}
		repaint();
		//System.out.println("showCurrentPictures().DONE"); 
	}

	// ==========================================================================
	// IShow Implementation
	// ==========================================================================

	public void showPageFromScroller()
	{
		//System.out.println("Thumbnails.showPageFromScroller()");	
		this.myCache.pause();
		invalidatePicPointer();
		showImage();
		if (this.myFileList.isVisible())
		{
			this.myFileList.sliderAdjusted();
		}
		// requestFocus() is needed to get the keys to work when something is 
		// picked in the FileList.  Not sure why auto reset did not work same as
		// for other viewers.
		this.requestFocus();
	}

	public void showImage()
	{
		if (!this.iAmShowing) { return; }
		quicklyShowPictures();
		repaint();
		this.myViewerFrame.statusCurrentPage();
	}

	public JComponent getImageDisplay()
	{
		return this;
	}

	public void connect()
	{
		//System.out.println("Thumbnails.connect() ############################==========================================================================");
		this.myViewerFrame.getScroller().setEndMessages(false);
		this.myViewerFrame.getScroller().addScrollerListener(this);
		super.addKeyListener(this.myThumbnailKeyListener);
		super.addKeyListener(this.myViewerFrame.getScroller().getScrollerKeyListener());		
		// This is actually setting cache for the first time.
		this.myCache.resetAndReshow(); // includes showPageFromScroller();		
	}

	public void disconnect()
	{
		// System.out.println("Thumbnails.disconnect()");
		setWaitCursor(true);
		this.myViewerFrame.getScroller().removeScrollerListener(this);
		super.removeKeyListener(this.myThumbnailKeyListener);
		super.removeKeyListener(this.myViewerFrame.getScroller().getScrollerKeyListener());				
		this.myCache.stopCache();
		setWaitCursor(false);
	}

	public void info(StringBuffer sb)
	{
		sb.append(Jibs.getString("Thumbnails.0")); //$NON-NLS-1$
	}

	public void jumpTo(File file)
	{
		// System.out.println("Thumbnails.jumpTo()");
		reloadScroller();
		this.myViewerFrame.getScroller().jumpTo(file);
		showPageFromScroller();
	}

	public JMenuItem makePreferencesItem()
	{
		return ThumbnailOptionsDialog.makeThumbnailPreferenceMenu(
				this.myViewerFrame, this.myOptions);
	}

	public void reloadScroller()
	{
		//System.out.println("Thumbnails.reloadScroller()");
		invalidatePicPointer();
		this.myViewerFrame.getScroller().reloadScroller();
	}

	public void reloadScroller(int index, boolean reloadScroller)
	{
		//System.out.println("Thumbnails.reloadScroller(index)");
		invalidatePicPointer();
		this.myViewerFrame.getScroller().reloadScroller(index, reloadScroller);
	}

	public void resetFocus()
	{
		this.myViewerFrame.getScroller().setAutoFocus(this);
	}

	/**
	 * Make sure all components show wait at same time.
	 * 
	 * @param wait
	 */
	public void setWaitCursor(boolean wait)
	{
		this.myGlassPane.setWaitCursor(wait);
	}

	public void startSlideShow()
	{
		this.myViewerFrame.getScroller().startSlideShow(
				ViewerPreferences.continuousShow());
	}

	public Range showingIndexes()
	{
		int startIndex = -1;
		int stopIndex = -1;

		// Find the first showing index.
		for (int i = 0; i < this.myPictures.length; i++)
		{
			int index = this.myPictures[i].getScrollerIndex();
			if (index > -1)
			{
				startIndex = index;
				break;
			}
		}

		// Find the last showing index.
		for (int i = this.myPictures.length - 1; i > -1; i--)
		{
			int index = this.myPictures[i].getScrollerIndex();
			if (index > 0)
			{
				stopIndex = index;
				break;
			}
		}

		return new Range(startIndex, stopIndex);
	}

	public void swapPictureFocus()
	{
		if (this.myFileList.hasFocus())
		{
			this.myViewerFrame.getScroller().forceFocus();
		}
		else
		{
			this.myFileList.grabFocus();
		}
	}

	// ==========================================================================
	// ScrollerListener Implementation
	// ==========================================================================
	public void scrollerChanged(ScrollerChangeEvent ce)
	{
		switch (ce.getEventType())
		{
		case ScrollerChangeEvent.CURRENT_IMAGE_CHANGED:
			if (!this.myViewerFrame.getScroller().getValueIsAdjusting())
			{
				// System.out.println("Thumbnails.CURRENT_IMAGE_CHANGED");
				invalidatePicPointer();
				showImage();
			}
			break;
		case ScrollerChangeEvent.CURRENT_PAGE_CHANGED:
			if (!this.myViewerFrame.getScroller().getValueIsAdjusting())
			{
				// System.out.println("Thumbnails.CURRENT_PAGE_CHANGED");
				showPageFromScroller();
			}
			break;
		case ScrollerChangeEvent.ONE_FILE_REMOVED:
			// System.out.println("Thumbnails.ONE_FILE_REMOVED");
			this.myCache.remove(ce.getFile());
			this.myRotations.remove(ce.getIndexRemoved());
			showPageFromScroller();
			break;
		case ScrollerChangeEvent.LIST_RELOADED:
			//System.out.println("Thumbnails.LIST_RELOADED");
			this.myRotations.resetRotations(this.myViewerFrame.getScroller().getSize());
			if (ce.getReloadCache() == IScroll.RELOAD_CACHE)
			{
				this.myCache.resetAndReshow(); // includes showPageFromScroller();
			}
			else
			{
				showPageFromScroller();
			}
			break;
		}
	}

	// ==========================================================================
	// KeyActions
	// ==========================================================================
	private class KeyActions implements KeyListener
	{
		protected KeyActions()
		{ /* just to get rid of synthetic warning...*/
		}

		public void keyTyped(KeyEvent event)
		{
			if (event.isConsumed()) { return; }
			// We don't care
			event.consume();
		}

		public void keyPressed(KeyEvent event)
		{
			if (event.isConsumed()) { return; }
			//System.out.println("event.getKeyCode(): " + event.getKeyCode());
			//System.out.println("KEY: " + event.getKeyText(event.getKeyCode()));
			switch (event.getKeyCode())
			{
			case KeyEvent.VK_ENTER:
				if ((event.getModifiersEx() & InputEvent.CTRL_DOWN_MASK) == InputEvent.CTRL_DOWN_MASK)
				{
					revert();
				}
				if ((event.getModifiersEx() & InputEvent.ALT_DOWN_MASK) == InputEvent.ALT_DOWN_MASK)
				{
					revert();
				}
				event.consume();
				return;
			}
		}

		public void keyReleased(KeyEvent event)
		{
			if (event.isConsumed()) { return; }
			//System.out.println("Thumbnails keyPressed: event.getKeyCode(): " + event.getKeyCode() + "  KEY: " + KeyEvent.getKeyText(event.getKeyCode()));
			switch (event.getKeyCode())
			{
			case KeyEvent.VK_ESCAPE:
				// This killed the selection moving from thumbnail to full view.
				// Very inconvenient when trying to decide if a particular picture needs to be selected.
				//myViewerFrame.getActions().getAction("edit", "clear").actionPerformed(null);
				Thumbnails.this.myViewerFrame.getShow().showPageFromScroller();
				event.consume();
				return;
			case KeyEvent.VK_F2:
				Thumbnails.this.myViewerFrame.gotoFullScreen(false);
				event.consume();
				return;
			case KeyEvent.VK_PAGE_DOWN:
			{
				int currentlyAt = Thumbnails.this.myViewerFrame.getScroller().getValueZeroBased();
				int jumpTo = currentlyAt + Thumbnails.this.myPicsPerPage;
				if (jumpTo > Thumbnails.this.myViewerFrame.getScroller()
						.getMaximumZeroBased())
				{
					jumpTo = Thumbnails.this.myViewerFrame.getScroller()
							.getMaximumZeroBased();
				}
				event.consume();
				Thumbnails.this.myViewerFrame.getScroller().jumpTo(jumpTo);
				Thumbnails.this.myViewerFrame.getScroller().updateSelectionInterval();
				return;
			}
			case KeyEvent.VK_PAGE_UP:
			{
				int currentlyAt = Thumbnails.this.myViewerFrame.getScroller().getValueZeroBased();
				int jumpTo = currentlyAt - Thumbnails.this.myPicsPerPage;
				if (jumpTo < 0)
				{
					jumpTo = 0;
				}
				event.consume();
				Thumbnails.this.myViewerFrame.getScroller().jumpTo(jumpTo);
				Thumbnails.this.myViewerFrame.getScroller().updateSelectionInterval();
				return;
			}
			}
		}
	}

	class Test implements Runnable
	{
		private long round = 10;
		long start = new Date().getTime();

		public void run()
		{

			System.out
					.println("==========================================================="); //$NON-NLS-1$
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(9);
			report(1);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(18);
			report(2);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(9);
			report(3);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(18);
			report(4);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(17);
			report(5);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(16);
			report(6);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(15);
			report(7);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(14);
			report(8);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(15);
			report(9);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(16);
			report(10);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(17);
			report(11);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(18);
			report(12);
			Thumbnails.this.myViewerFrame.getScroller().jumpTo(19);
			report(13);
			System.out
					.println("==========================================================="); //$NON-NLS-1$
			if (true) { return; }
		}

		void report(int i)
		{
			long elapsed = new Date().getTime() - this.start;
			System.out.println(i + ": " + Long.toString((elapsed / this.round))); //$NON-NLS-1$
			try
			{
				Thread.sleep(1500);
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
			this.start = new Date().getTime();
		}
	}
}
