/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.StringTokenizer;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.ILoadingMessage;
import org.gerhardb.lib.dirtree.DirectoryTreeNode;

/**
 * Utility performing various actions.
 */
public class DTNReader
{
	private static final String VERSION_TOKEN_TARGET_COUNTED = "# v2i "; //$NON-NLS-1$
	private static final String VERSION_TOKEN_NO_TARGET_COUNT = "# v2x "; //$NON-NLS-1$
	private static final String DATE_MARKER = "# Date: "; //$NON-NLS-1$
	private static final String ROOT_NAME_MARKER = "# Root Name: "; //$NON-NLS-1$
	private static final String ROOT_PATH_MARKER = "# Root Path: "; //$NON-NLS-1$
	public static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss"); //$NON-NLS-1$

	private DTNReaderWriter myBase;
	private DirectoryTree myTree;
   private ILoadingMessage myLoadingMsg;
   private boolean iCountTargets = false;
   private Date myLastReadDate;
   private String myStoredRootPath;
   private String myStoredVolumeName;
   
   boolean iNeedToShowMissingDirectoryMessage = true;
   
   // There is one level for each tree branch.
   // So 100 levels means that the subdirectory structure can not go
   // more than 100 levels deep. There can be an infinate number of
   // subdirectories at each level.
   DirectoryTreeNode[] levels = new DirectoryTreeNode[100];	  
   
   /**
	 * Used to restore from a file store.
	 * @param rootDirectory
	 * @return
	 */
   // iCountTargetFiles ViewerPreferences.countImageFilesInNodes()
	public DTNReader(File file, DirectoryTree tree, ILoadingMessage loadingMsg, DTNReaderWriter base)
	{
		this.myTree = tree;
		this.myLoadingMsg = loadingMsg;
		this.myBase = base;
		
		System.out.println("DTNReader reading file: " + file);
		
		// Actual test for failure is returning null from call to getRootNode()
		// so we don't need to report anything here.
		try
		{
			BufferedReader reader = openFileReader(file);
			if (reader != null)
			{
				readFile(reader);
			}
		}
		catch(Exception ex)
		{
			//ex.printStackTrace();
		}
	}

	public DirectoryTreeNode getRootNode()
	{
		return this.levels[0];
	}
	
	public boolean getImagesCounted()
	{
		return this.iCountTargets;
	}
	
	public Date getLastReadDate()
	{
		return this.myLastReadDate;
	}
	
	public String getStoredRootPath()
	{
		return this.myStoredRootPath;
	}
	
	public String getStoredVolumeName()
	{
		return this.myStoredVolumeName;
	}	
	
	// TODO FOO java 7
	public static String getNameOfDrive(String path) // NO_UCD
	{
		javax.swing.filechooser.FileSystemView view = 
			javax.swing.filechooser.FileSystemView.getFileSystemView();
		File dir = new File(path);
		String name = view.getSystemDisplayName(dir);
		if (name == null) { return null; }
		name = name.trim();
		if (name == null || name.length() < 1) { return null; }
		int index = name.lastIndexOf(" (");
		if (index > 0)
		{
			name = name.substring(0, index);
		}
		return name;
	}

   // ==========================================================================
   // Retrieve 
   // ==========================================================================
   private BufferedReader openFileReader(File file)
   throws Exception
   {
 		File passedInFile = this.myBase.myFileFromPathManagerStartingJibsFile;
		if (passedInFile != null && passedInFile.exists() && passedInFile.isFile())
		{
		  	// This is what we do if user double-clicked on *.jibs.
			file = passedInFile;
		}
		else
		{
			// This is what we do otherwise.
			// If file coming in is a directory, add JibTree.txt, otherwise
			// just use the file.
			if (file.isDirectory())
			{
				String fileName = file.getAbsolutePath() + "/" + this.myBase.getFileName();//$NON-NLS-1$
				file = new File(fileName);
			}
			
			// Make sure file is there before we start.
			if (file.exists())
			{
				this.myLoadingMsg.setText(AppStarter.getString("TreeManager.13") + AppStarter.getString("colon") + " " + file); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			}
			else
			{
				String msg = AppStarter.getString("TreeManager.14") + AppStarter.getString("colon") + " " + file;//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				this.myLoadingMsg.setText(msg); 
				System.out.println(msg);
				throw new Exception("File not found: " + file); //$NON-NLS-1$
			}
		}

		try
		{
			return new BufferedReader(new FileReader(file));
		}
		catch (java.io.FileNotFoundException ex)
		{
			System.out.println(AppStarter.getString("TreeManager.22") + AppStarter.getString("colon") + " " + file); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			System.out.println(AppStarter.getString("TreeManager.23")); //$NON-NLS-1$
			this.myLoadingMsg.setText(AppStarter.getString("TreeManager.24")); //$NON-NLS-1$
			return null;
		}   	
   }
   
	private void readFile(BufferedReader reader)
	{
		try
		{
			// Check the version.
			String aLine = reader.readLine();
			if (aLine == null)
			{
				this.myLoadingMsg.setText(AppStarter.getString("DTNReaderWriter.20")); //$NON-NLS-1$
				return;
			}
			
	      // If it has the correct version of images counted,
	      // just proceed.  No further check needed.
			if (aLine.startsWith(VERSION_TOKEN_TARGET_COUNTED))
			{
				this.iCountTargets = true;
			}
			else
	      {
		      if (this.myBase.iCountTargetFiles)
		      {
		      	// If its not counting images and its supposed to, do it now.
					this.myLoadingMsg.setText(AppStarter.getString("DTNReaderWriter.19")); //$NON-NLS-1$
					return;
		      }
		      // If its not the with or without images, it must the the wrong file version.
		      else if (!aLine.startsWith(VERSION_TOKEN_NO_TARGET_COUNT))
				{
					this.myLoadingMsg.setText(AppStarter.getString("DTNReaderWriter.18")); //$NON-NLS-1$
					return;
				}
	      }
			
	      // We got one of the two good file versions if we get to here.
	      
			// Get first data line.
			aLine = reader.readLine();
			while (aLine != null)
			{
				//Jibs.log(aLine);
				if (aLine.startsWith(DATE_MARKER))
				{
					String strDate = aLine.substring(DATE_MARKER.length());
					try
					{
						this.myLastReadDate = DATE_FORMAT.parse(strDate);
						AppStarter.log(strDate);
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
					}
				}
				else if (aLine.startsWith(ROOT_PATH_MARKER))
				{
					this.myStoredRootPath = aLine.substring(ROOT_PATH_MARKER.length());
					AppStarter.log(this.myStoredRootPath);
				}
				else if (aLine.startsWith(ROOT_NAME_MARKER))
				{
					this.myStoredVolumeName = aLine.substring(ROOT_PATH_MARKER.length());
					AppStarter.log(this.myStoredVolumeName);					
				}
				else if (!aLine.startsWith("#")) //$NON-NLS-1$
				{
					//System.out.println("===========================================");
					//System.out.println(aLine);
					// Pattern is:
					// depth ^t file ^t fileCount ^t fileCountTree ^t dirCountTree
					StringTokenizer tokens = new StringTokenizer(aLine, "\t"); //$NON-NLS-1$
					// Count is optional, so we only process lines with depth & file.
					// Skip empty lines
					if (tokens.countTokens() > 5)
					{
						// If any ONE line is broken,
						// we want to continue on to other lines.
						try
						{
							useLine(tokens);
						}
						catch (Exception ex)
						{
							ex.printStackTrace();
							this.myLoadingMsg.setText(AppStarter.getString("TreeManager.21")); //$NON-NLS-1$
						}
					}
				}
				aLine = reader.readLine();
			}
			reader.close();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			this.myLoadingMsg.setText(AppStarter.getString("TreeManager.25")); //$NON-NLS-1$ Generally unknown exception while loading...
		}
	}
   
	private void useLine(StringTokenizer tokens)
	throws Exception
   {
		// Understand the line.
		int depth = Integer.parseInt(tokens.nextToken());
   	String fileName = tokens.nextToken();
		
		fileName = this.myStoredRootPath + fileName.substring(1); // File.separator
		File file = new File(fileName);
	   try
	   {
          // If the file is not there, this will throw an error
          // and this line will be skipped.
          DirectoryTreeNode node =
             new DirectoryTreeNode(this.myTree, file, this.myLoadingMsg);
          
          restoreANode(tokens, node);

          // The actual hierarchy is stored in the nodes
          // with the parent having a complete list of its children.
          // The tree is built from root down.
          // The single root is level zero and we will ignore
          // anything less than zero.
          if ( depth > 0 )
          {
             // Here we add the node the tree by registering it as
             // a child of the parent node.
             this.levels[depth - 1].add(node);

             // Now we temporarily store the node in the level
             // so that any child nodes can attach themselves
             // to this node. The level hierarchy constantly
             // changes an only reflects the path the the current
             // subdirectory being loaded.
             this.levels[depth] = node;
          }
          else if ( depth == 0 )
          {
             // The file should have only one root at level zero.
             // Here we take only the first root we find.
             // Any other root pretenders will be under that root.
             if ( this.levels[0] == null  )
             {
                this.levels[0] = node;
             }
             else
             {
                this.levels[0].add(node);
             }
          }
       }
       catch (FileNotFoundException ex)
       {
          if ( this.iNeedToShowMissingDirectoryMessage )
          {
        	  this.iNeedToShowMissingDirectoryMessage = false;
             System.out.println(
                AppStarter.getString("TreeManager.16")); //$NON-NLS-1$
             System.out.println(
                AppStarter.getString("TreeManager.17")); //$NON-NLS-1$
             System.out.println(
                AppStarter.getString("TreeManager.18")); //$NON-NLS-1$
             System.out.println(
                AppStarter.getString("TreeManager.19")); //$NON-NLS-1$
          }
          System.out.println(ex.getMessage());
          this.myLoadingMsg.setText(AppStarter.getString("TreeManager.20")); //$NON-NLS-1$
       }
    }
	
   private static void restoreANode(StringTokenizer tokens, DirectoryTreeNode node)
   throws Exception
   {
   	node.myNodeDirCount = Integer.parseInt(tokens.nextToken());
   	node.myNodeAllFileCount = Integer.parseInt(tokens.nextToken());
   	node.myNodeAllFileSize = Long.parseLong(tokens.nextToken());    
   	node.myNodeTargetFileCount = Integer.parseInt(tokens.nextToken());    
   	node.myNodeTargetFileSize = Long.parseLong(tokens.nextToken());      
   }
   
}
