/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.PrintWriter;
import java.util.Date;
import java.util.Enumeration;
import javax.swing.filechooser.FileSystemView;

/**
 * Pairs with DTNReader
 */
public class DTNWriter
{
	DTNReaderWriter myBase;
	
	DTNWriter(DTNReaderWriter base)
	{
		this.myBase = base;
	}

	public static String getNameOfDrive(String path)
	{
		javax.swing.filechooser.FileSystemView view = 
			javax.swing.filechooser.FileSystemView.getFileSystemView();
		File dir = new File(path);
		String name = view.getSystemDisplayName(dir);
		if (name == null) { return null; }
		name = name.trim();
		if (name == null || name.length() < 1) { return null; }
		int index = name.lastIndexOf(" (");
		if (index > 0)
		{
			name = name.substring(0, index);
		}
		return name;
	}

   // ==========================================================================
   // Store 
   // ==========================================================================
  /**
    * If directoryToStoreIn is null, will store in directory of root node.
	 */
   public void store(DirectoryTreeNode rootNode)
      throws Exception
   {
   	if (rootNode == null || rootNode.getDirectory() == null)
   	{
   		return;
   	}
     	String dir = rootNode.getDirectory().toString();
     	
      File directoryToStoreIn = this.myBase.myFileFromPathManagerStartingJibsFile;
     	if (directoryToStoreIn == null)
   	{
         String fileName = dir + "/" + this.myBase.getFileName(); //$NON-NLS-1$
         directoryToStoreIn = new File(fileName);   		
   	}
      	
     	System.out.println("DTNWriter storing into: " + directoryToStoreIn);
     	
      PrintWriter writer =
         new PrintWriter(new BufferedWriter(new FileWriter(directoryToStoreIn)));
      
      // Add format version
      if (this.myBase.iCountTargetFiles)
      {
      	writer.print(DTNReaderWriter.VERSION_TOKEN_TARGET_COUNTED);
      }
      else
      {
      	writer.print(DTNReaderWriter.VERSION_TOKEN_NO_TARGET_COUNT);
      }
      writer.println(this.myBase.getFileName()); 
      
      writer.print(DTNReaderWriter.DATE_MARKER); 
      Date now = new Date();
      writer.println(DTNReaderWriter.DATE_FORMAT.format(now));

      writer.print(DTNReaderWriter.ROOT_PATH_MARKER); 
      File treeRootDir = rootNode.getDirectory();
      writer.println(treeRootDir.getAbsolutePath());
      
   	writer.print(DTNReaderWriter.ROOT_NAME_MARKER); 
      File fileSystemRoot = treeRootDir;
      FileSystemView fsv = FileSystemView.getFileSystemView();
     	while(fileSystemRoot != null && !fsv.isFileSystemRoot(fileSystemRoot))
   	{
      	fileSystemRoot = fileSystemRoot.getParentFile();
   	}
     	if (fileSystemRoot != null)
     	{
         writer.println(getNameOfDrive(fileSystemRoot.getAbsolutePath()));     		
     	}
      
      writer.println("# depth path directories fileCount fileSize imageCount imageSize");  //$NON-NLS-1$
      
      // Remove path from front of directory name to make it relative.
      int skip = dir.length();
      storeAllNodes(writer, rootNode, 0, skip);
      writer.flush();
      writer.close();
   }

   private void storeAllNodes(
      PrintWriter writer, DirectoryTreeNode node, int depth, int skip)
   {
   	storeANode(writer, node, depth, skip);
      Enumeration<?> kids = node.children();
      while (kids.hasMoreElements())
      {
         DirectoryTreeNode childNode = (DirectoryTreeNode)kids.nextElement();
         storeAllNodes(writer, childNode, depth + 1, skip);
      }
   }
   
   private void storeANode(PrintWriter writer, DirectoryTreeNode node, int depth, int skip)
   {
   	String path = node.getDirectory().getAbsolutePath();
    	//System.out.println("DTNWriter storing: " + path);
    	
      // What we need to recreate a node
   	writer.print(depth);
      writer.print("\t*"); //$NON-NLS-1$
      if (skip < path.length())
      {
         writer.print(path.substring(skip));      	
      }
      else
      {
         writer.print(File.separator);      	
      }
      
      // Following restored by restoreANode      
      writer.print("\t"); //$NON-NLS-1$
      writer.print(node.myNodeDirCount);
      
      writer.print("\t"); //$NON-NLS-1$
      writer.print(node.myNodeAllFileCount);
      
      writer.print("\t"); //$NON-NLS-1$
      writer.print(node.myNodeAllFileSize);
      
      writer.print("\t"); //$NON-NLS-1$
      writer.print(node.myNodeTargetFileCount);
      
      writer.print("\t"); //$NON-NLS-1$
      writer.print(node.myNodeTargetFileSize);
      
      writer.println(""); // now add the line ending //$NON-NLS-1$
   }
}
