/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.lib.dirtree;

import java.awt.Cursor;
import java.io.*;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import javax.swing.JOptionPane;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;

import org.gerhardb.jibs.diskUsage.ExplorerBox;
import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.ILoadingMessage;
import org.gerhardb.lib.io.FileNameComparatorInsensative;
import org.gerhardb.lib.io.FileNameObjectComparatorInsensative;
import org.gerhardb.lib.util.ModalProgressDialogFlex;

/**
 * A directory node designed to count its files and populate
 * all of its sub-directories.
 * <p>
 * The sub-directory population occurs in a depth-first fashion.
 * Specifically, for a given root, its children are identified and files counted.
 * Then the same recursively occurs for each sub-directory of the root.
 * <p>
 * This allows the highest level directory to be displayed immediately while
 * other directories are populating.  If a user chooses an unpopulated
 * sub-directory, then that sub-directory is immediately populated, but its
 * children are not recursed until the normal time.
 * <p>
 * This is ALWAYS a directory and is tested on construction.
 */
public class DirectoryTreeNode
   extends DefaultMutableTreeNode
{
	private DirectoryTree myTree;
   private String myDisplayName; // Name of this component
   File myDirectory; // Full pathname
   private static boolean clsCancelLoad = false;

   // Items to save and restore.
   protected int myNodeDirCount = 0;    
   protected int myNodeAllFileCount = 0; // used to show files underneath
   protected int myNodeTargetFileCount = 0; // used to show files underneath   
   protected long myNodeAllFileSize = 0;
   protected long myNodeTargetFileSize = 0;
   
 // boolean iAmPopulating = true;

   // ==========================================================================
   // Constructor
   // ==========================================================================
   public DirectoryTreeNode(DirectoryTree tree, File f, ILoadingMessage loadingMsg)
      throws FileNotFoundException
   {
      this.myTree = tree;
      this.myDisplayName = f.getName();

      if (this.myDisplayName.trim().equals("")) //$NON-NLS-1$
      {
         // This happens for windows drives which are not files so they have
         // no names.
         this.myDisplayName = f.toString();
      }
      this.myDirectory = f;
      if (!f.isDirectory())
      {
         throw new FileNotFoundException(f.toString()
            + " " + AppStarter.getString("DirectoryTreeNode.1")); //$NON-NLS-1$ //$NON-NLS-2$
      }
		if (loadingMsg != null)
		{
			loadingMsg.setMessage(loadingMsg.getNextIncrement() + " " + this.myDisplayName); //$NON-NLS-1$
		}      
   }

   // ==========================================================================
   // Getters
   // ==========================================================================

   public static void cancelRecursiveLoading()
   {
      clsCancelLoad = true;
   }

   public static void resetRecursiveLoading()
   {
      clsCancelLoad = false;
   }

   public File getDirectory()
   {
   	return this.myDirectory;
   }
   
   // Get full path
   public String getAbsolutePath()
   {
      return this.myDirectory.getAbsolutePath();
   }

   public File getFile()
   {
      return this.myDirectory;
   }

   // Get myDisplayName
   public String getName()
   {
      return this.myDisplayName;
   }

   public int getDirCountForTree()
   {
   	return this.myNodeDirCount;
   }

   public int getTotalFileCount()
   {
      return this.myNodeAllFileCount;
   }
   
   public int getTargetFileCount()
   {
      return this.myNodeTargetFileCount;
   }
   
   public long getTotalFileSize()
   {
      return this.myNodeAllFileSize;
   }

   public long getTargetFileSize()
   {
      return this.myNodeTargetFileSize;
   }
   
   public TreeTotals getTreeTotals()
   {
   	TreeTotals rtnMe = new TreeTotals();
   	addNodeToTotals(this, rtnMe);
   	loopOverKids(this, rtnMe);
		return rtnMe;
   }
   
   private void loopOverKids(DirectoryTreeNode node, TreeTotals tt)
   {
		Enumeration<?> kids = node.children();
		while (kids.hasMoreElements())
		{
			DirectoryTreeNode aChild = (DirectoryTreeNode)kids.nextElement();
			//System.out.println(aChild); 
			addNodeToTotals(aChild, tt);
			loopOverKids(aChild, tt);
		}      	
   }
   
   private void addNodeToTotals(DirectoryTreeNode node, TreeTotals tt)
   {
   	tt.aggregateDirectoryCount += node.myNodeDirCount;
   	tt.aggregateTotalFileCount += node.myNodeAllFileCount;
   	tt.aggregateTargetFileCount += node.myNodeTargetFileCount;
   	tt.aggregateTotalFileSize += node.myNodeAllFileSize;
   	tt.aggregateTargetFileSize += node.myNodeTargetFileSize;   	
   }

   public class TreeTotals
   {
   	public int aggregateDirectoryCount = 0;
   	public int aggregateTotalFileCount = 0;
   	public int aggregateTargetFileCount = 0;
   	public long aggregateTotalFileSize = 0;
   	public long aggregateTargetFileSize = 0;
   }

   // ==========================================================================
	// Recursive Builder
	// ==========================================================================
	public void populate(ILoadingMessage loadingMsg)
	{
		System.out.println("NODE POPULATING");
		// Count files and get Directories
		File[] dirArray = countFiles();
		
		try
		{
			Arrays.sort(dirArray, new FileNameComparatorInsensative());
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		for (int i = 0; i < dirArray.length; i++)
		{
			try
			{
				DirectoryTreeNode node = new DirectoryTreeNode(this.myTree,
						dirArray[i], loadingMsg);
				super.add(node);
				node.populate(loadingMsg);
		   }
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}

   public void recount()
   {
   	countFiles();
	   
	   this.myTree.updateTreeLabels();     
	   DefaultTreeModel dtm = (DefaultTreeModel)this.myTree.getModel();
	   dtm.nodeStructureChanged(this);
  }

   public void incrementFileCount(long newFileSize)
   {
   	this.myNodeAllFileCount++;
   	this.myNodeTargetFileCount++;
   	this.myNodeAllFileSize = this.myNodeAllFileSize + newFileSize;
   	this.myNodeTargetFileSize = this.myNodeTargetFileSize + newFileSize;
      this.myTree.updateTreeLabels();
   }

   public void decrementFileCount(long newFileSize)
   {
   	this.myNodeAllFileCount--;
   	this.myNodeTargetFileCount--;
   	this.myNodeAllFileSize = this.myNodeAllFileSize - newFileSize;
   	this.myNodeTargetFileSize = this.myNodeTargetFileSize - newFileSize;
      this.myTree.updateTreeLabels();
   }
   
   public void reloadNodes()
   {
   	final ModalProgressDialogFlex flex = new ModalProgressDialogFlex(
   			this.myTree.getTopWindow(), AppStarter.getString("DirectoryTreeNode.17"), //$NON-NLS-1$
   			AppStarter.getString("DirectoryTreeNode.18"), null, null, 300);   	 //$NON-NLS-1$

   	Runnable runMe = new Runnable()
		{
			public void run()
			{
				reloadNodes(flex);
			}
		};
		
		flex.run(runMe);			
   }
 
   /**
    * TODO REDO POOH -- This pulls in stuff from far away the needs to be fixed sometime.
    * Problem is that that thread continues while this starts working in another thread but this needs
    * to call back to originating routine.  Once we can pass in a function (Java 7?) fix will be real simple.
    * @param box
    * @param rootDirectory
    * @param supressRootDirectories
    * @param writer
    */
   public void reloadNodes(final ExplorerBox box, final boolean supressRootDirectories, final DTNWriter writer)
   {
   	final ModalProgressDialogFlex flex = new ModalProgressDialogFlex(
   			this.myTree.getTopWindow(), AppStarter.getString("DirectoryTreeNode.17"), //$NON-NLS-1$
   			AppStarter.getString("DirectoryTreeNode.18"), null, null, 300);   	 //$NON-NLS-1$

   	Runnable runMe = new Runnable()
		{
			public void run()
			{
				reloadNodes(flex);
				try
				{
					writer.store(DirectoryTreeNode.this);
				}
				catch(Exception ex)
				{
	            JOptionPane.showMessageDialog(
		            	 box,
		                "Unexpected Problem storing node list",
		                ex.getMessage(), 
		                JOptionPane.ERROR_MESSAGE );
				}
				
				try
				{
					// Reshow the ExplorerBox.
					box.setRootDirectory(DirectoryTreeNode.this.myDirectory, supressRootDirectories);
				}
				catch(Exception ex)
				{
					ex.printStackTrace();
				}
			}
		};
		
		flex.run(runMe);			
   }   
   
   public void reloadNodes(ModalProgressDialogFlex flex)
	{
		// Remove the children.
		super.removeAllChildren();

		// Populate handles adding the counts.
		this.populate(flex);
		
		// Cause labels to show
		((DefaultTreeModel) this.myTree.getModel()).nodeStructureChanged(this);
	}

   // ==========================================================================
	// Private Classes
	// ==========================================================================
   
   private File[] countFiles() 
   {
		//Date start = new Date();
		ArrayList<File> rtnDirectories = new ArrayList<File>(100);
   	boolean countTargetFiles = this.myTree.getCountTargetFiles();
   	
      this.myNodeDirCount = 0;       
      this.myNodeAllFileCount = 0; 
      this.myNodeTargetFileCount = 0; 
      this.myNodeAllFileSize = 0;
      this.myNodeTargetFileSize = 0;   	   	

		File[] files = this.myDirectory.listFiles();
		if (files == null || files.length == 0 || clsCancelLoad) { return new File[0]; }
		      
		// Get Directories
		for (int i = 0; i < files.length; i++)
		{
			//System.out.println("-----> a: " + (new Date().getTime() - start.getTime())); 
			//System.out.println(files[i]);
			if (files[i].isDirectory())
			{
				if (this.myTree.isNotAnExcludedDirectory(files[i]))
				{
					if (this.myTree.countHiddenDirectories())
					{			
						//System.out.println("Adding: "  + files[i]);
						rtnDirectories.add(files[i]);
						this.myNodeDirCount++;
					}
					else
					{		
						// Ignore hidden directories using the linux convention.
						// Also, we are not going to show hidden directories, so don't count them.
						if (!files[i].isHidden())
						{
							rtnDirectories.add(files[i]);
							this.myNodeDirCount++;
						}
					}
				}
			}
			else
			{
				//System.out.println("-----> c: " + (new Date().getTime() - start.getTime()));
				this.myNodeAllFileCount++;
				this.myNodeAllFileSize = this.myNodeAllFileSize + files[i].length();
				if (countTargetFiles)
				{
					//System.out.println("Filtering images");
					if (this.myTree.getFileFilter().accept(files[i])) //ImageFactory.getImageFactory().getFilter().accept(files[i]))
					{
						this.myNodeTargetFileCount++;
						this.myNodeTargetFileSize = this.myNodeTargetFileSize + files[i].length();
					}
				}
			}
		}   	
		//System.out.println("-----> d: " + (new Date().getTime() - start.getTime()));
		File[] rtnMe = new File[rtnDirectories.size()];
		rtnDirectories.toArray(rtnMe);
		return rtnMe;
   }
   
	/**
	 * Adding a new directory after constructing the FileTree.
	 */
	private synchronized void addNode(File dirFile)
	{
		if (!dirFile.isDirectory())
		{
			return;
		}

		// Do not add a new node if
		// the required node is already there
		int childCount = getChildCount();
		for (int i = 0; i < childCount; i++)
		{
			DirectoryTreeNode node = (DirectoryTreeNode) getChildAt(i);
			if (node.myDisplayName.equals(this.myDisplayName))
			{
				// Already exists
				return;
			}
		}

		// Remember what's showing to restore later.
		TreePath pathToShow = new TreePath(getPath());
		TreePath[] expanded = this.myTree.getExpandedDescendants(
				this.myTree.getRootNode(), pathToShow);

		// Add a new node
		try
		{
			DirectoryTreeNode node = 
				new DirectoryTreeNode(this.myTree, dirFile, null);
			super.add(node);
			pathToShow = pathToShow.pathByAddingChild(node);
		} 
		catch (Exception e)
		{
			e.printStackTrace();
		}

		// Sometimes the new sub-directory does not show.  
		// Hopefully this will fix.  31 May 2009
		// Cause labels to show
		((DefaultTreeModel) this.myTree.getModel()).nodeStructureChanged(this);

		sort();
		
		// Return path to prior position
		this.myTree.expandPath(expanded);
		
		// Sometimes the new sub-directory does not show.  
		// Hopefully this will fix.
		this.myTree.expandPath(pathToShow);	    
		
		// This was the key.  
		// It causes a new subdirectory to become visable if it would be hidden.
		this.myTree.scrollPathToVisible(pathToShow);
	}
	
	/**
	 * Sort child nodes, but not below current children.
	 * Triggers node structure change to show sort.
	 */
	private void sort()
	{
		ArrayList<DirectoryTreeNode> list = new ArrayList<DirectoryTreeNode>(super.getChildCount());
		Enumeration<?> kids = super.children();
		while (kids.hasMoreElements())
		{
			list.add((DirectoryTreeNode)kids.nextElement());
		}
		Object[] nodeArray = list.toArray();
		try{Arrays.sort(nodeArray, new FileNameObjectComparatorInsensative());}
		catch(Exception ex){ex.printStackTrace();}
		super.removeAllChildren();
		for (int i = 0; i < nodeArray.length; i++)
		{
			super.add((DirectoryTreeNode)nodeArray[i]);
		}		
		DefaultTreeModel model = (DefaultTreeModel) this.myTree.getModel();
		model.nodeStructureChanged(this);
	}

   // ==========================================================================
   // Directory Manipulation
   // ==========================================================================
   public void newSubDir()
   {
      String parentDir = this.myDirectory.getAbsolutePath();
      String newDirName =
         JOptionPane.showInputDialog(
         AppStarter.getString("DirectoryTreeNode.3")); //$NON-NLS-1$
      if (newDirName == null)
      {
         return;
      }
      newDirName = newDirName.trim();
      if (newDirName.length() == 0)
      {
         return;
      }

      try
      {
         File newDir = new File(parentDir, newDirName);
         if (newDir.mkdir())
         {
            //System.out.println("Added this directory: " + newDirName + " to " +  parentDir); //$NON-NLS-1$ //$NON-NLS-2$
            this.addNode(newDir);
         }
      }
      catch (SecurityException ex)
      {
         ex.printStackTrace();
      }
   }

   public void renameDir()
   {
      String fullOldName = null;
      String fullNewName = null;
      try
      {
          String oldName = this.myDirectory.getName();
          String question = AppStarter.getString("DirectoryTreeNode.4") + AppStarter.getString("colon") + " ";//$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
          String newDirName = JOptionPane.showInputDialog(question, oldName); 
          if (newDirName == null)
          {
             return;
          }
          newDirName = newDirName.trim();
          if (newDirName.length() == 0)
          {
             return;
          }
          
         String parentDir = this.myDirectory.getParent();
         if (parentDir != null)
         {
         	fullOldName = parentDir + File.separator + oldName; //$NON-NLS-1$
            fullNewName = parentDir + File.separator + newDirName; //$NON-NLS-1$
         }
         else
         {
         	fullOldName = oldName; 
            fullNewName = newDirName; 
         }
         
         if (!this.myDirectory.exists())
         {
            JOptionPane.showMessageDialog(
            		this.myTree,
                  AppStarter.getString("DirectoryTreeNode.23"), //$NON-NLS-1$
                  AppStarter.getString("DirectoryTreeNode.24"), //$NON-NLS-1$
                  JOptionPane.INFORMATION_MESSAGE); 
            return;
         }
         if (!this.myDirectory.canWrite())
         {
            JOptionPane.showMessageDialog(
            		this.myTree,
                  AppStarter.getString("DirectoryTreeNode.25"), //$NON-NLS-1$
                  AppStarter.getString("DirectoryTreeNode.26"), //$NON-NLS-1$
                  JOptionPane.INFORMATION_MESSAGE); 
            return;
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         JOptionPane.showMessageDialog(
         		this.myTree,
               AppStarter.getString("DirectoryTreeNode.27") + ex.getLocalizedMessage(), //$NON-NLS-1$
               AppStarter.getString("DirectoryTreeNode.28"), //$NON-NLS-1$
               JOptionPane.INFORMATION_MESSAGE);         	
      }
         
     //System.out.println("DTN.rename().fullOldName: " + fullOldName); //$NON-NLS-1$
     //System.out.println("DTN.rename().fullNewName: " + fullNewName); //$NON-NLS-1$
         
     try
     {
    	 File newDir = new File(fullNewName);
         if (newDir.exists())
         {
            JOptionPane.showMessageDialog(
            		this.myTree,
                  AppStarter.getString("DirectoryTreeNode.31"), //$NON-NLS-1$
                  AppStarter.getString("DirectoryTreeNode.32"), //$NON-NLS-1$
                  JOptionPane.INFORMATION_MESSAGE); 
            return;
         }

         // Physically rename on the file system.
         // Will return false if it is not successful.
         if (this.myDirectory.renameTo(newDir))
         {
	         this.myDirectory = newDir;
	         
	         this.myDisplayName = this.myDirectory.getName();
	         ((DefaultTreeModel)this.myTree.getModel()).nodeChanged(this);
	         
	         DirectoryTreeNode theParent = (DirectoryTreeNode)super.getParent();
	         if (theParent != null)
	         {
	         	theParent.sort();
	         }
	         
	         renameDirUpdateOther(fullOldName, fullNewName);
         }
         else
         {
            JOptionPane.showMessageDialog(
            		this.myTree,
                  AppStarter.getString("DirectoryTreeNode.33"), //$NON-NLS-1$
                  AppStarter.getString("DirectoryTreeNode.34"), //$NON-NLS-1$
                  JOptionPane.INFORMATION_MESSAGE);         	
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         JOptionPane.showMessageDialog(
         		this.myTree,
               AppStarter.getString("DirectoryTreeNode.35") + ex.getLocalizedMessage(), //$NON-NLS-1$
               AppStarter.getString("DirectoryTreeNode.36"), //$NON-NLS-1$
               JOptionPane.INFORMATION_MESSAGE);         	
      }
   } 

   protected void renameDirUpdateOther(@SuppressWarnings("unused") String fullOldName, @SuppressWarnings("unused") String fullNewName)
   {
   	// for subclasses
   }
   
   public void deleteDir()
   {
      File[] files = this.myDirectory.listFiles();
      if (files != null && files.length > 0)
      {
         int answer = JOptionPane.showConfirmDialog(
                      null,
                      this.myDisplayName
                      + " " + AppStarter.getString("DirectoryTreeNode.13") //$NON-NLS-1$ //$NON-NLS-2$
                      + AppStarter.getString("DirectoryTreeNode.14") //$NON-NLS-1$
                      + AppStarter.getString("DirectoryTreeNode.15"), //$NON-NLS-1$
                      AppStarter.getString("DirectoryTreeNode.12") + " " + this.myDisplayName + "?", //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
                      JOptionPane.YES_NO_OPTION);

         if (answer == JOptionPane.NO_OPTION)
         {
            return;
         }
      }

      DefaultMutableTreeNode aParent = (DefaultMutableTreeNode)super.getParent();
      if (aParent == null)
      {
         JOptionPane.showMessageDialog(
            null,
            AppStarter.getString("DirectoryTreeNode.11"), //$NON-NLS-1$
            AppStarter.getString("DirectoryTreeNode.0"), //$NON-NLS-1$
            JOptionPane.INFORMATION_MESSAGE);
         return;
      }

      if (this.myDirectory.delete() != true)
      {
         return;
      }

      TreePath[] expanded =
         this.myTree.getExpandedDescendants(
        		 this.myTree.getRootNode(), 
        		 new TreePath(getPath()));

      this.parent.remove(this);
      
      reloadNodes((ModalProgressDialogFlex)null);
      
      this.myTree.expandPath(expanded);
   }
   
   public void deleteEmptyDirectories()
   {
   	
      TreePath[] expanded =
         this.myTree.getExpandedDescendants(
        		 this.myTree.getRootNode(), 
        		 new TreePath(getPath()));

      recursivelyDeleteIfEmpty(this);
      
      this.myTree.myTopWindow.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      reloadNodes((ModalProgressDialogFlex)null);
      this.myTree.myTopWindow.setCursor(Cursor.getDefaultCursor());
      
      this.myTree.expandPath(expanded);
  }
   
  private boolean recursivelyDeleteIfEmpty(DirectoryTreeNode node)
  {	  
	  //System.out.println("Entered:    " + node.getFile().getAbsolutePath());
	  boolean relaodNodes = true;
	  
	  // Do kids first so node can see if it is empty.
	  // Redo enum every time a child is deleted.
	  reloadNodes:
	  while (relaodNodes)
	  {
	     Enumeration<?> kids = node.children();
	     while(kids.hasMoreElements())
	     {
	   	  DirectoryTreeNode childNode = (DirectoryTreeNode)kids.nextElement();
			  //System.out.println("Child Node: " + childNode.getFile().getAbsolutePath());
		     if (recursivelyDeleteIfEmpty(childNode))
		     {
		   	  continue reloadNodes;
		     }
	     }	
	     relaodNodes = false;
	  }
	     
     // Now see if this node can kill itself.
	  if (node.isLeaf())
	  {    
		  //System.out.println("Evaluating: " + node.getFile().getAbsolutePath());
		  // Make sure no files.
		  File[] files = node.myDirectory.listFiles();
		  if (files == null || files.length == 0)
		  {
			  try
			  {
				  System.out.println("Removing:   " + node.getFile().getAbsolutePath());
				  node.getFile().delete();
				  node.parent.remove(node);
				  return true;
			  }
			  catch(Exception ex)
			  {
				  ex.printStackTrace();
			  }
		  }	  
	  }
	  return false;
  }
  

   // ==========================================================================
   // Object Stuff
   // ==========================================================================
   // For display purposes, we return our own myDisplayName
   @Override
	public String toString()
   {
      StringBuffer buff = new StringBuffer(100);
      buff.append(this.myDisplayName);
      if (this.myTree.getCountTargetFiles())
      {
      	buff.append(" (" + this.myNodeTargetFileCount + " / " + this.myNodeAllFileCount + ")"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      }
      else
      {
      	buff.append(" (" + this.myNodeAllFileCount + ")"); //$NON-NLS-1$ //$NON-NLS-2$
      }
      return buff.toString();
   } 
   
   // ==========================================================================
   //                   TEST
   // ==========================================================================
   /*
   public static void main(String[] args)
   {
   	Iterator loop = Charset.availableCharsets().values().iterator();
   	while(loop.hasNext())
   	{
   		System.out.println(loop.next());
   	}
   	
   	File dir = new File("/news/bad/umlaut"); //$NON-NLS-1$
		File[] files = dir.listFiles();
		if (files == null || files.length == 0 || clsCancelLoad) { return; }

		
	  	if (!files[0].exists())
   	{
			System.out.println("Failed: " + files[0]); //$NON-NLS-1$
			try
			{
				byte[] bytes = files[0].getAbsolutePath().getBytes("UTF-16"); //$NON-NLS-1$
				String fileName = new String(bytes, "UTF-16"); //$NON-NLS-1$
				File aFile = new File(fileName);
				System.out.println("aFile:  " + aFile); //$NON-NLS-1$
			  	if (aFile.exists())
			  	{
					System.out.println("Succeeded: " + aFile);			  		 //$NON-NLS-1$
			  	}
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
			//System.out.println("Scroller: " + );
   	}
		
	  	System.exit(0);	
	  	
	  	FileFilter ff = org.gerhardb.lib.image.ImageFactory.getImageFactory().getFilter();
		for (int i = 0; i < files.length; i++)
		{
			System.out.println(files[i]);
			if (ff.accept(files[i])) 
			{
				System.out.println("Accepted"); //$NON-NLS-1$
			}
			if (!files[i].isDirectory())
			{
				System.out.println("Is File"); //$NON-NLS-1$
				String name = files[i].getName();
				System.out.println("name: " + name); //$NON-NLS-1$
				name = name.toLowerCase();
				System.out.println("lowercase name: " + name); //$NON-NLS-1$
				if (name.endsWith(".jpg")) //$NON-NLS-1$
				{
				System.out.println("jpg"); //$NON-NLS-1$
				}
			}
		}			
	}
   */
}		
		/*
 			// Mystry slow problem.  Windows ONLY!!
			//System.out.println("-----> a: " + (new Date().getTime() - start.getTime()));
			//System.out.println(files[i]);
			if (files[i].isDirectory())
			{
				//System.out.println("-----> b: " + (new Date().getTime() - start.getTime()));
				// Ignore hidden directories using the linux convention.
				// Also, we are not going to show hidden directories, so don't count them.
				if (!files[i].isHidden())
				{
					rtnDirectories.add(files[i]);
					myNodeDirCount++;
				}
			}
			else
			{
				//System.out.println("-----> c: " + (new Date().getTime() - start.getTime()));
				myNodeAllFileCount++;
				myNodeAllFileSize = myNodeAllFileSize + files[i].length();
				if (countImageFiles)
				{
					//System.out.println("Filtering images");
					if (ImageFactory.getImageFactory().getFilter().accept(files[i]))
					{
						myNodeTargetFileCount++;
						myNodeTargetFileSize = myNodeTargetFileSize + files[i].length();
					}
				}
			}
		}   	
		
		// Count files and get Directories
		ArrayList dirs = countFiles();
		
		// Sort directories and add them to the list.
		Object[] dirArray = dirs.toArray();
		
		try
		{
			Arrays.sort(dirArray, new FileNameComparatorInsensative());
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}

		for (int i = 0; i < dirArray.length; i++)
		{
			try
			{
				DirectoryTreeNode node = new DirectoryTreeNode(myTree,
						(File) dirArray[i], loadingMsg);
				super.add(node);
				node.populate(loadingMsg);
		   }
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}
   	*/
