/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree;

import java.awt.*;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.dnd.DnDConstants;
import java.awt.dnd.DropTarget;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.dnd.DropTargetListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.FileFilter;
import java.util.List;

import javax.swing.*;
import javax.swing.tree.TreePath;

import org.gerhardb.lib.dirtree.rdp.RDPmanager;
import org.gerhardb.lib.scroller.Scroller;

/**
 * Derived from code originally presented in Core Swing Advanced Programming by
 * Kim Topley, Printice Hall PTR, chapter 8.
 * 
 * Key part of Drag and Drop.  See File Drag for details.
 */
public class ExtendedDirectoryTree extends DirectoryTreeRDP implements
		DropTargetListener, KeyListener
{
	ITreePopupMenu myPopupGetter;
	
	// --------------------------------------------------------------------------
	// Constructor
	// --------------------------------------------------------------------------
   
	public ExtendedDirectoryTree(Window window, FileFilter filter, RDPmanager rdpManager, Scroller scroller, ITreePopupMenu getter, boolean countHiddenDirectories) 
	{
		super(window, filter, rdpManager, scroller, countHiddenDirectories);
		this.myPopupGetter = getter;
		
		this.addMouseListener(new MouseAdpt(this));		
		this.addKeyListener(this);
		// Amazingly, you don't need to know about this or do anything with it.
		// You just have to create it or else dropping will not work.
		new DropTarget(this, this);		
	} 
	
	protected JPopupMenu getPopupMenu(TreePath path)
	{
		return this.myPopupGetter.getPopupMenu(path);
	}
	
	// --------------------------------------------------------------------------
	// DropTargetListener Implementation
	// --------------------------------------------------------------------------
	public void dragEnter(DropTargetDragEvent event)
	{
		// System.out.println("ExtendedFileTree.dragEnter");
		// page 748 core swing advanced programming
		/*
		 * if ( 0 == ( event.getSourceActions() &
		 * DnDConstants.ACTION_COPY_OR_MOVE ) ) { event.rejectDrag(); }
		 */
		// System.out.println("event.getDropAction(): " +
		// event.getDropAction());
		// System.out.println(
		// "DnDConstants.ACTION_COPY_OR_MOVE: " +
		// DnDConstants.ACTION_COPY_OR_MOVE);
		switch (event.getDropAction())
		{
		case DnDConstants.ACTION_COPY_OR_MOVE:
		case DnDConstants.ACTION_COPY:
		case DnDConstants.ACTION_MOVE:

			// System.out.println("ExtendedFileTree.dragEnter ACCEPTING");
			event.acceptDrag(DnDConstants.ACTION_COPY_OR_MOVE);
			break;
		default:

			// System.out.println("ExtendedFileTree.dragEnter REJECTING");
			event.rejectDrag();
		}
	}

	public void dragExit(DropTargetEvent dte)
	{
		// System.out.println( "ExtendedFileTree.dragExit" );
		super.clearSelection();
	}

	public void dragOver(DropTargetDragEvent dtde)
	{
		// System.out.println( "ExtendedFileTree.dragOver" );
		dtde.getLocation();
		Point dropPoint = dtde.getLocation();
		TreePath tp = getClosestPathForLocation(dropPoint.x, dropPoint.y);
		if (tp == null)
		{
			return;
		}
		Object objPicked = tp.getLastPathComponent();
		if (objPicked == null || !(objPicked instanceof DirectoryTreeNode))
		{
			return;
		}
		// DirectoryTreeNode dirNode = (DirectoryTreeNode)objPicked;
		super.setSelectionPath(tp);
	}

	public void dropActionChanged(DropTargetDragEvent dtde)
	{
		//System.out.println( "ExtendedFileTree.dropActionChanged" );
	}
	
	/**
	 * This is where drops go.
	 * Uses DataFlavor.javaFileListFlavor.
	 * That allows transfering a list of files to/from Java 
	 * (and the underlying platform).
	 * Must transfer a java.util.List is used.
	 * Each element of the list is required to be of type java.io.File.
	 */
	public void drop(DropTargetDropEvent event)
	{
		// DROP done HERE
		//System.out.println("ExtendedFileTree.drop");

		// COULD DO MORE INTERNAL STUFF BY CHECKING
		// DropTargetDropEvent.isLocalTransfer() to see if in same JVM
		// and then put a pointer to the actual object starting the drop
		// in for the data flavor. However, current methods seems to work
		// reasonably enough.
		
		// Check the data flavors.
		if (!event.isDataFlavorSupported(DataFlavor.javaFileListFlavor))
		{
			//System.out.println("ExtendedFileTree Rejecting Drop, Bad Flavor.");
			event.rejectDrop();
			return;
		}

		// Find out directory to move file to.
		Point dropPoint = event.getLocation();
		TreePath tp = getClosestPathForLocation(dropPoint.x, dropPoint.y);
		if (tp == null)
		{
			event.rejectDrop();
			return;
		}
		Object objPicked = tp.getLastPathComponent();
		if (objPicked == null || !(objPicked instanceof DirectoryTreeNode))
		{
			event.rejectDrop();
			return;
		}
		DirectoryTreeNode dirNode = (DirectoryTreeNode) objPicked;
		
		//System.out.println("ExtendedFileTree evaluating Drop.");
		boolean moveNotCopy = false;
		
		// Determine the type of drop.
		// This is the last item, so we can accept the drop.
		switch (event.getDropAction())
		{
		case DnDConstants.ACTION_MOVE:

			//System.out.println("ExtendedFileTree Accepting Drop: MOVE");
			event.acceptDrop(DnDConstants.ACTION_MOVE);
			moveNotCopy = true;
			break;
		case DnDConstants.ACTION_COPY:

			//System.out.println("ExtendedFileTree Accepting Drop: COPY");
			event.acceptDrop(DnDConstants.ACTION_COPY);
			moveNotCopy = false;
			break;
		default:

			//System.out.println("ExtendedFileTree REJECTING Drop!");
			event.rejectDrop();
			return;
		}
		
		//System.out.println("ExtendedFileTree: Dropped File List");

		Transferable transfer = event.getTransferable();
		Object moveObj = null;
		try
		{
			// For drops from windows to work, you must accept the drop first.
			moveObj = transfer.getTransferData(DataFlavor.javaFileListFlavor);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			event.dropComplete(false);
			return;
		}
		if (moveObj == null || !(moveObj instanceof List<?>))
		{
			event.dropComplete(false);
			return;
		}
		List<?> fileList = (List<?>)moveObj;

		this.myRDPmanager.addToRememberedMoveList(dirNode);
		
		super.myRDPmanager.moveOrCopySeveral(moveNotCopy, dirNode, fileList.toArray());
		
		//System.out.println("event.dropComplete(true)!");

		// Event is irrelevant off of AWT thread
		// Set to false ONLY to signal this is NOT a windows drop and prevent a reload
		// in FileDrag and SingleDrag.
		// Otherwise we would return true, but that is somewhat dangerous here since 
		// we don't really know if it worked...
		event.dropComplete(false);
	}

	// --------------------------------------------------------------------------
	// KeyListener
	// --------------------------------------------------------------------------
	 public void keyPressed(KeyEvent e)
	 {
		 consume(e);
	 }
	 
    public void keyReleased(KeyEvent e)
	 {
		 consume(e);
		 
	 }
	 
    public void keyTyped(KeyEvent e) 
	 {
		 consume(e);
	 }
	 	
    private void consume(KeyEvent event)
    {
 			switch (event.getKeyCode())
			{
			case KeyEvent.VK_A: 
				if (event.isControlDown())
				{
					 event.consume();
					 JList list = this.myRDPmanager.getIScrollDirTree().getFileList();
					 list.setSelectionInterval(0, list.getModel().getSize()-1);
				}
			}
    }
    
	// --------------------------------------------------------------------------
	// Private Classes
	// --------------------------------------------------------------------------
	
	void showStats(DirectoryTreeNode node, boolean countImageFilesInNodes)
	{
   	this.myRDPmanager.showStats(node, countImageFilesInNodes);
	}
	
	// --------------------------------------------------------------------------
	// Inner Class: MouseAdapter
	// --------------------------------------------------------------------------
	private class MouseAdpt extends MouseAdapter
	{
		ExtendedDirectoryTree	myTree	= null;

		TreePath			selPath	= null;

		public MouseAdpt(ExtendedDirectoryTree tree)
		{
			this.myTree = tree;
		}

		@Override
		public void mouseReleased(MouseEvent e)
		{
			int selRow = this.myTree.getRowForLocation(e.getX(), e.getY());
			if (selRow < 0)
			{
				return;
			}

			// We only set the tree on the first click since the model
			// may shift before the second click comes in.
			if (e.getClickCount() == 1)
			{
				this.selPath = this.myTree.getPathForLocation(e.getX(), e.getY());
			}

			// Single right click is a pop up.
			if (((e.getModifiers() & InputEvent.BUTTON2_MASK) != 0)
					|| ((e.getModifiers() & InputEvent.BUTTON3_MASK) != 0))
			{
				// We worry about where the click is because the selection will
				// not change as a result of picking the item.
				getPopupMenu(this.selPath).show((Component) e.getSource(), e.getX(), e.getY());
				return;
			}
			// Double left click is a pop up.
			else
				if (e.getClickCount() == 2)
				{
					/*
					 * // First left click selects the item. if
					 * (mySelectionModel.getSelectionCount() == 0) { return; }
					 * TreePath selPath =
					 * mySelectionModel.getLeadSelectionPath();
					 */
					Object end = this.selPath.getLastPathComponent();
					if (end == null || !(end instanceof DirectoryTreeNode))
					{
						return;
					}
					ExtendedDirectoryTree.this.myRDPmanager.getIScrollDirTree().setCurrentDirectory((DirectoryTreeNode) end, false);
				}
			super.mouseReleased(e);
		}
	}
} 
