/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree;

import java.awt.Window;
import java.io.File;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.BoundedRangeModel;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoableEdit;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.dirtree.rdp.RDPmanager;
import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.io.FileUtilDeleteException;

/**
 Designed to provide undo support for drags from the main picture
 to the file tree.
 */
public class UndoableMovingFiles implements UndoableEdit
{
   private ArrayList<File> myMovingFileList = new ArrayList<File>(500);
   private ArrayList<File> myFailedFileList = new ArrayList<File>(500);
   private DirectoryTreeNode myToDirNode; // will not be null 
   private String myToDirPath;
   private File myToDir;
   private RDPmanager myPlugins;
   private boolean iMoveNotCopy = true;
   private boolean iNeedToBeAdded = true;
   private boolean iAmReadyToUndo = false;
   private boolean MOVE = true;
   private boolean COPY = false;
   private boolean DOING = true;
   private boolean UNDOING = false;

   // --------------------------------------------------------------------------
   //                     Constructor
   // --------------------------------------------------------------------------


   /**
    * updateRepeatButton is so that when you do delete button or park button
    * it doesn't override last move...
    */
   public UndoableMovingFiles(
	   DirectoryTreeNode toDirNode,
	   RDPmanager plugins,
	   boolean updateRepeatButton)
   throws Exception   
   {
   	this.myToDirNode = toDirNode;
   	this.myPlugins = plugins;
   	if ( this.myToDirNode == null )
   	{
   		throw new Exception("UndoableMovingFiles requires toDirNode");   	 //$NON-NLS-1$
   	}

   	if ( this.myPlugins == null )
   	{
   		throw new Exception("UndoableMovingFiles requires IFrame"); //$NON-NLS-1$
   	}
   	
      // Set up a file to hold the drop directory.
   	this.myToDirPath = this.myToDirNode.getAbsolutePath();
      this.myToDir = new File(this.myToDirPath);
      if (this.myToDir.isFile())
      {
         throw new Exception("Can't move onto a file!"); //$NON-NLS-1$
      }
   	this.myToDirPath = this.myToDirPath + File.separator;
      if (updateRepeatButton)
      {
	      // Now that we know where to move the file, update the repeat button.
	      this.myPlugins.setRepeatDir(this.myToDirNode);
      }
   }
   
   public UndoableMovingFiles(
   	   File toFile,
   	   RDPmanager plugins)
      throws Exception   
      {
      	this.myPlugins = plugins;
      	if ( this.myPlugins == null )
      	{
      		throw new Exception("UndoableMovingFiles requires IFrame"); //$NON-NLS-1$
      	}
      	
         // Set up a file to hold the drop directory.
      	this.myToDirPath = toFile.getAbsolutePath();
         this.myToDir = new File(this.myToDirPath);
         if (this.myToDir.isFile())
         {
            throw new Exception("Can't move onto a file!"); //$NON-NLS-1$
         }
      	this.myToDirPath = this.myToDirPath + File.separator;
      }   
   
   public File[] getFailedFilesShowingMessage(final Window ss)
   {
   	File[] failedFiles = new File[this.myFailedFileList.size()];
   	this.myFailedFileList.toArray(failedFiles);
		if (failedFiles.length > 0)
		{
			SwingUtilities.invokeLater(new Runnable()
			{
				public void run()
				{
					// Return result BEFORE showing error message.
					JOptionPane.showMessageDialog(ss, AppStarter.getString("ExtendedFileTree.3") //$NON-NLS-1$
							+ FileUtil.NEWLINE + AppStarter.getString("ExtendedFileTree.4") //$NON-NLS-1$
							+ FileUtil.NEWLINE, AppStarter.getString("ExtendedFileTree.5"), //$NON-NLS-1$
							JOptionPane.ERROR_MESSAGE);
				}
			});
		}
		return failedFiles;
   }
      
   /**
    * Set to false if you want to do a copy.
    * @param move Default is true.
    */
   public void setMoveNotCopy(boolean move)
   {
   	this.iMoveNotCopy = move;
   }
   
   public void add(Object[] filesPicked, BoundedRangeModel range)
   throws Exception
   {
		for (int i = 0; i < filesPicked.length; i++)
		{
			addAFile((File)filesPicked[i], false);
			range.setValue(i);
			if (Thread.currentThread().isInterrupted())
			{
				System.out.println("THREAD WAS INTERRUPTED"); //$NON-NLS-1$
				break;
			}
		}   	
		// Do here once we know what the right count is.
		this.myPlugins.getIUndo().addUndoable(this); 
   }
   
   public void add(File file)
   throws Exception
   {
   	addAFile(file, true);
   }
   
   private void addAFile(File file, boolean addUndoable)
   throws Exception
   {
      if ( file == null )
      {
         throw new IllegalArgumentException("Must move a file - null received"); //$NON-NLS-1$
      }
      
      if ( !file.exists())
      {
         throw new IllegalArgumentException(AppStarter.getString("UndoableMovingFiles.0") + AppStarter.getString("colon") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        		 + file.getAbsolutePath()); //$NON-NLS-1$
      }
      
      if ( !file.isFile())
      {
         throw new IllegalArgumentException(AppStarter.getString("UndoableMovingFiles.3") + AppStarter.getString("colon") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        		 + file.getAbsolutePath()); //$NON-NLS-1$
      }
      
      try
      {
      	//System.out.println("UndoableMovingFiles moving: " + file);
      	moveIt(file, this.myToDir);
      
	      // If the move worked, add it to the list to undo.
	      this.myMovingFileList.add(file);
	
	      // Will only be done once for a group of files, 
	      // but add here so it will be done for a single file.
	      if (this.iNeedToBeAdded)
	      {
	      	this.iNeedToBeAdded = false;
	      	if (addUndoable)
	      	{
	      		this.myPlugins.getIUndo().addUndoable(this); 
	      	}
	      }
      }
      catch(Exception ex)
      {
      	this.myFailedFileList.add(file);
      }
   }

   
 
   // --------------------------------------------------------------------------
   // UndoableEdit
   // --------------------------------------------------------------------------
   public void undo()
      throws CannotUndoException
   {
      if (this.iAmReadyToUndo)
      {
      	File firstFile = null;
      	Exception foundException = null;
      	Iterator<File> loop = this.myMovingFileList.iterator();
      	while(loop.hasNext())
      	{
      		File aFile = loop.next();
      		if ( aFile != null )
      		{
               try
               {
            	  if (firstFile == null)
            	  {
            		  firstFile = aFile;
            	  }
                 unmoveIt(aFile);
                }
               catch (Exception ex)
               {
               	foundException = ex;
               	ex.printStackTrace();
               }      		
            }
      	}
 
         if ( foundException != null)
         {
            CannotUndoException newEx = new CannotUndoException();
            newEx.initCause(foundException);
            throw newEx;
         }
         
         if (this.myPlugins != null)
         {
         	// Just go back to last one...
         	// Best guess.
         	this.myPlugins.getIUndo().undid(firstFile);
         }
      }
    } 

   public void redo()
      throws CannotRedoException
   {
      if (!this.iAmReadyToUndo)
      {
      	File firstFile = null;
      	Exception foundException = null;
      	Iterator<File> loop = this.myMovingFileList.iterator();
      	while(loop.hasNext())
      	{
      		File aFile = loop.next();
      		if ( aFile != null )
      		{
               try
               {
         			if (firstFile == null)
         			{
         				firstFile = aFile;
         			}
        			 	moveIt(aFile, this.myToDir);
               }
               catch (Exception ex)
               {
               	foundException = ex;
               	ex.printStackTrace();
               }      		
            }
      	}
      	
         if (this.myPlugins != null)
         {
            this.myPlugins.getIUndo().redid();
         }
         
         if ( foundException != null)
         {
            CannotUndoException newEx = new CannotUndoException();
            newEx.initCause(foundException);
            throw newEx;
         }
      }
   } 

   public boolean canRedo()
   {
      return !this.iAmReadyToUndo;
   } 

   public boolean canUndo()
   {
      return this.iAmReadyToUndo;
   }

   public void die()
   {
		// We don't care
   } 

   public String getPresentationName()
   {
	  	 String retrived = AppStarter.getString("UndoableMovingFiles.move"); //$NON-NLS-1$
		 MessageFormat form = new MessageFormat(retrived);
		 Object[] args = {new Integer(this.myMovingFileList.size()), this.myToDir.toString()};
		 String msg = form.format(args);
	    return msg;    
   } 

   public String getRedoPresentationName()
   {
	  	 String retrived = AppStarter.getString("UndoableMovingFiles.redo"); //$NON-NLS-1$
		 MessageFormat form = new MessageFormat(retrived);
		 //System.out.println("getRedoPresentationName: " + myMovingFileList.size());
		 Object[] args = {new Integer(this.myMovingFileList.size()), this.myToDir.toString()};
		 String msg = form.format(args);
	    return msg;    
   } 

   public String getUndoPresentationName()
   {
	  	 String retrived = AppStarter.getString("UndoableMovingFiles.undo"); //$NON-NLS-1$
		 MessageFormat form = new MessageFormat(retrived);
		 //System.out.println("getUndoPresentationName: " + myMovingFileList.size());
		 Object[] args = {new Integer(this.myMovingFileList.size()), this.myToDir.toString()};
		 String msg = form.format(args);
	    return msg;    
   }

   public boolean isSignificant()
   {
      return true;
   } 

   public boolean addEdit(UndoableEdit anEdit)
   {
      return false;
   } 

   public boolean replaceEdit(UndoableEdit anEdit)
   {
      return false;
   } 

   // --------------------------------------------------------------------------
   //                     Private 
   // -------------------------------------------------------------------------- 
   
   private void moveIt(File from, File toDir)
      throws Exception
   {

   	/*
   	new Throwable().printStackTrace();
           System.out.println( "\n\n" );
      System.out.println( "**************************************************" );
           System.out.println( "MoveIt From: " + from );
           System.out.println( " To: " + toDir );
       System.out.println( "**************************************************" );
           System.out.println( "\n\n" );
       */
   	
      File bugFixFile = new File(from.getCanonicalPath());
      if (this.iMoveNotCopy)
      {
         try
         {
            FileUtil.moveFile(new File(this.myToDirPath), bugFixFile);
            updateCountForOneFile(from, this.DOING, this.MOVE); 
            this.iAmReadyToUndo = true;
         }
         catch(FileUtilDeleteException ex)
         {
         	// Recount should occur on both in natural order of things.
            throw ex;
         }
      }
      else
      {
         // Make the copy.
         FileUtil.copyFileToDir(toDir, bugFixFile);
         updateCountForOneFile(from, this.DOING, this.COPY); 
        this.iAmReadyToUndo = true;
       }
   }

   private void unmoveIt(File originalLocation)
   throws Exception
	{
      String moveBackName =
         this.myToDir.getAbsolutePath() + File.separator + originalLocation.getName();
      File moveBackFile = new File(moveBackName);      
	   if (this.iMoveNotCopy)
	   {
	      // This was orginally a move.
	      try
	      {
	         FileUtil.moveFile(originalLocation.getParentFile(), moveBackFile);
	         updateCountForOneFile(originalLocation, this.UNDOING, this.MOVE); 
	         this.iAmReadyToUndo = false;
	      }
	      catch(FileUtilDeleteException ex)
	      {
	      	// Recount should occur on both in natural order of things.
	      }
	   }
	   else
	   {
	      // This was orginally a copy.
         // Undoing a copy means deleting the copy.
         String delName =
         	this.myToDir.getAbsolutePath() + File.separator + moveBackFile.getName();
         File delFile = new File(delName);
         try
         {
         	delFile.delete();
	         updateCountForOneFile(originalLocation, this.UNDOING, this.COPY); 
         }
         catch(Exception ex)
         {
         	ex.printStackTrace();
         }
         this.iAmReadyToUndo = false;
	   }
 	}
   
   /**
    * Make sure we have correct count.
    * @param file
    */
   private void updateCountForOneFile(File file, boolean doing, boolean move)
   {
   	//System.out.println("Undoable Recounting Foghorn");
   	// TO directory - always changes for move and copy.
   	if (this.myToDirNode != null)
   	{
   		if (doing)
   		{
   			this.myToDirNode.incrementFileCount(file.length()); 
   		}
   		else
   		{
   			this.myToDirNode.decrementFileCount(file.length()); 
   		}   			
   	}
   	
   	// FROM directory only changes on moves.
      if (move && file instanceof DTNFile)
      { 	
      	DirectoryTreeNode fromNode = ((DTNFile)file).getDirectoryTreeNode();      
	      if (fromNode != null)
	      {

	   		if (doing)
	   		{
	   			fromNode.decrementFileCount(file.length()); 
	   		}
	   		else
	   		{
	   			fromNode.incrementFileCount(file.length()); 
	   		}   		      
	      }
      }
   }
   
   @Override
	public boolean equals(Object obj)
   {
      //System.out.println("UNDOABLE MOVING FILE EQUALS CALLED!!!");
      //System.out.println(this +"   " + obj);
      //System.out.println(obj.getClass().getName());
      if (obj == this)
      {
         return true;
      }
      return false;
   } 
   
   @Override
	public int hashCode()
   {
   	return this.myToDirNode.getName().hashCode();
   }

   // --------------------------------------------------------------------------
   //                                    Main
   // --------------------------------------------------------------------------
/*
   public static void main(String[] args)
   {
   	ILoadingMessage fakeLoading = new ILoadingMessage()
   	{
   	   private int myIncrement = 1;
   	   public void setText(String msg){}
   	   public void setMessage(String msg){}
   	   public int getNextIncrement(){return myIncrement++;}
   	};
		//final String fromPath = "E:/testpics/proof/move/from/"; //$NON-NLS-1$
		//final String toPath = "E:/testpics/proof/move/to/"; //$NON-NLS-1$
		final String fromPath = "/mnt/win_e/testpics/proof/move/from/"; //$NON-NLS-1$
		final String toPath = "/mnt/win_e/testpics/proof/move/to/"; //$NON-NLS-1$
		final File fromDir = new File(fromPath);
		final File toDir = new File(toPath);
		File fromFiles[] = new File[40];
		File toFiles[] = new File[40];
		File fromFilesShort[] = new File[1];
		File toFilesShort[] = new File[1];
		for(int i=1;i<10; i++)
		{
			fromFiles[i-1] = new File(fromPath + "hapc10" + i + ".jpg"); //$NON-NLS-1$ //$NON-NLS-2$
			toFiles[i-1] = new File(toPath + "hapc10" + i + ".jpg"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		for(int i=10;i<41; i++)
		{
			fromFiles[i-1] = new File(fromPath + "hapc1" + i + ".jpg"); //$NON-NLS-1$ //$NON-NLS-2$
			toFiles[i-1] = new File(toPath + "hapc1" + i + ".jpg"); //$NON-NLS-1$ //$NON-NLS-2$
		}
		fromFilesShort[0] = new File(fromPath + "hapc101.jpg"); //$NON-NLS-1$
		toFilesShort[0] = new File(toPath + "hapc101.jpg"); //$NON-NLS-1$
		

   	final IFrame fakeFrame = new LoadingShow();
   	final DirectoryTree fakeTree = new DirectoryTree(null, ImageFactory.getImageFactory().getFilter(), null);
		ViewerPreferences.imprintImageFileCountingOptions(fakeTree);
   	final BoundedRangeModel fakeRange = new DefaultBoundedRangeModel();
		
		try
		{
		  	final DirectoryTreeNode fromDirNode = 
	   		new DirectoryTreeNode(fakeTree, fromDir, fakeLoading);
		  	final DirectoryTreeNode toDirNode = 
	   		new DirectoryTreeNode(fakeTree, toDir, fakeLoading);
	   	
		  	long startTime = System.currentTimeMillis();
			for(int i=0;i<fromFilesShort.length; i++)
			{
				FileUtil.moveFile(toPath, fromFilesShort[i]); 
			}
			for(int i=0;i<toFilesShort.length; i++)
			{
				FileUtil.moveFile(fromPath, toFilesShort[i]); 
			}
			
			long stopTime = System.currentTimeMillis();
			long basicTime = stopTime - startTime;
			System.out.println("Basic: " + basicTime); //$NON-NLS-1$
			startTime = System.currentTimeMillis();
			
			UndoableMovingFiles undoable = 
				new UndoableMovingFiles(toDirNode, fakeFrame,  false);
			undoable.add(fromFilesShort, fakeRange);
			
			undoable = new UndoableMovingFiles(fromDirNode, fakeFrame,  false);
			undoable.add(toFilesShort, fakeRange);
			
			stopTime = System.currentTimeMillis();
			long undoTime = stopTime - startTime;
			System.out.println("Undoable: " + undoTime); //$NON-NLS-1$
			//float ratio = undoTime/basicTime;
			//System.out.println("ratio: " + ratio);
			
			
			
			
		  	startTime = System.currentTimeMillis();
			for(int i=0;i<fromFiles.length; i++)
			{
				FileUtil.moveFile(toPath, fromFiles[i]); 
			}
			for(int i=0;i<toFiles.length; i++)
			{
				FileUtil.moveFile(fromPath, toFiles[i]); 
			}
			
			stopTime = System.currentTimeMillis();
			basicTime = stopTime - startTime;
			System.out.println("Basic: " + basicTime); //$NON-NLS-1$
			startTime = System.currentTimeMillis();
			
			undoable = 
				new UndoableMovingFiles(toDirNode, fakeFrame,  false);
			undoable.add(fromFiles, fakeRange);
			
			undoable = new UndoableMovingFiles(fromDirNode, fakeFrame,  false);
			undoable.add(toFiles, fakeRange);
			
			stopTime = System.currentTimeMillis();
			undoTime = stopTime - startTime;
			System.out.println("Undoable: " + undoTime); //$NON-NLS-1$
			//float ratio = undoTime/basicTime;
			//System.out.println("ratio: " + ratio);
			
			
			
			
		}
		catch (Exception ex)
		{
			System.out.println("Move ERROR: " + ex.getMessage()); //$NON-NLS-1$
		}
   } 
   */
}
