/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.gerhardb.lib.dirtree.filelist.popup;

import java.awt.BorderLayout;
import java.awt.EventQueue;
import java.io.File;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.gerhardb.lib.dirtree.DirectoryTree;
import org.gerhardb.lib.dirtree.filelist.FileListPlugins;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 *
 * @author  Gerhard Beck
 */
class FileSeveralNameChangeDialog extends FileNameChangeDialog
{
	private static final int ADD_NUMBER = 1;
	private static final int COMPLETE = 2;

	FileNameChangeCompleteTab myCompleteTab;
	FileNameChangeAddNumberTab myNumberOnlyTab;
	String mySampleText = "abcdefghijklmnopqrstuvwxyz.123"; //$NON-NLS-1$;
	private boolean iRenamed = false;

	//===========================================================================
	//                              Constructor
	//===========================================================================
	public FileSeveralNameChangeDialog(FileNameChanger fnc, FileListPlugins plugins, DirectoryTree tree,
			boolean allTabs, boolean showRenameTabAtTop)
	{
		super(fnc, plugins, tree, allTabs); 
		
		if (allTabs && showRenameTabAtTop)
		{
			this.myTabs.setSelectedComponent(this.myCompleteTab);
		}		
		// setVisible(true) must be after tab selection
		this.setVisible(true);
	}
	
	boolean renamed()
	{
		return this.iRenamed;
	}

	@Override
	void layoutDialog()
	{
		// Set up list of files to process and pick the sample name.
		long testLength = 1000;
		for (int i = 0; i < this.myFileNameChanger.myInputFileList.length; i++)
		{
			if (this.myFileNameChanger.myInputFileList[i] != null)
			{
				String name = this.myFileNameChanger.myInputFileList[i].getName();
				if (name != null && name.length() < testLength)
				{
					this.mySampleText = this.myFileNameChanger.myInputFileList[i].getName();
					testLength = this.mySampleText.length();
				}
			}
		}
		
		this.myCompleteTab = new FileNameChangeCompleteTab(this, this.mySampleText,
				this.myFileNameChanger.myInputFileList.length, this.myPlugins.getRDPmanager().getAppIDandTargetType());
		this.myNumberOnlyTab = new FileNameChangeAddNumberTab(this, this.mySampleText);

		this.myTabs.addChangeListener(new ChangeListener()
		{
			public void stateChanged(ChangeEvent e)
			{
				EventQueue.invokeLater(new Runnable()
				{
					public void run()
					{
						if (FileSeveralNameChangeDialog.this.iShowAll)
						{
							switch (FileSeveralNameChangeDialog.this.myTabs.getSelectedIndex())
							{
							case MOVE_FILES:
								FileSeveralNameChangeDialog.this.getRootPane().setDefaultButton(
										FileSeveralNameChangeDialog.this.myMoveTab.getDefaultButton());
								break;
							case COMPLETE:
								FileSeveralNameChangeDialog.this.getRootPane().setDefaultButton(
										FileSeveralNameChangeDialog.this.myCompleteTab.getDefaultButton());
								FileSeveralNameChangeDialog.this.myCompleteTab.myTextFld.requestFocus();
								break;
							case ADD_NUMBER:
								FileSeveralNameChangeDialog.this.getRootPane().setDefaultButton(
										FileSeveralNameChangeDialog.this.myNumberOnlyTab.getDefaultButton());
								FileSeveralNameChangeDialog.this.myNumberOnlyTab.myTextFld.requestFocus();
								break;
							}
						}
						else
						{
							FileSeveralNameChangeDialog.this.myCompleteTab.myTextFld.requestFocus();
						}
					}
				});
			}
		});

		Icon icon = null;
		if (this.iShowAll)
		{
			this.myTabs.addTab(AppStarter.getString("FileNameChangeDialog.6"), icon, this.myMoveTab, AppStarter.getString("FileNameChangeDialog.3")); //$NON-NLS-1$ //$NON-NLS-2$
			this.myTabs.addTab(AppStarter.getString("FileNameChangeDialog.4"), icon, this.myNumberOnlyTab, AppStarter.getString("FileNameChangeDialog.3")); //$NON-NLS-1$ //$NON-NLS-2$
		}
		this.myTabs.addTab(AppStarter.getString("FileNameChangeDialog.2"), icon, this.myCompleteTab, AppStarter.getString("FileNameChangeDialog.1")); //$NON-NLS-1$ //$NON-NLS-2$

		JPanel thePanel = new JPanel(new BorderLayout());
		thePanel.add(this.myTabs, BorderLayout.CENTER);
		//thePanel.add(btnPanel, BorderLayout.SOUTH);

		finishLayout(thePanel);
		
		this.myNumberOnlyTab.setSampleText(this.mySampleText);
	}
	
	void doNameChange()
	{
		this.iRenamed = true;
		File[] inputFileList = this.myFileNameChanger.myOutputFileList;
		File[] outputFileList = new File[inputFileList.length];
		String failureMsg = AppStarter.getString("FileList.10");
		this.myFileNameChanger.iRan = true;
		boolean failures = false;
		setWaitCursor(true);
		try
		{
			int position = 0; // used for null names - keep in sequence over null names.
			
			// First pass we will get the new names and see if there are any
			// duplicates.
			for (int i = 0; i < inputFileList.length; i++)
			{
				if (inputFileList[i] != null)
				{
					//System.out.println("Looking at: " + inputFileList[i]);
					try
					{
						String dir = inputFileList[i].getParentFile().getCanonicalPath();
						String newName = dir + "/" //$NON-NLS-1$
								+ getNewName(position + 1, inputFileList[i].getName());
						position++;
						outputFileList[i] = new File(newName);
						//System.out.println("Renamed to: " + outputFileList[i]);
						if (outputFileList[i].exists())
						{
							failures = true;
							failureMsg =   "There is already a file named:\n" + outputFileList[i];
							break;
						}
					}
					catch (Exception ex)
					{
						inputFileList[i] = this.myFileNameChanger.myInputFileList[i];
						System.out.println("FileNameChangeDialog A: " + ex.getMessage()); //$NON-NLS-1$
						failures = true;
					}
				}
			}
			
			// Now we do the actual rename.
			if (!failures)
			{
				for (int i = 0; i < inputFileList.length; i++)
				{
					if (inputFileList[i] != null)
					{
						try
						{
							System.out.println("Renaming: " + inputFileList[i]);
							inputFileList[i].renameTo(outputFileList[i]);
							inputFileList[i] = outputFileList[i];
						}
						catch (Exception ex)
						{
							inputFileList[i] = this.myFileNameChanger.myInputFileList[i];
							System.out.println("FileNameChangeDialog B: " + ex.getMessage()); //$NON-NLS-1$
							failures = true;
						}
					}
				}
			}
		}
		finally
		{
			setWaitCursor(false);
		}

		if (failures)
		{
			JOptionPane.showMessageDialog(this, failureMsg, 
					AppStarter.getString("FileList.11"), //$NON-NLS-1$
					JOptionPane.WARNING_MESSAGE);
		}
	}

	/**
	 * @param i
	 * @param oldName Just the base name, not the path.
	 * @return
	 */
	private String getNewName(int i, String oldName)
	{
		if (this.iShowAll)
		{
			switch (this.myTabs.getSelectedIndex())
			{
			case COMPLETE:
				//myCompleteTab.setupTemplate(); 
				return this.myCompleteTab.getNewName(i, oldName, false);
			case ADD_NUMBER:
				return this.myNumberOnlyTab.getNewName(oldName);
			default:
				return oldName;
			}
		}
		return this.myCompleteTab.getNewName(i, oldName, false);
	}

	/**
	 * Tests this class.
	 */
	/*
	public static void main(String[] args)
	{
		File[] files = new File[11];
		files[0] = new File("a.jpg");
		files[1] = new File("ab.jpg");
		files[2] = new File("abc.jpg");
		files[3] = new File("abcd.jpg");
		files[4] = new File("abcde.jpg");
		files[5] = new File("abcdef.jpg");
		files[6] = new File("frog.jpg");
		files[7] = new File("x.jpg");
		files[8] = new File("xx.jpg");
		files[9] = new File("xxx.jpg");
		files[10] = new File("xxxx.jpg");
		
		FileNameChangeDialog fncd =
		     new FileNameChangeDialog(files, null); //$NON-NLS-1$
		
		for (int i=0; i<files.length; i++)
		{
		  System.out.println( fncd.getNewName(0, files[i].getName())); //$NON-NLS-1$
		}
	}
		*/
}
