/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.lib.dirtree.rdp;

import java.awt.Cursor;
import java.awt.EventQueue;
import java.io.File;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.undo.UndoableEdit;

import org.gerhardb.jibs.diskUsage.ExplorerBox;
import org.gerhardb.lib.dirtree.DTNReaderWriter;
import org.gerhardb.lib.dirtree.DirectoryTreeNode;
import org.gerhardb.lib.dirtree.ExtendedDirectoryTree;
import org.gerhardb.lib.dirtree.IScrollDirTree;
import org.gerhardb.lib.dirtree.ITreePopupMenu;
import org.gerhardb.lib.dirtree.IUndo;
import org.gerhardb.lib.dirtree.filelist.FileList;
import org.gerhardb.lib.dirtree.filelist.FileListManager;
import org.gerhardb.lib.dirtree.filelist.FileListPlugins;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.scroller.ScrollerListener;
import org.gerhardb.lib.util.Range;
import org.gerhardb.lib.util.startup.ActiveActions;
import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.startup.Loading;

/**
 * 
 */
public class ListShowTreeCoordinator implements RDPplugins, IUndo, IScrollDirTree, TreeManagerSet, FileListPlugins
{
	public static String JIBS_DIR_NAME = "/JIBS/textpad"; //$NON-NLS-1$

	private Scroller myScroller = new Scroller();
	private ExtendedDirectoryTree myTree;
	private FileList myFileList;
	
	private Preferences myPrefs;
	IListShowTree myLSTplugins;
	PathManager myPathManager;
	TreeManagerRDP myTreeManager;
	RDPmanager myRDPmanager;
	FileListManager myFileListManager;
	StatusBarManager myStatusBarManager;
	SortManager mySortManager = new SortManager(this);
	BasicOptionsManager myBasicOptionsManager = new BasicOptionsManager(this);
	ActiveActions myActions;
	
	// ==========================================================================
	// Constructor
	// ==========================================================================	
	public ListShowTreeCoordinator(IListShowTree lst)
	{
		this.myLSTplugins = lst;
		this.myPrefs = Preferences.userRoot().node("/" + this.myLSTplugins.getAppIDandTargetType() + "/org/gerhardb/lib/dirtree/rdp/ListShowTreeCoordinator");
	}
	
	/**
	 * Give descendants a chance to set themselves up before making calls back.
	 */
	public void init(ITreePopupMenu getter)
	{				
		this.myPathManager = new PathManager(this, this.myLSTplugins.getAppIDandTargetType());

		this.myRDPmanager = new RDPmanager(
				this, 
				this, 
				this,
				this.myLSTplugins.getAppIDandTargetType());
				
		this.myFileListManager = new FileListManager(this, this.myLSTplugins.getFilenameFileFilter());
		
		
		boolean countHiddenDirectories = false;
		this.myTree = new ExtendedDirectoryTree(
				this.myLSTplugins.getTopFrame(), 
				this.myLSTplugins.getFilenameFileFilter(), 
				this.myRDPmanager, 
				this.myScroller, getter, countHiddenDirectories);
		
		this.myTreeManager = new TreeManagerRDP(this); 
		
		this.myTree.setCountTargetFiles(this.myLSTplugins.getCountTargetFiles());
		this.myFileList = new FileList(this, this.myTree);
		this.myFileList.setModel(this.myScroller);
		
		
		this.myScroller.setFrame(this.myLSTplugins.getTopFrame());
		this.myScroller.getScrollerKeyListener().setKeypadOps(this.myRDPmanager);
		this.myScroller.sort(this.mySortManager.getSortOrder()); 
		
		// Some apps set this manually and change mid stream.
		ScrollerListener listener = this.myLSTplugins.getScrollerListener();
		if (listener != null)
		{
			this.myScroller.addScrollerListener(listener); 
		}
		
		this.myStatusBarManager = new StatusBarManager(this);
		
		this.myBasicOptionsManager.init();
		
		// OK, let's roll!
		this.myScroller.requestFocus();
	}

	public void addActions(ActiveActions aa)
	{
		this.myActions = aa;
		this.myRDPmanager.addActions(this.myActions); 
		this.myPathManager.init();
	}
	
	// ==========================================================================
	//                           Basic Access
	// ==========================================================================	 
	public TreeManagerRDP getTreeManager()
	{
		return this.myTreeManager;
	}
	
	public FileListManager getFileListManager()
	{
		return this.myFileListManager;
	}
	
	public ExtendedDirectoryTree getTree()
	{
		return this.myTree;
	}
	
	public StatusBarManager getStatusBarManager()
	{
		return this.myStatusBarManager;
	}
	
	public SortManager getSortManager()
	{
		return this.mySortManager;
	}
	
	public ActiveActions getActions()
	{
		return this.myActions;
	}
	
	public BasicOptionsManager getBasicOptionsManager()
	{
		return this.myBasicOptionsManager;
	}
	
	public void storeDirectories()
	{
		if (this.myBasicOptionsManager.getRememberDirectories())
		{
			try
			{
				getTreeManager().store();
			}
			catch (Exception ex)
			{
				JOptionPane.showMessageDialog(getTopFrame(),
						AppStarter.getString("SortScreen.49") + ex.getMessage(), //$NON-NLS-1$
						AppStarter.getString("SortScreen.50"), JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
				ex.printStackTrace();
			}
		}
		flush();	
	}
	
	public void flush()
	{
		try
		{
			this.myPrefs.flush();
		}
		catch (BackingStoreException ex)
      {
      	ex.printStackTrace();
      }
	}
	
	// ==========================================================================
	//                           TreeManagerSet
	// ==========================================================================

	public Scroller getScroller(){return this.myScroller;}
	public RDPplugins getRDPplugins(){return this;}
	//public PathManager getPathManager(); // duplicate...
	public MoveManager getMoveManager(){return this.myRDPmanager.getMoveManager();}
		
	// ==========================================================================
	//                           FileListPlugins
	// ==========================================================================
	//public void setWaitCursor(boolean wait); // duplicate...
	//public void reloadScroller(int scrollBackTo); // duplicate...
	//public Scroller getScroller(); // duplicate...
	public Range showingIndexes()
	{
		int index = this.myScroller.getValueZeroBased();
		//System.out.println("Showing index: " + index);
		return new Range(index, index);
	}

	public RDPmanager getRDPmanager()
	{
		return this.myRDPmanager;
	}
	
	// ==========================================================================
	//                           IScrollDirTree
	// ==========================================================================
	public FileList getFileList()
	{
		return this.myFileList;
	} 
	
	public void setCurrentDirectory(DirectoryTreeNode[] nodes, boolean nonTarget)
	{
		this.myFileListManager.setTargetDir(nodes, nonTarget);
	}

	public void setCurrentDirectory(DirectoryTreeNode node, boolean nonTarget)
	{
		this.myFileListManager.setTargetDir(node, nonTarget);
	}
	
	public void selectFiles(File[] filesToSelect)
	{
		this.myFileList.selectFiles(filesToSelect);
	}

	public void fileListUpdated()
	{
		// n/a for this applicaiton
	}
	
	// ==========================================================================
	//                           RDPplugins
	// ==========================================================================
	public JFrame getTopFrame(){return this.myLSTplugins.getTopFrame();} 
	public ExtendedDirectoryTree getExtendedDirectoryTree(){return this.myTree;} 
	public PathManager getPathManager(){return this.myPathManager;} 
	
	public String getDirectoryForParkAndDelete()
	{
		return AppStarter.getMyDocumentsDir().getAbsoluteFile() + JIBS_DIR_NAME;
	} 
	
	public String getPreference(String lookup, String aDefault)
	{
		return this.myPrefs.get(lookup, null);
	}
	
	public Preferences getPreferences()
	{
		return this.myPrefs;
	}
	
	public File getCurrentFile()
	{
		return this.myScroller.getCurrentFile();
	}
	
	public void setWaitCursor(boolean wait)
	{
		if (wait)
		{
			this.myLSTplugins.getTopFrame().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		}
		else
		{
			this.myLSTplugins.getTopFrame().setCursor(Cursor.getDefaultCursor());
		}
	}

	public void setCurrentDirectory(String dirName)
	{
		this.myFileListManager.setTargetDir(dirName);
	} 
	
	public void removeCurrentFile()
	{
		this.myScroller.removeCurrentPage();
	}
	
	public void reloadScroller(int scrollBackTo, boolean reloadScroller)
	{
		this.myScroller.reloadScroller(scrollBackTo, reloadScroller);
	}
	

	public void reloadScroller()
	{
		this.myScroller.reloadScroller();
	}
	
	public void loadParkAndDeleteNodes()
	{
		this.myRDPmanager.loadParkAndDeleteNodes();
	}
	
	public void setTreeDirectory(String dirName)
	{
		this.myTreeManager.setDirectory(dirName);
	}
	
	public void removePreference(String lookup)
	{
		this.myPrefs.remove(lookup);		
	}
	
	public void showStats(DirectoryTreeNode node, boolean countImageFilesInNodes)
	{
		if (node == null)
		{
			JOptionPane.showMessageDialog(this.myLSTplugins.getTopFrame(),
					AppStarter.getString("Popups.5"), AppStarter.getString("Popups.6"), //$NON-NLS-1$ //$NON-NLS-2$
					JOptionPane.INFORMATION_MESSAGE);
			return;
		}

		//targetType, countTargetFiles, StartingJibsFile
		DTNReaderWriter dtn = new DTNReaderWriter(this.myLSTplugins.getAppIDandTargetType(), countImageFilesInNodes, PathManager.getStartingJibsFile());
		// standAlone, label, IScroll, dtn
		boolean countHiddenDirectories = false;
		ExplorerBox explorer = new ExplorerBox(false, false, this.myScroller, dtn, countHiddenDirectories, false);
		explorer.setRootNode(node);
		/*
		NumberFormat intFormat = NumberFormat.getIntegerInstance();
		
		DirectoryTreeNode.TreeTotals treeTotals = node.getTreeTotals();
		StringBuffer buff = new StringBuffer(200);
		buff.append(intFormat.format(treeTotals.aggregateTotalFileCount));
		buff.append(" total files");
		if (ViewerPreferences.countImageFilesInNodes())
		{
			buff.append(" (");
			buff.append(intFormat.format(treeTotals.aggregateImageFileCount));
			buff.append(" image files)");
		}
		buff.append(" in ");
		int dirCount = treeTotals.aggregateDirectoryCount;
		// Leaves don't count themselves in the directory count to make rollups work.
		// So we have to do it for them.
		// Basically adding one for the root.
		dirCount++;
		buff.append(intFormat.format(dirCount));
		buff.append(" directories");
		
		JOptionPane.showMessageDialog(
				myPlugins.getTopFrame(), 
				buff.toString(), 
				node.getName(),  
				JOptionPane.INFORMATION_MESSAGE);
				*/   	
	}
		
	// We don't do this, so this is just hard coded.
	public boolean isFullScreen(){return false;}
	
	// ==========================================================================
	//                           IUndo
	// ==========================================================================
	
	public void addUndoable(UndoableEdit edit){this.myRDPmanager.getMoveManager().addUndoable(edit);}
	public void redid(){	this.myScroller.reloadScroller(); } 	
	public void undid(File file)
	{ 
		// from: myShow.jumpTo(file); 
		this.myScroller.reloadScroller();
		this.myScroller.jumpTo(file);
		showPageFromScroller();
	} 	
 	
	// ==========================================================================
	//                           Other
	// ==========================================================================	 
	public void showPageFromScroller()
	{
		showFile();
		// May want to add status later...
      //myViewerFrame.statusCurrentPage();
      if (this.myFileList.isVisible())
      {
      	this.myFileList.sliderAdjusted();
      }		
	} 	

	// equal to showImage() in Viewer.
	public void showFile()
	{
		this.myLSTplugins.showFile(this.myScroller.getCurrentFile());
	}
	
	// ==========================================================================
	//                           Initial Loading
	// ==========================================================================
	
	public void load(final File loadFile, final Loading loading, final ILoad load, final Thread loadingThread)
	{
		// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		// WANT TO START THIS AT THE SAME TIME AS THIS PARTICULAR THREAD!!!
		// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
		// Let's start loading this ASAP. ///////////////////////////////////////////////////////////////
		// I'm not sure this was separately threaded for quite a while!!!
		Runnable runMe = new Runnable()
		{
			public void run()
			{
				getTreeManager().showViewTree(
						loadFile, 
						loading);
				getRDPmanager().loadParkAndDeleteNodes();
				try
				{
					loadingThread.join();									
				}
				catch(InterruptedException ex){ex.printStackTrace();}
				
				EventQueue.invokeLater(new Runnable()
				{
					public void run()
					{
						load.awtComplete();
					}
				});
			}
		};
		new Thread(runMe).start();	
	}
	
}