/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree.rdp;


import java.awt.Cursor;
import java.io.File;

import javax.swing.*;
import javax.swing.undo.UndoManager;
import javax.swing.undo.UndoableEdit;

import org.gerhardb.lib.dirtree.DirectoryTreeNode;
import org.gerhardb.lib.dirtree.UndoableMovingFiles;
import org.gerhardb.lib.dirtree.filelist.FileList;
import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.scroller.IScroll;
import org.gerhardb.lib.scroller.Scroller;
import org.gerhardb.lib.util.ModalProgressDialogFast;
import org.gerhardb.lib.util.startup.ActiveActions;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * Manages Moves
 */
public class MoveManager
{
	static final int MOVE_NONE = 0;
	static final int MOVE_ONE = 1;
	static final int MOVE_MANY = 2;
	private UndoManager myUndoManager = new UndoManager(); 
	private JButton myUndoBtn;  
	private JButton myRedoBtn;	
	private int myResetRecommendation = -1;
	
	RDPmanager myRDPmanager;
	
	// ==========================================================================
	// Constructor
	// ==========================================================================
	public MoveManager(RDPmanager manager)
	{
		this.myRDPmanager = manager;
	}
	
	public void addActions(ActiveActions aa)
	{
		this.myUndoBtn = aa.getToolBarButton("edit", "undo"); //$NON-NLS-1$ //$NON-NLS-2$
		this.myRedoBtn = aa.getToolBarButton("edit", "redo"); //$NON-NLS-1$ //$NON-NLS-2$		
	}
	
	public UndoManager getUndoManager()
	{
		return this.myUndoManager;
	}
	
	public JButton getUndoButton()
	{
		return this.myUndoBtn;
	}
	
	public JButton getRedoButton()
	{
		return this.myRedoBtn;
	}
	
	// ==========================================================================
	// Move Stuff
	// ==========================================================================

	void moveCurrentFile(DirectoryTreeNode toNode)
	{
		try
		{
			UndoableMovingFiles undo = 
				new UndoableMovingFiles(toNode, this.myRDPmanager, false);	
			moveCurrentFile(undo);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this.myRDPmanager.myPlugins.getTopFrame(),
					AppStarter.getString("Single.0") + ex.getMessage(), //$NON-NLS-1$
					AppStarter.getString("Single.1"), JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
		}		
	}
	
	void moveCurrentFile(File toDir)
	{
		try
		{
			UndoableMovingFiles undo = new UndoableMovingFiles(toDir, this.myRDPmanager);
			moveCurrentFile(undo);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this.myRDPmanager.myPlugins.getTopFrame(),
					AppStarter.getString("Single.0") + ex.getMessage(), //$NON-NLS-1$
					AppStarter.getString("Single.1"), JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
		}
	}
	
	/**
	 * Undo handle updating file counts for us.
	 * @param undo
	 */
	void moveCurrentFile(UndoableMovingFiles undo)
	{
		File moveThis = this.myRDPmanager.myPlugins.getCurrentFile();
		this.myRDPmanager.setWaitCursor(true);
		try
		{		
			undo.add(moveThis);
			
			File[] failedFiles = undo.getFailedFilesShowingMessage(this.myRDPmanager.myPlugins.getTopFrame());
			if (failedFiles.length == 0)
			{
				updateResetRecommendation();
				
				// Get rid of the old current item from the scroller,
				// if the move worked.
				this.myRDPmanager.myPlugins.removeCurrentFile();
			}
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this.myRDPmanager.myPlugins.getTopFrame(),
					AppStarter.getString("Single.0") + ex.getMessage(), //$NON-NLS-1$
					AppStarter.getString("Single.1"), JOptionPane.ERROR_MESSAGE); //$NON-NLS-1$
		}
		finally
		{
			this.myRDPmanager.setWaitCursor(false);
		}	
	}
	
	/**
	 * Pass in EITHER toNode or toDir.
	 * @param moveNotCopy
	 * @param node
	 * @param dir
	 * @param filesPicked
	 */
	void moveOrCopySeveral(boolean moveNotCopy, 
			DirectoryTreeNode toNode, String toDir,
			Object[] filesPicked, boolean updateRepeatButton)
	{
		// Range is used for reporting progress to popUpDialog.
		BoundedRangeModel range = new DefaultBoundedRangeModel();		
		new MoveOrCopyIt(moveNotCopy, toNode, toDir, filesPicked, range, updateRepeatButton);
	}

	int confirmMove(String label, DirectoryTreeNode node, String dir,
			Object[] filesPicked)
	{
		if (filesPicked.length == 0) { return MOVE_ONE; }

		String location = "unknown"; //$NON-NLS-1$
		if (node != null)
		{
			location = node.getAbsolutePath();
		}
		else
		{
			location = dir;
		}

		int answer = JOptionPane.showConfirmDialog(this.myRDPmanager.myPlugins.getTopFrame(),
				AppStarter.getString("MoveManager.11") + " " + filesPicked.length //$NON-NLS-1$ //$NON-NLS-2$
						+ " " + AppStarter.getString("MoveManager.12") + location //$NON-NLS-1$ //$NON-NLS-2$
						+ "?\n", AppStarter.getString("MoveManager.13") + " " + label, JOptionPane.YES_NO_OPTION); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

		if (answer == JOptionPane.NO_OPTION) { return MOVE_NONE; }
		return MOVE_MANY;
	}
	
	private class MoveOrCopyIt implements Runnable
	{
		boolean iMoveNotCopy;
		DirectoryTreeNode myToNode;
		String myToDir;
		Object[] myFilesPicked;
		BoundedRangeModel myRange;
		boolean iSetRepeatButtonToThisDirectory;
		ModalProgressDialogFast myDialog;
		JFrame myTopFrame =  MoveManager.this.myRDPmanager.myPlugins.getTopFrame();

		/**
		 * Pass in EITHER toNode or toDir.
		 * @param toNode
		 * @param toDir
		 * @param filesPicked
		 * @param range
		 */
		MoveOrCopyIt(boolean moveNotCopy, DirectoryTreeNode toNode, String toDir,
				Object[] filesPicked, BoundedRangeModel range, boolean setRepeatButtonToThisDirectory)
		{
			this.iMoveNotCopy = moveNotCopy;
			this.myToNode = toNode;
			this.myToDir = toDir;
			this.myFilesPicked = filesPicked;
			this.myRange = range;
			this.iSetRepeatButtonToThisDirectory = setRepeatButtonToThisDirectory;
			
			if (this.myToNode == null && this.myToDir == null)
			{
				SwingUtilities.invokeLater(new Runnable()
				{
					public void run()
					{
						// Return result BEFORE showing error message.
						JOptionPane.showMessageDialog(MoveOrCopyIt.this.myTopFrame, AppStarter.getString("MoveManager.3") //$NON-NLS-1$
								+ FileUtil.NEWLINE + AppStarter.getString("MoveManager.4") //$NON-NLS-1$
								+ FileUtil.NEWLINE, AppStarter.getString("MoveManager.5"), //$NON-NLS-1$
								JOptionPane.ERROR_MESSAGE);
					}
				});
				return;
			}
			
			// Text for a move
			String title = AppStarter.getString("MoveManager.0");
			String message = AppStarter.getString("MoveManager.1");
			String cancelBtnText = AppStarter.getString("MoveManager.2"); 
			
			// Text for a copy
			if (!this.iMoveNotCopy)
			{
				title = AppStarter.getString("MoveManager.6");
				message = AppStarter.getString("MoveManager.7");
				cancelBtnText = AppStarter.getString("MoveManager.8"); 				
			}

			this.myDialog = 
				new ModalProgressDialogFast(this.myTopFrame,
	      		title,
	      		filesPicked.length + FileUtil.SPACE + message,  
	      		cancelBtnText,  
	      		this.myRange, 
	      		this);
			this.myDialog.start();
		}

		@SuppressWarnings("null")
		public void run()
		{
			//System.out.println("MoveManager MoveOrCopyIt Running... with FilesPicked of: " + this.myFilesPicked.length); //$NON-NLS-1$
			
			updateResetRecommendation(this.myFilesPicked); 
			final int returnToIndex = getResetRecommendation();		
			
			this.myRange.setMaximum(this.myFilesPicked.length -1 );
			UndoableMovingFiles undo = null;
			try
			{
				if (this.myToNode != null)
				{
					undo = new UndoableMovingFiles(this.myToNode, MoveManager.this.myRDPmanager, this.iSetRepeatButtonToThisDirectory);
					undo.setMoveNotCopy(this.iMoveNotCopy);
					
					try
					{
						undo.add(this.myFilesPicked, this.myRange);
					}
					catch (org.gerhardb.lib.io.TargetFileExistsException ex)
					{
						System.out.println("TargetFileExistsException Issue in MoveManager MoveOrCopyIt NODE"); //$NON-NLS-1$
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
					}
				}
				else if (this.myToDir != null)
				{
					// Can't change repeat without a node...
					undo =  new UndoableMovingFiles(new File(this.myToDir), MoveManager.this.myRDPmanager);	
					undo.setMoveNotCopy(this.iMoveNotCopy);				
					try
					{
						undo.add(this.myFilesPicked, this.myRange);
					}
					catch (org.gerhardb.lib.io.TargetFileExistsException ex)
					{
						System.out.println("TargetFileExistsException Issue in MoveManager MoveOrCopyIt DIRECTORY"); //$NON-NLS-1$
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
					}
				}
				else
				{
					System.out.println("MoveOrCopyIt: Move Pressed with empty directory: THIS SHOULD NEVER HAPPEN"); //$NON-NLS-1$
					return;
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}

			// Redisplay
			final File[] failedFiles = undo.getFailedFilesShowingMessage(this.myTopFrame);	
			if (failedFiles.length > 0)
			{
				System.out.println("MoveOrCopyIt: Number of files that failed: " + failedFiles.length);	
			}
			
			//System.out.println("returnToIndex: " + returnToIndex);

			// Get rid of dialog.
			// Do before queueing anything else on AWT thread.
			this.myDialog.done();
			
			// 31 May 2009 Coordinated Fix
			SwingUtilities.invokeLater(new Runnable()
			{
				public void run()
				{
					if (MoveOrCopyIt.this.iMoveNotCopy)
					{
						MoveOrCopyIt.this.myTopFrame.setCursor(Cursor.getPredefinedCursor( Cursor.WAIT_CURSOR ));
						//System.out.println("MoveManager CLEARING SELECTION"); //$NON-NLS-1$"
						MoveManager.this.myRDPmanager.myTreeManager.myCoordinator.getFileList().clearSelection();  
						
						MoveManager.this.myRDPmanager.myPlugins.reloadScroller(returnToIndex, IScroll.KEEP_CACHE);	 
						MoveManager.this.myRDPmanager.getIScrollDirTree().selectFiles(failedFiles);
						MoveOrCopyIt.this.myTopFrame.setCursor(Cursor.getPredefinedCursor( Cursor.DEFAULT_CURSOR ));
					}
					/*
					else
					{
						System.out.println("MoveManager NOT eloading Scroller because it's a COPY!"); //$NON-NLS-1$"						
					}
					*/

					// Make sure the scroller has focus after we have finished the move.
					MoveManager.this.myRDPmanager.myTreeManager.myCoordinator.getScroller().requestFocus();
				}
			});
		}
	}

	public void setUndoOff()
	{
		this.myUndoManager.discardAllEdits();
		updateUndoRedoButtons();
	} 

	public void addUndoable(UndoableEdit edit)
	{
		//new Error("foo").printStackTrace();
		this.myUndoManager.addEdit(edit);
		updateUndoRedoButtons();
	} 

	public void updateUndoRedoButtons()
	{
		if (this.myUndoManager.canUndo())
		{
			Action action = this.myUndoBtn.getAction();
			action.setEnabled(true);
			action.putValue(Action.SHORT_DESCRIPTION, this.myUndoManager.getUndoPresentationName());
		}
		else
		{
			Action action = this.myUndoBtn.getAction();
			action.setEnabled(false);
			String resourseKey = "SortScreen.menu.edit.undo.tip"; //$NON-NLS-1$
			String retrieved = AppStarter.getString(resourseKey);
			action.putValue(Action.SHORT_DESCRIPTION, retrieved);
		}

		if (this.myUndoManager.canRedo())
		{
			Action action = this.myRedoBtn.getAction();
			action.setEnabled(true);
			action.putValue(Action.SHORT_DESCRIPTION, this.myUndoManager.getRedoPresentationName());
		}
		else
		{
			Action action = this.myRedoBtn.getAction();
			action.setEnabled(false);
			String resourseKey = "SortScreen.menu.edit.redo.tip"; //$NON-NLS-1$
			String retrieved = AppStarter.getString(resourseKey);
			action.putValue(Action.SHORT_DESCRIPTION, retrieved);
		}
	}
	
	// ==========================================================================
	//                 Resetting FileList after moving a picture
	// ==========================================================================
	
	int getResetRecommendation()
	{
		int rtnMe = this.myResetRecommendation;
		//System.out.println("Move Manager Reset Recommendation: " + rtnMe);
		
		Scroller scroller = this.myRDPmanager.myTreeManager.myCoordinator.getScroller();
		
		
		// Don't go past bottom of list.
		if (rtnMe > scroller.getMaximumZeroBased())
		{
			rtnMe = scroller.getMaximumZeroBased();
			//System.out.println("USING MAX getMaximumZeroBased(): " + rtnMe);
		}		

		// If we still don't know, just use whatever is showing.
		if (rtnMe < 0)
		{
			rtnMe = scroller.getValueZeroBased();
			//System.out.println("USING MIN getMaximumZeroBased(): " + rtnMe);
		}		
		
		// Reset checkers
		this.myResetRecommendation = -1;
		
		// OK, we are done.
		return rtnMe;		
	}
	
	private void updateResetRecommendation()
	{
		this.myResetRecommendation = this.myRDPmanager.myTreeManager.myCoordinator.getScroller().getValueZeroBased(); 
		//System.out.println("Move Manager SINGLE FILE setting to: " + this.myResetRecommendation);
	}
	
	void updateResetRecommendation(Object[] filesPicked)
	{		
		// What is wanted is the first picture which was not moved.
		// So, we find the first picture which was moved and that is our
		// reset recommendation.  The first not moved picture will occupy
		// that position after all the moved pictures are gone.
		
		// If we don't find anything, just stay where we are.
		this.myResetRecommendation = this.myRDPmanager.myTreeManager.myCoordinator.getScroller().getValueZeroBased();  
		FileList fileList = this.myRDPmanager.myTreeManager.myCoordinator.getFileList();
		int listSize = fileList.getModel().getSize();
		
		// Scan the files starting at zero.
		fileListLoop:
		for(int fileIndex=0; fileIndex<listSize; fileIndex++)
		{
			Object item = fileList.getModel().getElementAt(fileIndex);
			if (item instanceof File)
			{
				File file = (File)item;
				for(int pickedIndex=0; pickedIndex<filesPicked.length; pickedIndex++)
				{
					if (file.equals(filesPicked[pickedIndex]))
					{
						this.myResetRecommendation = fileIndex; 
						break fileListLoop;
					}
				}
			}
		}
		//System.out.println("LSTC MULTI FILE setting to: " + this.myResetRecommendation);
	}
}
