/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
package org.gerhardb.lib.dirtree.rdp;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.*;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSlider;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;

import org.gerhardb.lib.io.FileUtil;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.util.ActionHelpers;
import org.gerhardb.lib.util.Misc;
import org.gerhardb.lib.util.startup.AppStarter;


/**
 * Accommodates relative paths.
 * Paths may be stored either absolute or relative.
 * Relative paths are stored starting with a "."
 * Examples are "." or "./cow".
 * Queries will return absolute or relative as desired.
 * Absolute is reconstituted from a relative by adding the startup directory path.
 * Default directories are used if the indicated directory can't be found.
 * @author  Gerhard Beck
 */
public class PathManager
{
	public static final int DIR_TREE = 0;
	public static final int DIR_DELETE = 1;
	public static final int DIR_PARK = 2;
	public static final int DIR_START = 3;
	public static final int DIR_LAST_VIEWED = 4;
	
	// MUST MATCH INDEX ABOVE!!
	public static final String[] DIR_LOOKUPS = 
		{"treeDirectory", "deleteDirectory", "parkDirectory", "viewDirectory", "lastDirectory"};
	
	public static final String VIEW_DIR_BTN_TEXT = AppStarter.getString("VPDMainPanel.0"); //$NON-NLS-1$
   
   private static final int DIR_COL_WIDTH = 50;
   private static final String RELATIVE_DIR = getRelativeDirectoryBase();
  	private static final String DOT = ".";
   private static final String RELATIVE_TOKEN = "./";

   static String RECENT_TREE = "RECENT_TREE";
	static String RECENT_VIEW = "RECENT_VIEW";
	static int DEFAULT_MAX_COUNT = 9;
   static String RECENT_TREE_COUNT = "RECENT_TREE_COUNT";
	static String RECENT_VIEW_COUNT = "RECENT_VIEW_COUNT";
	private static int MAX_DIRECTORY_COUNT = 20;
	
	private static File clsClickedOnJibs = null;
	private static String overrideStartDir = null;
	private static String overrideTreeDir = null;
	
	Preferences myPrefs;
	
   RelativePathTextField treeFld;
   RelativePathTextField deleteFld;
   RelativePathTextField parkFld;
   RelativePathTextField startFld;
   
   DirButton btnTree;
   DirButton btnDelete;
   DirButton btnPark;
   DirButton btnStartDir;
   
   JRadioButton pathRelative =
      new JRadioButton("Subdirectory of (for thumbdrives): "); //$NON-NLS-1$

   JRadioButton pathAbsolute =
      new JRadioButton("Fixed path"); //$NON-NLS-1$   
   
   JPanel basicPanel = new JPanel(false);
   JMenu myRecentTreeMenu;
	JMenu myRecentDirMenu;

	RDPplugins myPlugins;
   
   //===========================================================================
   //                           Constructor & Public
   //===========================================================================
   public PathManager(RDPplugins plugins, String appStartID)
   {
      this.myPlugins = plugins;
      this.myPrefs = Preferences.userRoot().node("/" + appStartID + "org/gerhardb/jibs/viewer/PathManager");  //$NON-NLS-1$
   }
   
   void init()
   {
      //System.out.println("RELATIVE_DIR: " + RELATIVE_DIR);
   	final String INITIALIAZATION_NEEDED = "initializationNeeded";
    	if (this.myPrefs.getBoolean(INITIALIAZATION_NEEDED, true))
   	{
    		if (AppStarter.isThumbdrive())
	      {
	      	initializeThumbdrive();
	      }
	      else
			{
       		doFirstTimeInitialization();
			}
     		this.myPrefs.putBoolean(INITIALIAZATION_NEEDED, false);
		}
     
	   this.treeFld = new RelativePathTextField(DIR_COL_WIDTH, "pics", getDirectoryRelative(DIR_TREE));
	   this.deleteFld = new RelativePathTextField(DIR_COL_WIDTH, "delete", getDirectoryRelative(DIR_DELETE));
	   this.parkFld = new RelativePathTextField(DIR_COL_WIDTH, "park", getDirectoryRelative(DIR_PARK));
	   this.startFld = new RelativePathTextField(DIR_COL_WIDTH, "pics", getDirectoryRelative(DIR_START));
	   
      this.treeFld.setToolTipText(AppStarter.getString("VPDMainPanel.33")); //$NON-NLS-1$ //$NON-NLS-2$
      this.deleteFld.setToolTipText(AppStarter.getString("VPDMainPanel.35")); //$NON-NLS-1$ //$NON-NLS-2$
      this.parkFld.setToolTipText(AppStarter.getString("VPDMainPanel.37")); //$NON-NLS-1$ //$NON-NLS-2$
      this.startFld.setToolTipText("<html>" + AppStarter.getString("VPDMainPanel.40") + " <i>" + VIEW_DIR_BTN_TEXT + "</i>" + AppStarter.getString("VPDMainPanel.43") + ".</html>"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$ //$NON-NLS-6$     

      this.btnTree = new DirButton(this.treeFld, "Tree" ); 
      this.btnDelete = new DirButton(this.deleteFld, "Delete" ); 
      this.btnPark = new DirButton(this.parkFld, "Park" ); 
      this.btnStartDir = new DirButton(this.startFld, "Start" );   
   }

   public static boolean overrideStartPassedIn()
   {
   	if (overrideStartDir == null)
   	{
   		return false;
   	}
		return true;
   }
   
   public void setDirectory(int type, File dir)
   {
   	setDirectory(type, dir.getAbsolutePath());
   }
   
   public void setDirectory(int type, String dir)
   {
   	if (dir == null)
   	{
			this.myPrefs.remove(DIR_LOOKUPS[type]);  
			return;
   	}
   	
   	String directory = null;
   	dir = normalizePathSeparator(dir);
   	if (dir.startsWith(DOT))
   	{
   		if (dir.equals(RELATIVE_TOKEN))
   		{
   			directory = DOT;
   		}
   		else
   		{
   			directory = dir;
   		}
			this.myPrefs.put(DIR_LOOKUPS[type], directory);
   	}
   	else
   	{
	 		String validated = FileUtil.safeValidateDirectory(dir);
			if (validated != null)
			{
   			directory = normalizePathSeparator(validated);
				this.myPrefs.put(DIR_LOOKUPS[type], directory);
			}
			else
			{
				this.myPrefs.remove(DIR_LOOKUPS[type]);
			}
   	}
   	
      switch(type)
   	{
      case DIR_TREE:
      	{
      	overrideTreeDir = null;          
			int maxListSize = this.myPrefs.getInt(RECENT_TREE_COUNT, DEFAULT_MAX_COUNT);
      	storeUniqueDirectories(RECENT_TREE, directory, maxListSize, this.myPrefs);
      	break;
      	}
      case DIR_START:
      	overrideStartDir = null;
      	break;
   	case DIR_DELETE:
   	case DIR_PARK:
   		this.myPlugins.loadParkAndDeleteNodes();
         break;
   	case DIR_LAST_VIEWED:
   		{
   	      int maxListSize = this.myPrefs.getInt(RECENT_VIEW_COUNT, DEFAULT_MAX_COUNT);
         	storeUniqueDirectories(RECENT_VIEW, directory, maxListSize, this.myPrefs);
         break;
   		}
   	}
   }   
   
	public String getDirectoryAbsolute(int type)
	{
		return getDirectory(type, true);
	}
	
	public String getDirectoryRelative(int type)
	{
		return getDirectory(type, false);	
	}
	
	public static File getStartingJibsFile()
	{
		return clsClickedOnJibs;
	}
		
	public static void setStartingJibsFile(File jibs)
	{
		clsClickedOnJibs = jibs;
	}
	
	public JMenu getRecentTreeMenu()
	{
		if (this.myRecentTreeMenu == null)
		{
			this.myRecentTreeMenu = ActionHelpers.makeMenu("recenttrees");
			this.myRecentTreeMenu.addMenuListener(new TreeMenuListener());
		}
		return this.myRecentTreeMenu;
	}
	
	public JMenu getRecentDirMenu()
	{
		if (this.myRecentDirMenu == null)
		{
			this.myRecentDirMenu = ActionHelpers.makeMenu("recentviews");
			this.myRecentDirMenu.addMenuListener(new ViewMenuListener());
		}
		return this.myRecentDirMenu;
	}
  //===========================================================================
   //                          Not Public
   //===========================================================================
	public static ArrayList<String> getDirectoryFromDisk(String listKey, Preferences prefs) //this.myPrefs
	{
		ArrayList<String> theList = new ArrayList<String>(MAX_DIRECTORY_COUNT);
		// Get the array from disk.
      try
      {
      	byte[] byteArray = null;
      	byteArray = prefs.getByteArray(listKey, byteArray); 
      	Object object = Misc.toObject(byteArray);
      	if (object != null && object instanceof ArrayList<?>)
      	{
      		ArrayList<?> untypedList = (ArrayList<?>)object;
      		for(int i=0; i<untypedList.size(); i++)
      		{
      			theList.add((String)untypedList.get(i));
      		}
      	}
      }
      catch(Exception ex)
      {
      	ex.printStackTrace();
      }		
      return theList;
	}
	
	/**
	 * Store the current list of directories to show the file menu for 
	 * both last several viewed directories and last several tree
	 * directories.  Array should be limited to 20 items.
	 * @param listKey
	 * @param theList
	 * @param directory
	 */
	public static void storeUniqueDirectories(String listKey, String directory, int maxListSize, Preferences prefs)
	{
		ArrayList<String> theList = getDirectoryFromDisk(listKey, prefs);
     // Move latest selection to the top.
      int index = theList.indexOf(directory);
      if (index > -1)
      {
      	theList.remove(index);
      }		      
      theList.add(0, directory);
      
      while(theList.size() > maxListSize)
      {
      	theList.remove(theList.size() - 1);
      }
            
      // Put the array back to disk.
      try
      {
      	byte[] bytes = Misc.toByteArray(theList);
      	prefs.putByteArray(listKey, bytes);
      	prefs.flush();
      }
      catch(Exception ex)
      {
      	ex.printStackTrace();
      }	
	}
	
	private String getDirectory(int type, boolean absolute)
	{
		String dir = null;
		switch (type)
		{
			case DIR_TREE:
				if (overrideTreeDir != null && overrideTreeDir.length() > 0) 
				{ dir = overrideTreeDir; }
				break;
			case DIR_START:
				if (overrideStartDir != null && overrideStartDir.length() > 0) 
				{ dir = overrideStartDir; }
				break;
			//case DIR_LAST_VIEWED:
			//	if (overrideStartDir != null) { dir = overrideStartDir; }
		}

		if (dir == null)
		{
			String rawDir = this.myPrefs.get(DIR_LOOKUPS[type], null);
			//System.out.println("PathManager LOOKED UP: " + rawDir);
			dir = normalizePathSeparator(rawDir);
			dir = convert(dir, absolute);
		}
				
		if (dir == null)
		{
			dir = getDefaultPath(type);
			dir = convert(dir, absolute);
			//System.out.println("PathManager GRABBED DEFAULT: " + dir);
		}
		else
		{
			//System.out.println("PathManager FOUND: " + dir);			
		}		
		
		return dir;
	}
	
	static String convert(String dir, boolean absolute)
	{
		if (dir == null){return null;}
		dir = normalizePathSeparator(dir);
		//System.out.println("Converted Dir: " + dir);
		if (absolute)
		{
			if (dir.startsWith(DOT))
			{
				if (dir.length() > 2)
				{
					dir = RELATIVE_DIR + "/" + dir.substring(2);
				}
				else
				{
					dir = RELATIVE_DIR;
				}
			}
			// Just keep absolute file if you get to here.
			// Check to see if it is still there.
			return FileUtil.checkDirectory(dir);
		}
		
		if (dir.startsWith(DOT))
		{
			if (dir.length() > 2)
			{
				String fullPath = RELATIVE_DIR + "/" + dir.substring(2);
		      if (FileUtil.checkDirectory(fullPath) != null)
		      {
		      	// dir was a valid subdirectory
		      	return dir;
		      }
				// dir was bad, return flag for default
				dir =  null;
			}
			else
			{
				// . or ./ is same as .
				return DOT;					
			}
		}
		else if (dir.startsWith(RELATIVE_DIR))
		{			
			if (dir.length() > RELATIVE_DIR.length())
			{
				// Check to see if it is still there.
				if (FileUtil.checkDirectory(dir) != null)
				{
					String rtnMe = RELATIVE_TOKEN + dir.substring(RELATIVE_DIR.length() + 1);
					return rtnMe;						
				}
				return null;
			}
			return DOT;
		}
		// Just keep absolute file if you get to here.
		// We don't know how to make it relative because it's not...		
		return FileUtil.checkDirectory(dir);
	}
	
	/**
	 * Shows the dialog box where you ask the user what directory is wanted.
	 * Comes off the ... buttons.
	 * @param field
	 * @param label
	 * @return
	 */
   public void showDirDialog(RelativePathTextField field, String label)
   {
      String dir = field.getText();
      // Make sure we have the absolute version.
      dir = convert(dir, true);
      
      if (dir == null)
      {
      	if (this.pathRelative.isSelected())
      	{
       		dir = calcDefaultRelDir(field);
      	}
      	else
      	{
            dir = System.getProperty("user.home"); //$NON-NLS-1$     		
      	}
      }
      else
      {
	     	if (this.pathRelative.isSelected())
	   	{
	     		if (!dir.startsWith(RELATIVE_DIR))
	     		{
	        		dir = calcDefaultRelDir(field);
	     		}
	   	}
      }
      // Do the lookup now.
      dir = FileUtil.lookupDir(label, dir, this.basicPanel); //$NON-NLS-1$
      if (dir != null)
      {
      	field.setText(dir);
      }
   }
   
   private String calcDefaultRelDir(RelativePathTextField field)
   {
		String relDefault = RELATIVE_DIR + field.myDefaultName;
 		String dir = FileUtil.checkDirectory(relDefault);
		if (dir == null)
		{
     		dir = RELATIVE_DIR; 
		}
   	return dir;
   }
   
   public JComponent directoryPanel()
   {

      JLabel relPath = new JLabel(RELATIVE_DIR);
      
      ButtonGroup pathGroup = new ButtonGroup();
      pathGroup.add(this.pathAbsolute);
      pathGroup.add(this.pathRelative); 
      
      // Non thumbdrive behavior
      if (getAbsolutePath())
      {
      	this.pathAbsolute.setSelected(true);
      }
      else
      {
      	this.pathRelative.setSelected(true);
      }
      
      // Thumbdrive behavior
      if (AppStarter.isThumbdrive())
      {
      	this.pathAbsolute.setSelected(false);
      	this.pathAbsolute.setEnabled(false);
      	this.pathRelative.setSelected(true);
      }
      
      adjustFields();
   	
      // --------------- Action Listeners -------------------------

      this.pathAbsolute.addActionListener
      (
      new ActionListener()
   {
      public void actionPerformed(ActionEvent ae)
      {
      	adjustFields();
      }
   }
   );

      this.pathRelative.addActionListener
      (
      new ActionListener()
   {
      public void actionPerformed(ActionEvent ae)
      {
      	adjustFields();
      }
   }
   );
      
      final JSlider treeSlider = new JSlider(0, MAX_DIRECTORY_COUNT, this.myPrefs.getInt(RECENT_TREE_COUNT, DEFAULT_MAX_COUNT));
      treeSlider.setSnapToTicks(true);
      treeSlider.setPaintLabels(true);
      treeSlider.setPaintTicks(true);
      treeSlider.setMinimum(1);
      treeSlider.setMajorTickSpacing(5);
      treeSlider.addChangeListener(
            new ChangeListener()
         {
            public void stateChanged(ChangeEvent e)
            {
            	PathManager.this.myPrefs.putInt(RECENT_TREE_COUNT, treeSlider.getValue());
            	try{PathManager.this.myPrefs.flush();}catch(Exception ex){ex.printStackTrace();}
            }
         });
      JButton treeClearBtn = new JButton("Clear");
      treeClearBtn.addActionListener
      (
            new ActionListener()
         {
            public void actionPerformed(ActionEvent ae)
            {
        			PathManager.this.myPrefs.remove(RECENT_TREE);
            	try{PathManager.this.myPrefs.flush();}catch(Exception ex){ex.printStackTrace();}
            }
         }
         );
      
      final JSlider dirSlider = new JSlider(0, MAX_DIRECTORY_COUNT, this.myPrefs.getInt(RECENT_VIEW_COUNT, DEFAULT_MAX_COUNT));
      dirSlider.setSnapToTicks(true);
      dirSlider.setPaintLabels(true);
      dirSlider.setPaintTicks(true);
      dirSlider.setMinimum(1);
      dirSlider.setMajorTickSpacing(5);
      dirSlider.addChangeListener(
            new ChangeListener()
         {
            public void stateChanged(ChangeEvent e)
            {
            	PathManager.this.myPrefs.putInt(RECENT_VIEW_COUNT, treeSlider.getValue());
            	try{PathManager.this.myPrefs.flush();}catch(Exception ex){ex.printStackTrace();}
            }
         });
      JButton dirClearBtn = new JButton("Clear");
      dirClearBtn.addActionListener
      (
            new ActionListener()
         {
            public void actionPerformed(ActionEvent ae)
            {
        			PathManager.this.myPrefs.remove(RECENT_VIEW);
        			try{PathManager.this.myPrefs.flush();}catch(Exception ex){ex.printStackTrace();}
            }
         }
         );

     // --------------- Panel  -------------------------

   	JPanel leftPanel = new JPanel(new GridLayout(4,1));
   	leftPanel.add(new JLabel(AppStarter.getString("VPDMainPanel.19") + ": ")); //$NON-NLS-1$ //$NON-NLS-2$
   	leftPanel.add(new JLabel(AppStarter.getString("VPDMainPanel.21") + ": ")); //$NON-NLS-1$ //$NON-NLS-2$
   	leftPanel.add(new JLabel(AppStarter.getString("VPDMainPanel.23") + ": ")); //$NON-NLS-1$ //$NON-NLS-2$
   	leftPanel.add(new JLabel(AppStarter.getString("VPDMainPanel.55") + ": ")); //$NON-NLS-1$ //$NON-NLS-2$
	
   	JPanel centerPanel = new JPanel(new GridLayout(4,1));
   	centerPanel.add(this.treeFld);
   	centerPanel.add(this.deleteFld);
   	centerPanel.add(this.parkFld);
   	centerPanel.add(this.startFld);
   	
   	JPanel rightPanel = new JPanel(new GridLayout(4,1));
   	rightPanel.add(this.btnTree);
   	rightPanel.add(this.btnDelete);
   	rightPanel.add(this.btnPark);
   	rightPanel.add(this.btnStartDir);
	    	
   	JPanel daGrid = new JPanel(new BorderLayout());
   	daGrid.add(leftPanel, BorderLayout.WEST);
   	daGrid.add(centerPanel, BorderLayout.CENTER);
   	daGrid.add(rightPanel, BorderLayout.EAST);
   	
      JPanelRows relPathPanel = new JPanelRows();
      JPanel nextRow = relPathPanel.topRow();
      nextRow.add(this.pathAbsolute);

      nextRow = relPathPanel.nextRow();
      nextRow.add(this.pathRelative);   
      nextRow.add(relPath);   	
   	
      nextRow = relPathPanel.nextRow();
      nextRow.add(daGrid);   

      nextRow = relPathPanel.nextRow();
      nextRow.add(new JLabel("View directories to remember: "));   
      nextRow.add(dirSlider);
      nextRow.add(dirClearBtn);

      nextRow = relPathPanel.nextRow();
      nextRow.add(new JLabel("Tree directories to remember: "));   
      nextRow.add(treeSlider);
      nextRow.add(treeClearBtn);

     	return relPathPanel;
   }
   

   public void adjustView(boolean startParticularDir)
   {
      // Turn the view directory on and off
      if (startParticularDir)
      {
         this.startFld.setEnabled(true);
         this.btnStartDir.setEnabled(true);
      }
      else
      {
         this.startFld.setEnabled(false);
         this.btnStartDir.setEnabled(false);
      }
   }

   public void save() 
	{
		try
		{
			// Verify the input and try to create directories if missing.
			String tree = convert(this.treeFld.getText(), true);
			if (!checkDirectory(tree, this.basicPanel))
			{
				JOptionPane.showMessageDialog(this.basicPanel, AppStarter
						.getString("VPDMainPanel.72") + "\n" //$NON-NLS-1$ //$NON-NLS-2$
						+ AppStarter.getString("VPDMainPanel.74"), //$NON-NLS-1$
						AppStarter.getString("save.failed"), //$NON-NLS-1$
						JOptionPane.WARNING_MESSAGE);
				return;
			}

			String trash = convert(this.deleteFld.getText(), true);
			if (trash != null && trash.length() > 0 && !checkDirectory(trash, this.basicPanel))
			{
				JOptionPane
						.showMessageDialog(
								this.basicPanel,
								AppStarter.getString("VPDMainPanel.76") + AppStarter.getString("VPDMainPanel.77") //$NON-NLS-1$ //$NON-NLS-2$
										+ AppStarter.getString("VPDMainPanel.78"), //$NON-NLS-1$
								AppStarter.getString("save.failed"), //$NON-NLS-1$
								JOptionPane.WARNING_MESSAGE);
				return;
			}

			String park = convert(this.parkFld.getText(), true);
			if (park != null && park.length() > 0 && !checkDirectory(park, this.basicPanel))
			{
				JOptionPane.showMessageDialog(this.basicPanel, AppStarter
						.getString("VPDMainPanel.80") + "\n" //$NON-NLS-1$ //$NON-NLS-2$
						+ AppStarter.getString("VPDMainPanel.82"), //$NON-NLS-1$
						AppStarter.getString("save.failed"), //$NON-NLS-1$
						JOptionPane.WARNING_MESSAGE);
				return;
			}
			
			if (this.startFld.isEnabled())
     		{
				String start = convert(this.startFld.getText(), true);
   			if (!checkDirectory(start, this.basicPanel))
   			{
   				JOptionPane.showMessageDialog(this.basicPanel, AppStarter
   						.getString("VPDMainPanel.84") + "\n" //$NON-NLS-1$ //$NON-NLS-2$
   						+ AppStarter.getString("VPDMainPanel.86"), //$NON-NLS-1$
   						AppStarter.getString("save.failed"), //$NON-NLS-1$
   						JOptionPane.WARNING_MESSAGE);
   			}     			
     		}
   		
			setAbsolutePath(this.pathAbsolute.isSelected());

			setDirectory(DIR_TREE, this.treeFld.getText());
			setDirectory(DIR_DELETE, this.deleteFld.getText());
     		setDirectory(DIR_PARK, this.parkFld.getText());
     		setDirectory(DIR_START, this.startFld.getText());
     		
 		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			JOptionPane.showMessageDialog(this.basicPanel, ex.getMessage(), AppStarter
					.getString("save.failed"), //$NON-NLS-1$
					JOptionPane.ERROR_MESSAGE);
			return;
		}
	}
  
   void adjustFields()
   {
   	this.treeFld.adjustText();
   	this.deleteFld.adjustText();
   	this.parkFld.adjustText();
   	this.startFld.adjustText();
   }

   private static String getRelativeDirectoryBase()
   {
   	String path = AppStarter.getStartupDirectory().getAbsolutePath();
   	path = normalizePathSeparator(path);
   	if (path.endsWith("/"))
   	{
   		path = path.substring(0, path.length()-1);
   	}
   	return path;
   }
   
   //===========================================================================
   //                              Inner Classes 
   //===========================================================================
   public class RelativePathTextField extends JTextField
   {
   	String myDefaultName;
   	
   	RelativePathTextField(int cols, String defaultName, String text)
   	{
   		super(cols);
   		this.myDefaultName = RELATIVE_TOKEN + defaultName;
   		setText(text);   			
   	}
   	
   	@Override
		public void setText(String text)
   	{
   		if (text == null)
   		{
   			super.setText(this.myDefaultName);
   		}
   		else
   		{
   			super.setText(convert(text, PathManager.this.pathAbsolute.isSelected()));
   		}
   	}
   	
   	public void adjustText()
   	{
   		String text = convert(getText(), PathManager.this.pathAbsolute.isSelected());
      	super.setText(text);   		
   	}
   }
   
   class DirButton extends JButton
   {
   	DirButton(final RelativePathTextField field, final String toolTip)
   	{
   		super("..."); //$NON-NLS-1$
   		super.setToolTipText(toolTip);
         addActionListener
         (
		      new ActionListener()
		      {
		         public void actionPerformed(ActionEvent ae)
		         {
		          	showDirDialog(field, AppStarter.getString("PathManager.1") + toolTip); //$NON-NLS-1$
		         }
		      }
		   );
   	}
   }

   //===========================================================================
   //                              Static Methods
   //===========================================================================
   public static boolean checkDirectory(String dir, Component showOnTopOf)
   {
      if (dir == null || dir.trim().length() == 0)
      {
         return false;
      }
      try
      {
         FileUtil.validateDirectory(dir.toString());
      }
      catch (FileNotFoundException ex)
      {
         int create = JOptionPane.showConfirmDialog(
         				showOnTopOf,
                      dir
                      + " " + AppStarter.getString("VPDMainPanel.90"), //$NON-NLS-1$ //$NON-NLS-2$
                      AppStarter.getString("VPDMainPanel.91"), //$NON-NLS-1$
                      JOptionPane.YES_NO_OPTION);
         if (create == JOptionPane.YES_OPTION)
         {
            boolean rtnMe = false;
            File file = new File(dir);
            try
            {
               rtnMe = file.mkdir();
            }
            catch (Exception exx)
            {
               JOptionPane.showMessageDialog(
               		showOnTopOf,
                  exx.getMessage(),
                  AppStarter.getString("VPDMainPanel.92"), //$NON-NLS-1$
                  JOptionPane.ERROR_MESSAGE);
               return false;
            }
            return rtnMe;
         }
         return false;
      }
      catch (Exception ex)
      {
         return false;
      }
      return true;
   }
   
   private static String normalizePathSeparator(String in)
   {
   	if (in == null){return null;}
   	return in.replace('\\', '/');
   }
   //===========================================================================
   //                              Preferences
   //===========================================================================
   //---------------------------------------------------------------------------
	// Start & Tree Directory
	// Overrides
   //---------------------------------------------------------------------------

	/**
	 * Overrides the stored tree directory. Designed to be use when requesting a
	 * specific tree directory from the command line.
	 */
	public static void overrideInitialDirectories(String tree, String start)
	{
		overrideTreeDir = FileUtil.checkDirectory(tree);
		overrideStartDir = FileUtil.checkDirectory(start);
		AppStarter.log("overrideTreeDir: " + overrideTreeDir);
		AppStarter.log("overrideStartDir: " + overrideStartDir);
	}
	
   //---------------------------------------------------------------------------
   //                         Common Preference Functions
   //---------------------------------------------------------------------------

	public void flush()
   {
      try
      {
         this.myPrefs.flush();
      }
      catch (BackingStoreException ex)
      {
      	ex.printStackTrace();
      }
   }
   
	// ===========================================================================
	// Initialization Subroutines
	// ===========================================================================
	private String getDefaultPath(int type)
	{			
      switch(type)
   	{
	      case DIR_DELETE:
	   		/**
	   		 * DIR_DELETE:
	   		 * Does not guess for Windows since the Windows Recycle Bin exists separately
	   		 * on each drive, so this could potentially vary as the user moves around.
	   		 * Otherwise it is likely a hidden directory at c:\RECYCLED.
	   		 */
	      	return null;
	   	case DIR_PARK:
	   		// Reads from old location for park.
	   		String park = this.myPrefs.get(DIR_LOOKUPS[DIR_PARK], null);
	   		if (park != null){return park;}
	   		// fall through to default
	   		//$FALL-THROUGH$
	   	default:
	   		return AppStarter.getMyDocumentsDir().getAbsolutePath();
	  	}		
 	}
	
	private void doFirstTimeInitialization()
	{
   	System.out.println("===================================================");
   	System.out.println("Doing First Time Initialization");
   	System.out.println("===================================================");
   	
   	
		//Default is to set everything to root picture directory.
		File picsDir = AppStarter.getMyPicturesDir();
		setDirectory(DIR_LAST_VIEWED, picsDir);
  		setDirectory(DIR_START, picsDir);
		setDirectory(DIR_TREE, picsDir);

		// Put park and delete under My Pictures/jibs/.
		File baseDir = new File(this.myPlugins.getDirectoryForParkAndDelete());
		if (!baseDir.exists())
		{
			try
			{
				// Making JIBS first.
				baseDir.mkdir();
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
				baseDir = null;
			}
		}
		
		if (baseDir != null)
		{
			makeNewDir(baseDir, DIR_PARK, "park"); //$NON-NLS-1$
			makeNewDir(baseDir, DIR_DELETE, "delete"); //$NON-NLS-1$
		}
				
 		for(int i=0; i<DIR_LOOKUPS.length; i++)
		{
			String item = this.myPlugins.getPreference(DIR_LOOKUPS[i], null);
			//String item = viewerPrefs.get(DIR_LOOKUPS[i], null);
			//System.out.println("item: " + item);
			if (item != null)
			{
				this.myPrefs.put(DIR_LOOKUPS[i], item);  
				this.myPlugins.removePreference(DIR_LOOKUPS[i]);
			}
		}
		/*
      <entry key="lastDirectory" value="D:/gerhard/My Pictures"/>
      <entry key="treeDirectory" value="D:/testpics/Fixed Files/Tree Alpha"/>
      <entry key="viewDirectory" value="D:/testpics/Fixed Files/Tree Alpha/view"/>
      <entry key="parkDirectory" value="D:/testpics/Fixed Files/Tree Alpha/park"/>
      <entry key="deleteDirectory" value="D:/testpics/Fixed Files/Tree Alpha/delete"/>		
		*/
		
		flush();
	}

	private void initializeThumbdrive()
	{
		File baseDir = new File(RELATIVE_DIR);
		makeNewDir(baseDir, DIR_LAST_VIEWED, "pictures"); //$NON-NLS-1$
		makeNewDir(baseDir, DIR_START, "pictures"); //$NON-NLS-1$
		makeNewDir(baseDir, DIR_TREE, "pictures"); //$NON-NLS-1$
		makeNewDir(baseDir, DIR_PARK, "pictures/park"); //$NON-NLS-1$
		makeNewDir(baseDir, DIR_DELETE, "pictures/delete"); //$NON-NLS-1$
		flush();
	}

	void makeNewDir(File baseDir, int dirType, String dirName)
	{
		File newDir = new File(baseDir.getAbsoluteFile() + "/" + dirName); //$NON-NLS-1$
		if (newDir.exists())
		{
			setDirectory(dirType, newDir);
		}
		else
		{
			try
			{
				newDir.mkdir();
				setDirectory(dirType, newDir);
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}		
	}
	
	class TreeMenuListener implements MenuListener
	{
		public void menuSelected(MenuEvent eA)
		{
			PathManager.this.myRecentTreeMenu.removeAll();
			int maxCount = PathManager.this.myPrefs.getInt(RECENT_TREE_COUNT, DEFAULT_MAX_COUNT);
			
			ArrayList<String> theList = getDirectoryFromDisk(RECENT_TREE, PathManager.this.myPrefs);
			for(int i=0; i<=maxCount && i < theList.size(); i++)
			{
				final String dirName = theList.get(i);
				if (dirName != null)
				{
					JMenuItem anItem = new JMenuItem(dirName);
					anItem.addActionListener(new ActionListener()
					{
						public void actionPerformed(ActionEvent eB)
						{
							PathManager.this.myPlugins.setTreeDirectory(dirName);
						}
					});
					PathManager.this.myRecentTreeMenu.add(anItem);
				}
			}
		}

		public void menuDeselected(MenuEvent e)
		{
			// We don't care
		}
	
		public void menuCanceled(MenuEvent e)
		{
			// We don't care
		}
	}

	class ViewMenuListener implements MenuListener
	{
		public void menuSelected(MenuEvent eA)
		{
			PathManager.this.myRecentDirMenu.removeAll();
			int maxCount = PathManager.this.myPrefs.getInt(RECENT_VIEW_COUNT, DEFAULT_MAX_COUNT);
			
			ArrayList<String> theList = getDirectoryFromDisk(RECENT_VIEW, PathManager.this.myPrefs);
			for(int i=0; i<=maxCount && i < theList.size(); i++)
			{
				final String dirName = theList.get(i);
				if (dirName != null)
				{
					JMenuItem anItem = new JMenuItem(dirName);
					anItem.addActionListener(new ActionListener()
					{
						public void actionPerformed(ActionEvent eB)
						{
							PathManager.this.myPlugins.setCurrentDirectory(dirName);
						}
					});
					PathManager.this.myRecentDirMenu.add(anItem);
				}
			}
		}

		public void menuDeselected(MenuEvent e)
		{
			// We don't care
		}
	
		public void menuCanceled(MenuEvent e)
		{
			// We don't care
		}
	}
	
	// ===========================================================================
	// Directory Paths
	// ===========================================================================
	private static final String PATH_ABSOLUTE = "absolutePath"; //$NON-NLS-1$

	private boolean getAbsolutePath()
	{
		return this.myPrefs.getBoolean(PATH_ABSOLUTE, true);
	}
	
	private void setAbsolutePath(boolean absolute)
	{
		this.myPrefs.putBoolean(PATH_ABSOLUTE, absolute);
	}

   //===========================================================================
   //                              MAIN
   //===========================================================================
	public static void main(String args[])
	{
		
		PathManager pm = new PathManager(null, "PathManager");			
		pm.setDirectory(DIR_START, "D:\\dev\\workspaces\\JIBS\\bin\\aaa");
		pm.flush();
		System.out.println("==========================================");
		System.out.println(RELATIVE_DIR);
		System.out.println("==========================================");
		
		runTest(pm, "D:\\dev\\workspaces\\JIBS\\bin\\aaa");
		runTest(pm, "D:/dev/workspaces/JIBS/bin/aaa");
		runTest(pm, ".\\aaa");
		runTest(pm, "./aaa");
		runTest(pm, "D:/dev/workspaces/JIBS/bin");
		runTest(pm, ".");
		runTest(pm, "./");
		
		runTest(pm, "D:/dev/workspaces/JIBS/bin/bbb");
		runTest(pm, "./bbb");
	}
	
	private static void runTest(PathManager pm, String test)
	{
		System.out.println("The Test: " + test); 
		pm.setDirectory(DIR_START, test);
		pm.flush();
		System.out.println("Absolute: " + pm.getDirectoryAbsolute(PathManager.DIR_START));
		System.out.println("Relative: " + pm.getDirectoryRelative(PathManager.DIR_START));
		System.out.println("=========================================");
	}
}

