/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree.rdp;

import java.awt.event.ActionEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.prefs.Preferences;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BoundedRangeModel;
import javax.swing.DefaultBoundedRangeModel;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JToolBar;

import org.gerhardb.lib.dirtree.*;
import org.gerhardb.lib.scroller.KeypadOps;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.util.Misc;
import org.gerhardb.lib.util.ModalProgressDialogFlex;
import org.gerhardb.lib.util.startup.ActiveActions;
import org.gerhardb.lib.util.startup.AppStarter;

/**
 * RPDManager runs the Repeat, Delete, and Park buttons.
 */
public class RDPmanager implements KeypadOps
{
	Preferences myPrefs;

	private static final String DIRECTORY_LIST_KEY = "DIRECTORY_LIST_KEY";

	private static final int waitTime = 0;
	private static final boolean updateRepeateButtonBecauseDeleteOrParkNotPressed = true;
	private static final boolean doNotUpdateRepeateButton = false;

	DirectoryTreeNode myRepeatNode = null;
	DirectoryTreeNode myDeleteNode = null;
	DirectoryTreeNode myParkNode = null;

	File myDeleteFile = new File(""); //$NON-NLS-1$
	File myParkFile = new File(""); //$NON-NLS-1$
	File myRepeatFile = new File(""); //$NON-NLS-1$

	String myDeleteDirAsString;
	String myParkDirAsString;
	String myRepeatDirAsString;

	JButton myRepeatBtn;
	JButton myDeleteBtn;
	JButton myParkBtn;
	JButton myDirectoryListBtn;

	MoveManager myMoveManager;

	// Used by DirectoryListPopup
	ArrayList<File> myDirectoryList = new ArrayList<File>(30);
	RDPplugins myPlugins;
	IUndo myIUndo;
	IScrollDirTree myIScrollDirTree;

	TreeManagerRDP myTreeManager;

	// ==========================================================================
	// Constructor
	// ==========================================================================
	public RDPmanager(RDPplugins ss, IUndo iu, IScrollDirTree is,
			String appStartID)
	{
		this.myPlugins = ss;
		this.myIUndo = iu;
		this.myIScrollDirTree = is;
		this.myPrefs = Preferences.userRoot().node(
				"/" + appStartID + "/org/gerhardb/jibs/viewer/frame/RDPmanager"); //$NON-NLS-1$

		this.myMoveManager = new MoveManager(this);

		try
		{
			byte[] byteArray = null;
			byteArray = this.myPrefs.getByteArray(DIRECTORY_LIST_KEY, byteArray);
			Object object = Misc.toObject(byteArray);
			if (object != null && object instanceof ArrayList<?>)
			{
				ArrayList<?> readIn = (ArrayList<?>)object;
				this.myDirectoryList = new ArrayList<File>(readIn.size());
				for (int i=0; i<readIn.size(); i++)
				{
					Object item = readIn.get(i);
					if (item instanceof File)
					{
						this.myDirectoryList.add((File)item);
					}
				}
			}
		}
		catch (Exception ex)
		{
			// eat it
		}

	}

	public String getAppIDandTargetType()
	{
		return this.myTreeManager.myCoordinator.myLSTplugins.getAppIDandTargetType();
	}
	
	public void addActions(ActiveActions aa)
	{
		this.myDeleteBtn = makeDeleteButton(aa);
		this.myParkBtn = makeParkButton(aa);
		this.myRepeatBtn = makeRepeatButton(aa);
		this.myDirectoryListBtn = makeDirectoryListButton(aa);
		this.myMoveManager.addActions(aa);
	}

	public RDPplugins getRDPplugins()
	{
		return this.myPlugins;
	}

	// ==========================================================================
	// Used to be DirectoryTreePlugins & ITree, but this directly replaced.
	// ==========================================================================

	public IUndo getIUndo()
	{
		return this.myIUndo;
	}

	public IScrollDirTree getIScrollDirTree()
	{
		return this.myIScrollDirTree;
	}

	public void showStats(DirectoryTreeNode node, boolean countImageFilesInNodes)
	{
		this.myPlugins.showStats(node, countImageFilesInNodes);
	}

	// ==========================================================================
	// KeypadOps interface
	// ==========================================================================
	public void clickPark()
	{
		this.myParkBtn.doClick(waitTime);
	}

	public void clickTrash()
	{
		this.myDeleteBtn.doClick(waitTime);
	}

	public void clickRepeat()
	{
		this.myRepeatBtn.doClick(waitTime);
	}

	public void clickDirectoryList()
	{
		if (isFullScreen())
		{
			new DirectoryListPopup(this.myPlugins.getTopFrame(), this, this.myPlugins
					.getExtendedDirectoryTree());
		}
		else
		{
			this.myDirectoryListBtn.doClick(waitTime);
		}
	}

	public boolean isFullScreen()
	{
		// Menu accelerators are off because there is no menu full screen.
		return this.myPlugins.isFullScreen();
	}

	// ==========================================================================
	// Public
	// ==========================================================================	

	public void setWaitCursor(boolean b)
	{
		this.myPlugins.setWaitCursor(b);
	}

	public MoveManager getMoveManager()
	{
		return this.myMoveManager;
	}

	public void updateDirNameIfNeeded(String oldName, String newName,
			DirectoryTreeNode node)
	{
		File oldFileName = new File(oldName);
		//System.out.println("oldName: " + oldName);
		//System.out.println("newName: " + newName);
		if (this.myRepeatFile != null)
		{
			if (this.myRepeatFile.equals(oldFileName))
			{
				setRepeatDir(node);
			}
		}
		if (this.myDeleteFile != null)
		{
			if (this.myDeleteFile.equals(oldFileName))
			{
				PathManager.checkDirectory(newName, this.myPlugins.getTopFrame());
				loadParkAndDeleteNodes();
			}
		}
		if (this.myParkFile != null)
		{
			if (this.myParkFile.equals(oldFileName))
			{
				this.myPlugins.getPathManager().setDirectory(PathManager.DIR_PARK,
						newName);
				loadParkAndDeleteNodes();
			}
		}
	}

	public boolean isParkActive()
	{
		if (this.myParkNode != null) { return true; }
		return false;
	}

	public boolean isDeleteActive()
	{
		if (this.myDeleteNode != null) { return true; }
		return false;
	}

	public void moveOrCopySeveral(boolean moveNotCopy, DirectoryTreeNode node, 
			Object[] filesPicked)
	{
		 // This is not called from delete or park buttons so we always
		 // want to set repeat back to whatever directory it came from. 
		this.myMoveManager.moveOrCopySeveral(moveNotCopy, node, null, filesPicked, true);
	}
	
	
	/**
	 * Safe operation.  Can pass in a null.  
	 * Repeated calls with same directory node is no problem.
	 */
	public void setRepeatDir(DirectoryTreeNode dirNode)
	{
		if (dirNode != null && this.myRepeatNode != dirNode)
		{
			this.myRepeatNode = dirNode;
			this.myRepeatDirAsString = this.myRepeatNode.getAbsolutePath();
			this.myRepeatFile = new File(this.myRepeatDirAsString);
			this.myRepeatBtn.getAction().setEnabled(true);
			this.myRepeatBtn
					.getAction()
					.putValue(
							Action.SHORT_DESCRIPTION,
							"<html>" + AppStarter.getString("SortScreen.menu.file.repeat.moved") //$NON-NLS-1$ //$NON-NLS-2$
									+ "<br>" + this.myRepeatDirAsString + "</html>"); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	public void loadParkAndDeleteNodes()
	{
		this.myDeleteDirAsString = null;
		this.myDeleteFile = null;
		this.myDeleteNode = null;

		this.myDeleteDirAsString = this.myPlugins.getPathManager().getDirectoryAbsolute(
				PathManager.DIR_DELETE);
		if (this.myDeleteDirAsString == null)
		{
			this.myDeleteBtn.getAction().setEnabled(false);
			this.myDeleteBtn.getAction().putValue(Action.SHORT_DESCRIPTION,
					AppStarter.getString("SortScreen.menu.file.trash.tip")); //$NON-NLS-1$
		}
		else
		{
			this.myDeleteFile = new File(this.myDeleteDirAsString);
			this.myDeleteBtn.getAction().setEnabled(true);
			this.myDeleteBtn
					.getAction()
					.putValue(
							Action.SHORT_DESCRIPTION,
							"<html>" + AppStarter.getString("SortScreen.menu.file.trash.tip") //$NON-NLS-1$ //$NON-NLS-2$
									+ "<br>" + this.myDeleteDirAsString + "</html>"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		this.myParkDirAsString = null;
		this.myParkFile = null;
		this.myParkNode = null;
		this.myParkDirAsString = this.myPlugins.getPathManager().getDirectoryAbsolute(
				PathManager.DIR_PARK);
		if (this.myParkDirAsString == null)
		{
			this.myParkBtn.getAction().setEnabled(false);
			this.myParkBtn.getAction().putValue(Action.SHORT_DESCRIPTION,
					AppStarter.getString("SortScreen.menu.file.park.tip")); //$NON-NLS-1$
		}
		else
		{
			this.myParkFile = new File(this.myParkDirAsString);
			this.myParkBtn.getAction().setEnabled(true);
			this.myParkBtn.getAction().putValue(Action.SHORT_DESCRIPTION,
					"<html>" + AppStarter.getString("SortScreen.menu.file.park.tip") //$NON-NLS-1$ //$NON-NLS-2$
							+ "<br>" + this.myParkDirAsString + "</html>"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		rescan();
	}

	public void rescan()
	{
		//System.out.println("Rescanning");
		this.myDeleteNode = null;
		this.myParkNode = null;
		this.myRepeatNode = null;

		DirectoryTreeNode root = this.myTreeManager.getRootNode();
		if (root == null)
		{
			System.out
					.println(">>>>>>>>................>>>>>>>>>>>>>> RDPManager.rescan() got a null!!!  FIX ME!!!");
			return;
		}
		scanADirectory(root);

		Enumeration<?> breadth = root.breadthFirstEnumeration();
		while (breadth.hasMoreElements())
		{
			Object obj = breadth.nextElement();
			DirectoryTreeNode node = (DirectoryTreeNode) obj;
			scanADirectory(node);
			if (this.myDeleteNode != null && this.myParkNode != null && this.myRepeatNode != null)
			{
				break;
			}
		}
	}

	/**
	 * Fails silently if there is a problem adding the diretory tree node.
	 * @param toDirNode
	 */
	public void addToRememberedMoveList(DirectoryTreeNode toDirNode)
	{
		if (toDirNode == null) { return; }
		// Set up a file to hold the drop directory.
		String toDirPath = toDirNode.getAbsolutePath();
		addDirectoryToList(toDirPath);
	}

	public void addDirectoryToList(String toDirPath)
	{
		File toDir = new File(toDirPath);
		if (!toDir.exists() || toDir.isFile()) { return; }
		// Move latest selection to the top.
		int index = this.myDirectoryList.indexOf(toDir);
		if (index > -1)
		{
			this.myDirectoryList.remove(index);
		}
		this.myDirectoryList.add(0, toDir);
		flushList(false);
	}


	public void directoryListPressed()
	{
		new DirectoryListPopup(this.myPlugins.getTopFrame(), this, this.myPlugins
				.getExtendedDirectoryTree());
	}

	public void emptyTrash()
	{
		if (this.myDeleteDirAsString == null) { return; }
		File deleteDir = new File(this.myDeleteDirAsString);
		if (!deleteDir.isDirectory()) { return; }

		File[] delList = deleteDir.listFiles();
		if (delList != null)
		{
			BoundedRangeModel range = new DefaultBoundedRangeModel();
			EmptyTrash doIt = new EmptyTrash(delList, range);
			ModalProgressDialogFlex
					.popUp(
							this.myPlugins.getTopFrame(),
							AppStarter.getString("RDPmanager.23"), delList.length //$NON-NLS-1$
									+ " " + AppStarter.getString("RDPmanager.24"), AppStarter.getString("RDPmanager.25"), //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
							range, doIt, 750);
		}
	}

	public void moveFilesToDirectory(Object obj, DirectoryTree tree)
	{
		if (obj == null || !(obj instanceof File))
		{
			System.out.println("moveFilesToDirectory got a non file");
			return;
		}
		File toDirectory = (File) obj;
		if (!toDirectory.isDirectory())
		{
			System.out.println("moveFilesToDirectory got a non directory file");
			return;
		}

		Object[] filesPicked = this.myIScrollDirTree.getFileList().getSelectedValues();
		if (tree != null)
		{
			DirectoryTreeNode node = tree.findANode(toDirectory);
			if (node != null)
			{
				if (filesPicked.length > 0)
				{
					this.myMoveManager.moveOrCopySeveral(true, node, toDirectory
							.getAbsolutePath(), filesPicked,
							updateRepeateButtonBecauseDeleteOrParkNotPressed);
				}
				else
				{
					this.myMoveManager.moveCurrentFile(node);
				}
				this.addToRememberedMoveList(node);
				return;
			}
		}

		// You only get to here if no node was found.
		if (filesPicked.length > 0)
		{
			// Don's update repeat button because nothing happened.
			this.myMoveManager.moveOrCopySeveral(true, null, toDirectory
					.getAbsolutePath(), filesPicked, doNotUpdateRepeateButton);
		}
		else
		{
			this.myMoveManager.moveCurrentFile(toDirectory);
		}
		this.addDirectoryToList(toDirectory.toString());
	}

	public void flushList(boolean clearListFirst)
	{
		if (clearListFirst)
		{
			this.myDirectoryList = new ArrayList<File>(20);
		}
		try
		{
			this.myPrefs.putByteArray(DIRECTORY_LIST_KEY, Misc
					.toByteArray(this.myDirectoryList));
			this.myPrefs.flush();
		}
		catch (Exception ex)
		{
			// eat it
		}
	}

	public void addButtonsToToolbar(JToolBar toolbar)
	{
		toolbar.add(this.myDeleteBtn);
		toolbar.add(this.myParkBtn);
		toolbar.add(this.myRepeatBtn);
		toolbar.add(this.myDirectoryListBtn);
	}

	// ==========================================================================
	// Actual Moves
	// ==========================================================================	
	public void movePressed()
	{
		doRDPmove(this.myParkNode, this.myParkDirAsString, this.myParkFile, 
				updateRepeateButtonBecauseDeleteOrParkNotPressed,
				AppStarter.getString("RDPmanager.3"));
	}
	

	public void deletePressed()
	{
		doRDPmove(this.myDeleteNode, this.myDeleteDirAsString, this.myDeleteFile, 
				!updateRepeateButtonBecauseDeleteOrParkNotPressed,
				AppStarter.getString("RDPmanager.19"));
	}

	public void repeatPressed()
	{
		doRDPmove(this.myRepeatNode, this.myRepeatDirAsString, this.myRepeatFile, 
				!updateRepeateButtonBecauseDeleteOrParkNotPressed,
				AppStarter.getString("RDPmanager.21"));
	}
	
	public void doRDPmove(DirectoryTreeNode toNode, String toDirAsString, File fileToMove, 
			boolean updateButton, String confirmationDialogLabel)
	{
		Object[] filesPicked = this.myIScrollDirTree.getFileList().getSelectedValues();

		// First check for moving from the File List
		int confirmation = this.myMoveManager.confirmMove(
				confirmationDialogLabel, //AppStarter.getString("RDPmanager.21"), 
				toNode, //this.myRepeatNode, 
				toDirAsString, //this.myRepeatDirAsString, 
				filesPicked);
		switch (confirmation)
		{
		case MoveManager.MOVE_NONE:
			return;
		case MoveManager.MOVE_ONE:
			// If you get here, do a single move.
			if (toNode != null)
			{
				this.myMoveManager.moveCurrentFile(toNode);
			}
			else if (toDirAsString != null)
			{
				this.myMoveManager.moveCurrentFile(fileToMove); //this.myRepeatFile);
			}
			else
			{
				//System.out.println("Repeat Pressed with empty directory");
			}
			return;
		case MoveManager.MOVE_MANY:
			this.myMoveManager.moveOrCopySeveral(true, 
					toNode, toDirAsString, 
					filesPicked,
					updateButton); //$NON-NLS-1$
			//System.out.println("RDPmanager.MoveManager.MOVE_MANY DONE");
			return;
		}
	}

	
	// ==========================================================================
	// Private
	// ==========================================================================

	private void scanADirectory(DirectoryTreeNode node)
	{
		// These are all "if" not "if else"
		// because one directory could be all three.
		// System.out.println("scanADirectory: " + myDeleteDirAsString + " " +
		// node.getAbsolutePath());
		File nodeFile = node.getFile();
		if (nodeFile != null)
		{
			if (this.myDeleteNode == null && nodeFile.equals(this.myDeleteFile))
			{
				this.myDeleteNode = node;
			}

			if (this.myParkNode == null && nodeFile.equals(this.myParkFile))
			{
				this.myParkNode = node;
			}
			if (this.myRepeatNode == null && nodeFile.equals(this.myRepeatFile))
			{
				this.myRepeatNode = node;
			}
		}
	}

	public JButton makeRepeatButton(ActiveActions aa)
	{
		Action menuAction = aa.getAction("file", "repeat"); //$NON-NLS-1$ //$NON-NLS-2$
		Action toolbarAction = new RepeatToolbarAction();
		toolbarAction.putValue(Action.NAME, menuAction.getValue(Action.NAME));
		toolbarAction.putValue(Action.SHORT_DESCRIPTION, menuAction
				.getValue(Action.SHORT_DESCRIPTION));
		return ActiveActions.makeToolBarButton(toolbarAction);
	}

	public JButton makeParkButton(ActiveActions aa)
	{
		Action menuAction = aa.getAction("file", "park"); //$NON-NLS-1$ //$NON-NLS-2$
		Action toolbarAction = new ParkToolbarAction();
		toolbarAction.putValue(Action.NAME, menuAction.getValue(Action.NAME));
		toolbarAction.putValue(Action.SHORT_DESCRIPTION, menuAction
				.getValue(Action.SHORT_DESCRIPTION));
		return ActiveActions.makeToolBarButton(toolbarAction);
	}

	public JButton makeDeleteButton(ActiveActions aa)
	{
		Action menuAction = aa.getAction("file", "trash"); //$NON-NLS-1$ //$NON-NLS-2$
		Action toolbarAction = new DeleteToolbarAction();
		toolbarAction.putValue(Action.NAME, menuAction.getValue(Action.NAME));
		toolbarAction.putValue(Action.SHORT_DESCRIPTION, menuAction
				.getValue(Action.SHORT_DESCRIPTION));
		return ActiveActions.makeToolBarButton(toolbarAction);
	}

	public JButton makeDirectoryListButton(ActiveActions aa)
	{
		Action menuAction = aa.getAction("file", "directoryList"); //$NON-NLS-1$ //$NON-NLS-2$
		Action toolbarAction = new DirectoryListToolbarAction();
		toolbarAction.putValue(Action.NAME, menuAction.getValue(Action.NAME));
		toolbarAction.putValue(Action.SHORT_DESCRIPTION, menuAction
				.getValue(Action.SHORT_DESCRIPTION));
		return ActiveActions.makeToolBarButton(toolbarAction);
	}


	class DeleteToolbarAction extends AbstractAction
	{
		DeleteToolbarAction()
		{
			super(null, Icons.getIcon(Icons.DELETE));
		}

		public void actionPerformed(ActionEvent ae)
		{
			/*
			if (((ae.getModifiers() & ActionEvent.CTRL_MASK) != 0))
			{
				mySortScreen.deletePressed(true);
			}
			else
			{
				mySortScreen.deletePressed(false);
			}
			*/
			RDPmanager.this.deletePressed();
		}
	}

	class ParkToolbarAction extends AbstractAction
	{
		ParkToolbarAction()
		{
			super(null, Icons.getIcon(Icons.PARK));
		}

		public void actionPerformed(ActionEvent ae)
		{
			RDPmanager.this.movePressed();
		}
	}

	class RepeatToolbarAction extends AbstractAction
	{
		RepeatToolbarAction()
		{
			super(null, Icons.getIcon(Icons.REPEAT));
		}

		public void actionPerformed(ActionEvent ae)
		{
			RDPmanager.this.repeatPressed();
		}
	}

	class DirectoryListToolbarAction extends AbstractAction
	{
		DirectoryListToolbarAction()
		{
			super(null, Icons.getIcon(Icons.DIRECTORY_LIST));
		}

		public void actionPerformed(ActionEvent ae)
		{
			RDPmanager.this.directoryListPressed();
		}
	}

	// ==========================================================================
	// Inner Classes
	// ==========================================================================
	class EmptyTrash implements Runnable
	{
		File[] myDelList;

		BoundedRangeModel myRange;

		EmptyTrash(File[] delList, BoundedRangeModel range)
		{
			this.myDelList = delList;
			this.myRange = range;
		}

		public void run()
		{
			this.myRange.setMaximum(this.myDelList.length);
			for (int i = 0; i < this.myDelList.length; i++)
			{
				try
				{
					this.myDelList[i].delete();
					this.myRange.setValue(i);
				}
				catch (Exception ex)
				{
					JOptionPane
							.showMessageDialog(RDPmanager.this.myPlugins.getTopFrame(), AppStarter
									.getString("RDPmanager.26") + this.myDelList[i] + "\n" //$NON-NLS-1$ //$NON-NLS-2$
									+ ex.getMessage(), AppStarter
									.getString("RDPmanager.28"), //$NON-NLS-1$
									JOptionPane.WARNING_MESSAGE);
				}
			}
			if (RDPmanager.this.myDeleteNode != null)
			{
				RDPmanager.this.myDeleteNode.recount();
				RDPmanager.this.myPlugins.getExtendedDirectoryTree().treeDidChange();

				// In case delete was showing, reshow it empty.
				RDPmanager.this.myPlugins.reloadScroller();
			}
		}
	}
}
