/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.dirtree.rdp;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;

import org.gerhardb.lib.scroller.ScrollerChangeEvent;
import org.gerhardb.lib.scroller.ScrollerListener;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.util.startup.AppStarter;

import com.saic.isd.printscreen.Util;

public class StatusBarManager implements ScrollerListener
{
	ListShowTreeCoordinator myLST;
	JLabel myStatusCount = new JLabel();
	JTextField myStatusFileName = new JTextField();
	StatusBarPopup myStatusBarPopup;
	
	
	public StatusBarManager(ListShowTreeCoordinator lst)
	{
		this.myLST = lst;
		this.myStatusFileName.setEditable(false);
		this.myStatusFileName.addMouseListener(new MouseAdpt());	
		this.myLST.getScroller().addScrollerListener(this);
	}
	
	// ==========================================================================
	// ScrollerListener
	// ==========================================================================
	public void scrollerChanged(ScrollerChangeEvent ce)
	{
		statusCurrentPage();
	}

	// ==========================================================================
	// Other
	// ==========================================================================
	public JPanel getStatusPanel()
	{
		JPanel statusPanel = new JPanel(new BorderLayout());
		statusPanel.add(this.myStatusCount, BorderLayout.WEST);
		statusPanel.add(this.myStatusFileName, BorderLayout.CENTER);
		return statusPanel;
	}

	public AutocopyFilenamePanel getAutocopyFilenamePanel()
	{
		return new AutocopyFilenamePanel();
	}
	/**
	 * Set status bar to current page.
	 */ 
	public void statusCurrentPage()
	{
    	// added 12 Jan 2008 for: Exception in thread "Thread-17" java.lang.Error: Interrupted attempt to aquire write lock
		// It locked application up so this will hopefully prevent lockup in the future.
		try 
		{
			this.myStatusCount.setText(this.myLST.getScroller().getStatusBarCount() + "  "); //$NON-NLS-1$
			this.myStatusFileName.setText(this.myLST.getScroller().getStatusBarFileName());
			if (isAutocopyFilenameEnabled())
			{
				Util.copyToClipboard(this.myStatusFileName.getText());
			}		
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
	}
	
	private static final String AUTO_COPY_FILE_NAME_ENABLED = "autocopyFilename"; //$NON-NLS-1$

	boolean isAutocopyFilenameEnabled()
	{
		return this.myLST.getPreferences().getBoolean(AUTO_COPY_FILE_NAME_ENABLED, false);
	}

	void setAutocopyFilenameEnabled(boolean enabled)
	{
		this.myLST.getPreferences().putBoolean(AUTO_COPY_FILE_NAME_ENABLED, enabled);
	}

	// --------------------------------------------------------------------------
	// Inner Classes
	// --------------------------------------------------------------------------
	private class MouseAdpt extends MouseAdapter
	{
		protected MouseAdpt() { /* just to get rid of synthetic warning...*/ } 
		
		@Override
		public void mouseReleased(MouseEvent e)
		{
			if (StatusBarManager.this.myStatusBarPopup != null)
			{
				StatusBarManager.this.myStatusBarPopup.menuSelectionChanged(false);
				StatusBarManager.this.myStatusBarPopup = null;
			}
			else
			{
				StatusBarManager.this.myStatusBarPopup = new StatusBarPopup();
				StatusBarManager.this.myStatusBarPopup.show((Component) e.getSource(), e.getX(), e.getY());
			}
		}
	}
	
	/**
	 * Derived from code originally presented in
	 * Core Swing Advanced Programming
	 * by Kim Topley, Printice Hall PTR, chapter 8.
	 */
	class StatusBarPopup extends JPopupMenu
	{
	   /**
	    * This is what you get when you right click on the status bar showing
	    * the file name.
	    * @param tree popuped up on
	    * @param path path
	    */
	   public StatusBarPopup()
	   {
	      super();
	      
	      JPanel center = new JPanel(new FlowLayout(FlowLayout.CENTER));
	      center.add(new JLabel(AppStarter.getString("FileNamePopUp.0")), 0);  //$NON-NLS-1$
	      insert(center, 0);
	      addSeparator();
	
	      JMenuItem copy = add(AppStarter.getString("FileNamePopUp.1"));  //$NON-NLS-1$
	      JMenuItem autoCopy = null;
	      if (isAutocopyFilenameEnabled())
	      {
	      	autoCopy = add(AppStarter.getString("FileNamePopUp.2")); //$NON-NLS-1$
	      }
	      else
	      {
	      	autoCopy = add(AppStarter.getString("FileNamePopUp.3")); //$NON-NLS-1$
	      }
	      
	      copy.addActionListener(
	         new ActionListener()
	      {
	         public void actionPerformed(ActionEvent event)
	         {
	            Util.copyToClipboard(StatusBarManager.this.myStatusFileName.getText());
	         }
	      });
	
	      autoCopy.addActionListener(
	            new ActionListener()
	         {
	            public void actionPerformed(ActionEvent event)
	            {
	            	// Just flips from one state to the other.
	            	if (isAutocopyFilenameEnabled())
	            	{
	            		setAutocopyFilenameEnabled(false);
	            	}
	            	else
	            	{
	            		setAutocopyFilenameEnabled(true);   
	                  Util.copyToClipboard(StatusBarManager.this.myStatusFileName.getText());         		
	            	}
	            }
	         });
	   }
	}
	
   public class AutocopyFilenamePanel extends JPanelRows
   {
      JCheckBox myAutocopyBox = new JCheckBox("Auto-copy current file name to clipboard on each selection"); 
      
      AutocopyFilenamePanel()
   	{
      	this.myAutocopyBox.setSelected(isAutocopyFilenameEnabled());	
	      
	      JPanel aRow = super.topRow();
	      aRow.add(this.myAutocopyBox);
   	}
   	
		public void save()
		{
			setAutocopyFilenameEnabled(this.myAutocopyBox.isSelected());
      	if (isAutocopyFilenameEnabled())
      	{
            Util.copyToClipboard(StatusBarManager.this.myStatusFileName.getText());         		
      	}
		}
   }

}