/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import java.awt.image.BufferedImage;
import java.io.File;
import java.util.Iterator;
import java.util.Locale;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReadParam;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriter;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.metadata.IIOMetadataFormat;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageOutputStream;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

/**
 * This is used to read images and write them back out to a file.
 * This wraps the stream thereby keeping the Metadata.
 */
public class IOImageExample
{
   static
   {
      ImageIO.setUseCache(false);
   }

   // ==========================================================================
   //                            Main
   // ==========================================================================
   public static void main(String[] args)
   {
      writeTestNEW();
      //writeTestOLD();
      //readEverythingTest();
   }

   // ==========================================================================
   //                Write Test
   // ==========================================================================
   static void writeTestNEW()
   {
      /*
             String[] formats = ImageIO.getWriterFormatNames();
             for( int i=0; i<formats.length; i++ )
             {
         System.out.println(formats[i]);
             }
       */

      //[Exif] Date/Time - 2004:01:02 12:43:44
      File inFile = new File("D:/testpics/aaaa.jpg"); //$NON-NLS-1$
      File outFile = new File("D:/testpics/bbbb.jpg"); //$NON-NLS-1$
      try
      {
         // Really EZ way!!
         //IOImage inImage = new IOImage(inFile);
         //IIOMetadata meta = inImage.getMetaData();
         //BufferedImage image = inImage.getImage();
         //ImageIO.write(image,"jpeg",outFile);

         ImageInputStream iis = ImageIO.createImageInputStream(inFile);
         ImageReader reader = null;
         Iterator<ImageReader> readers = ImageIO.getImageReaders(iis);
         while (readers.hasNext())
         {
            reader = readers.next();
            System.out.println("Reader: " + reader.toString()); //$NON-NLS-1$
         }
         if (reader != null)
         {
	         reader.setInput(iis);
	         ImageReadParam readParam = null;
	         IIOImage image = reader.readAll(0, readParam);
	         showMeta(image.getMetadata());
	
	         ImageWriter writer = null;
	         Iterator<ImageWriter> writers = ImageIO.getImageWritersByFormatName("jpeg"); //$NON-NLS-1$
	         while (writers.hasNext())
	         {
	            writer = writers.next();
	            System.out.println("Writer: " + writer.toString()); //$NON-NLS-1$
	         }
	         if (writer != null)
	         {
		         ImageOutputStream ios = ImageIO.createImageOutputStream(outFile);
		         writer.setOutput(ios);
		         //writer.prepareWriteSequence(meta);
		         //writer.replaceStreamMetadata(meta);
		         writer.write(image);
	         }
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }

   static void writeTestOLD()
   {
      //[Exif] Date/Time - 2004:01:02 12:43:44
      File inFile = new File("D:/testpics/aaaa.jpg"); //$NON-NLS-1$
      File outFile = new File("D:/testpics/bbbb.jpg"); //$NON-NLS-1$
      try
      {
         IOImage inImage = new IOImageJava(inFile);
         IOImage outImage = new IOImageJava(outFile);
         BufferedImage image = inImage.getImage();
         outImage.saveJPEG(image, 1.0f);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }

   // ==========================================================================
   //                Read Everything Test
   // ==========================================================================
   static void readEverythingTest()
   {
      File file = new File("D:/testpics/aaaa.jpg"); //$NON-NLS-1$
      try
      {
         //IOImage ioImage = new IOImage(file);
         //IIOMetadata meta = ioImage.getMetaData();
         //showMeta(meta);

         ImageInputStream iis = ImageIO.createImageInputStream(file);
         Iterator<ImageReader> it = ImageIO.getImageReaders(iis);
         // Actually, there appears to be only one in Java 1.4.2
         while (it.hasNext())
         {
            ImageReader reader = it.next();
            System.out.println("############################################"); //$NON-NLS-1$
            System.out.println("READER: " + reader); //$NON-NLS-1$
            System.out.println("############################################"); //$NON-NLS-1$
            reader.setInput(iis);
            IIOMetadata meta = reader.getImageMetadata(reader.getMinIndex());
            showMeta(meta);
            reader.dispose();
         }
         try
         {
            iis.close();
         }
         catch (Exception ex)
         {
            ex.printStackTrace();
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }

   static void showMeta(IIOMetadata meta)
   {
      try
      {
         if (meta == null)
         {
            System.out.println("No Meta Data Available"); //$NON-NLS-1$
            return;
         }
         System.out.println("Reading Meta Data!!"); //$NON-NLS-1$
         System.out.println(
            "Native Metadata Format Name: " //$NON-NLS-1$
            + meta.getNativeMetadataFormatName());
         if (meta.isStandardMetadataFormatSupported())
         {
            System.out.println("Stadard Metadata Format IS Supported"); //$NON-NLS-1$
         }
         else
         {
            System.out.println("Stadard Metadata Format is NOT Supported"); //$NON-NLS-1$
         }

         System.out.println(
            "++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"); //$NON-NLS-1$
         System.out.println("Metadata Format Names: "); //$NON-NLS-1$
         String[] names = meta.getMetadataFormatNames();
         if (names != null)
         {
            for (int i = 0; i < names.length; i++)
            {
               /*
                   IIOMetadataFormat format = meta.getMetadataFormat(names[i]);
                               if (format != null)
                               {
                  String root = format.getRootName();
                  System.out.println("Root Name: " + root);
                  showKids(format, root, 0);
                               }
                */
               System.out.println(
                  "~~~~~~~~~~~~~~~~~~~~Printing DOM~~~~~~~~~~~~~~~~~~~~~~~~~~"); //$NON-NLS-1$
               System.out.println("Format Name: " + names[i]); //$NON-NLS-1$
               Node node = meta.getAsTree(names[i]);
               displayMetadata(node);
               System.out.println(
                  "----------------------------------------------------------"); //$NON-NLS-1$
            }
         }

         System.out.println(
            "++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++"); //$NON-NLS-1$
         System.out.println("Extra Metadata Format Names: "); //$NON-NLS-1$
         String[] extra = meta.getExtraMetadataFormatNames();
         if (extra != null)
         {
            for (int i = 0; i < extra.length; i++)
            {
               System.out.println(extra[i]);
            }
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }

   }

   static void showKids(IIOMetadataFormat format, String nodeName, int level)
   {
      String[] kids = format.getChildNames(nodeName);
      if (kids == null)
      {
         indent(level);
         System.out.println("No Children!"); //$NON-NLS-1$
         return;
      }
      for (int i = 0; i < kids.length; i++)
      {
         indent(level);
         System.out.println("Child Name: " + kids[i] + " -- " //$NON-NLS-1$ //$NON-NLS-2$
            + format.getElementDescription(kids[i], Locale.US));
         String[] attributeNames = format.getAttributeNames(kids[i]);
         if (attributeNames != null)
         {
            for (int j = 0; j < attributeNames.length; j++)
            {
               indent(level);
               System.out.println(j + ": " + attributeNames[j]); //$NON-NLS-1$
            }
         }
         showKids(format, kids[i], level + 1);
      }
   }

   static void indent(int level)
   {
      int spaces = level * 2;
      for (int i = 0; i < spaces; i++)
      {
         System.out.print("  "); //$NON-NLS-1$
      }
   }

   public static void displayMetadata(Node root)
   {
      displayMetadata(root, 0);
   }

   static void displayMetadata(Node node, int level)
   {
      indent(level); // emit open tag
      System.out.print("<" + node.getNodeName()); //$NON-NLS-1$
      NamedNodeMap map = node.getAttributes();
      if (map != null)
      { // print attribute values
         int length = map.getLength();
         for (int i = 0; i < length; i++)
         {
            Node attr = map.item(i);
            System.out.print(" " + attr.getNodeName() + //$NON-NLS-1$
               "=\"" + attr.getNodeValue() + "\""); //$NON-NLS-1$ //$NON-NLS-2$
         }
      }

      Node child = node.getFirstChild();
      if (child != null)
      {
         System.out.println(">"); // close current tag //$NON-NLS-1$
         while (child != null)
         { // emit child tags recursively
            displayMetadata(child, level + 1);
            child = child.getNextSibling();
         }
         indent(level); // emit close tag
         System.out.println("</" + node.getNodeName() + ">"); //$NON-NLS-1$ //$NON-NLS-2$
      }
      else
      {
         System.out.println("/>"); //$NON-NLS-1$
      }
   }

   /*
          try
          {
      JPanel center = new JPanel(new BorderLayout());
      JLabel originalLbl = new JLabel();
      originalLbl.setMinimumSize(new java.awt.Dimension(200, 100));
      JLabel scaledLbl = new JLabel();
      scaledLbl.setMinimumSize(new java.awt.Dimension(200, 100));
      JLabel rotatedLbl = new JLabel();
      scaledLbl.setMinimumSize(new java.awt.Dimension(200, 100));
      IOImage agent = new IOImage();
      BufferedImage image =
         //agent.open( new File( "E:/testpics/wide.jpg" ) );
         //agent.open( new File( "E:/testpics/high.jpg" ) );
         agent.open(new File("E:/testpics/box.jpg"));
      //agent.open( new File( "f:/gerhard/ann/flower.jpg" ) );
      originalLbl.setIcon(new ImageIcon(image));
      BufferedImage scaledImage = agent.scale(image, 0.5f);
      scaledLbl.setIcon(new ImageIcon(scaledImage));
      //BufferedImage rotatedImage = agent.rotate( scaledImage, 0 );
      BufferedImage rotatedImage = agent.rotateRight(scaledImage);
      rotatedLbl.setIcon(new ImageIcon(rotatedImage));
      agent.save(rotatedImage, new File("E:/testpics/savetest.jpg"), 1.0f);
      JFrame f = new JFrame("File Tree Test");
      f.getContentPane().setLayout(new BorderLayout());
      f.getContentPane().add(originalLbl, BorderLayout.WEST);
      f.getContentPane().add(center, BorderLayout.CENTER);
      center.add(scaledLbl, BorderLayout.NORTH);
      center.add(rotatedLbl, BorderLayout.SOUTH);
      f.pack();
      //f.setSize(900, 300);
      f.addWindowListener(
         new java.awt.event.WindowAdapter()
      {
         public void windowClosing(java.awt.event.WindowEvent evt)
         {
            System.exit(0);
         }
      });
      f.setVisible(true);
          }
          catch (Exception ex)
          {
      System.out.println(ex.getMessage());
          }
          System.out.println("Main Exiting Normally");
       } // Main method
    */
}
