/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.image;

import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import javax.imageio.ImageIO;
import javax.imageio.stream.ImageInputStream;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.io.FileUtil;

/**
 * This is used to read images and write them back out to a file.
 * This wraps the stream thereby keeping the Metadata.
 */
class IOImageJava extends IOImage
{
   static
   {
      ImageIO.setUseCache(false);
   }

   // ==========================================================================
   //                            Constructor
   // ==========================================================================
   public IOImageJava(File file) throws Exception
   {
      super(file);
      ImageInputStream iis = ImageIO.createImageInputStream(this.myFile);
      // Can happen on out of memory issues.
      if (iis != null)
      {
	      this.myImage = ImageIO.read(iis);
	      if (this.myImage != null)
	      {
		      try
		      {
		         // This will cause an exception if the file is not a valid image.
		         this.myImage.getWidth();
		         // If you close the input stream here, it has not fully read the
		         // file, so you will never see the picture.
		      }
		      catch (Exception x)
		      {
		         throw new IOException(AppStarter.getString("IOImage.1")); //$NON-NLS-1$
		      }
	      }
      }
   }

   // ==========================================================================
   //                        Super Class Implementation
   // ==========================================================================
   
   @Override
	public String getImageFactoryName()
   {
   	return "Java-ImageIO"; //$NON-NLS-1$
   }
   
   @Override
	public void save(BufferedImage inputImage, File saveAsFile)
   throws IOException
	{
   	String ext = FileUtil.getExtension(saveAsFile.getName()).toLowerCase();
   	//System.out.println("IOImage ext: " + ext); //$NON-NLS-1$
   	ImageUtil.save(inputImage, 1.0f, saveAsFile, ext);
   	
   	/*
      //Iterator writers = ImageIO.getImageWritersByFormatName("png");
      Iterator writers = ImageIO.getImageWritersByFormatName(ext);
      ImageWriter writer = (ImageWriter)writers.next();
      ImageOutputStream ios = ImageIO.createImageOutputStream(saveAsFile);
      writer.setOutput(ios);
      writer.write(inputImage);
      ios.close();
      writer.dispose();
      */
	}
   
   // ==========================================================================
   //                            Failed Methods
   // ==========================================================================
   /**
    * THIS FUNCTION IS NOT USED.
    * INSTEAD WE USE THE NOAKES LIBRARIES.
    * @return
    * @throws IOException
    */
   /*
   IIOMetadata getMetaData()
      throws IOException
   {
      IIOMetadata rtnMe = null;
      ImageInputStream iis = ImageIO.createImageInputStream(myFile);
      Iterator it = ImageIO.getImageReaders(iis);
      // Actually, there appears to be only one in Java 1.4.2
      while (it.hasNext())
      {
         ImageReader reader = (ImageReader)it.next();
         //System.out.println("#### READER: " + reader);
         reader.setInput(iis);
         rtnMe = reader.getImageMetadata(reader.getMinIndex());
         reader.dispose();
      }
      try
      {
         iis.close();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      return rtnMe;
   }

   void exifSaveJPEG(BufferedImage image, float quality)
      throws IOException
   {
      // First, read the old image to get the EXIF data.
      byte[] exif = getExif();

      //System.out.println("Got EXIF: " + exif.length);

      // Save the file
      saveJPEG(image, quality);

      // Insert the EXIF data.
      if (exif.length > 0)
      {
         //System.out.println("Putting EXIF: " + exif.length);
         putExif(exif);
      }
   }
*/
   // ==========================================================================
   //                            Private Methods
   // ==========================================================================
   /*
   private byte[] getExif()
      throws IOException
   {
      byte[] fileByte = new byte[(int)myFile.length()];
      FileInputStream reader = new FileInputStream(myFile);
      reader.read(fileByte, 0, fileByte.length);
      reader.close();

      int start = 0;
      int end = 0;

      // Search for markers
      int searchEnd = fileByte.length - 1;
      for (int i = 0; i < searchEnd; i++)
      {
         //System.out.println("Bypte: " + fileByte[i]);
         if (fileByte[i] == -1)//0xff)
         {
            //System.out.println("Found ff " + i);
            // Start at first app segment.
            if (start == 0)
            {
               if (isAppMarker(fileByte[i + 1]))
               {
                  //System.out.println("Found Start at: " + i);
                  start = i;
               }
            }
            // Once started, stop at first non app segment.
            else if (start > 0 && end == 0)
            {
               if (!isAppMarker(fileByte[i + 1]))
               {
                  //System.out.println("Found End at: " + i);
                  end = i;
                  break;
               }
            }
         }
      }

      if (start == 0)
      {
         return new byte[0];
      }
      if (end == 0)
      {
         throw new IOException("Exif end marker not found"); //$NON-NLS-1$
      }
      byte[] rtnMe = new byte[end - start];
      int put = 0;
      for (int i = start; i < end; i++)
      {
         rtnMe[put] = fileByte[i];
         put++;
      }
      return rtnMe;
   }

   private boolean isAppMarker(byte aByte)
   {
      //System.out.println("xxx: " + aByte);
      int test = aByte;
      if (test < 0)
      {
         test = test + 256;
      }
      //System.out.println("yyy: " + test);

      if (test == 0xe0)
      {
         return true;
      }
      if (test == 0xe1)
      {
         return true;
      }
      if (test == 0xe2)
      {
         return true;
      }
      if (test == 0xe3)
      {
         return true;
      }
      if (test == 0xe4)
      {
         return true;
      }
      if (test == 0xe5)
      {
         return true;
      }
      if (test == 0xe6)
      {
         return true;
      }
      if (test == 0xe7)
      {
         return true;
      }
      if (test == 0xe8)
      {
         return true;
      }
      if (test == 0xe9)
      {
         return true;
      }
      if (test == 0xea)
      {
         return true;
      }
      if (test == 0xeb)
      {
         return true;
      }
      if (test == 0xec)
      {
         return true;
      }
      if (test == 0xed)
      {
         return true;
      }
      if (test == 0xee)
      {
         return true;
      }
      if (test == 0xef)
      {
         return true;
      }
      return false;
   }

   private void putExif(byte[] exif)
      throws IOException
   {
      byte[] fileByte = new byte[(int)myFile.length()];
      // FileReader is for character, not binary data.
      FileInputStream reader = new FileInputStream(myFile);
      reader.read(fileByte, 0, fileByte.length);
      reader.close();

      int putHere = 0;

      // Search for markers
      int searchEnd = fileByte.length - 1;
      // Skip first marker.
      for (int i = 2; i < searchEnd; i++)
      {
         if (fileByte[i] == -1)//0xff)
         {
            if (!isAppMarker(fileByte[i + 1]))
            {
               putHere = i;
               //System.out.println("Put Here: " + putHere);
               break;
            }
         }
      }
      if (putHere == 0)
      {
         throw new IOException("Quantization table marker not found"); //$NON-NLS-1$
      }
      FileOutputStream writer = new FileOutputStream(myFile);
      for (int i = 0; i < putHere; i++)
      {
         writer.write(fileByte[i]);
      }
      for (int i = 0; i < exif.length; i++)
      {
         writer.write(exif[i]);
      }
      for (int i = putHere; i < fileByte.length; i++)
      {
         writer.write(fileByte[i]);
      }
      writer.close();
   }
*/
   // ==========================================================================
   //                            Test Bed
   // ==========================================================================
   public static void main(String[] args)
   {
      try
      {
      	IOImageJava image =
            new IOImageJava(new File("d:/testpics/aaa.jpg")); //$NON-NLS-1$
         BufferedImage imgBuff = image.getImage();
         image.save(imgBuff);
         System.out.println("Image Saved!"); //$NON-NLS-1$
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }
}