/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.io;

import java.awt.BorderLayout;
import java.awt.Cursor;
import java.awt.FlowLayout;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.prefs.Preferences;

import javax.swing.*;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.swing.JPanelRows;

import com.saic.isd.swing.filechooser.SimpleFileFilter;

public class DirListController
{
   Preferences myPrefs;
   private static final String LAST_SAVE = "-LastSave"; //$NON-NLS-1$
   private static final String AUTO_RESTORE = "-AutoRestore"; //$NON-NLS-1$

   String myPreferencesLabel;
   private String myPopupName;
   JLabel myStatusBar = new JLabel(" ");

   DirListModel myDirListModel = new DirListModel(this.myStatusBar);
   DragDropList myDirList = new DragDropList(this.myDirListModel, false);
   JScrollPane myDirScroll = new JScrollPane(this.myDirList);
   JPanel myDirPanel = new JPanel(false);

   String myLastSave;
   Clearable myClearable;
   boolean iJustShowKids = false;
   
   /*
    =============================================================================
                              Constructor
    =============================================================================
    */
   
   public interface Clearable
   {
   	public void clear();
   }

   public DirListController(String popupName, String preferencesLabel, boolean justShowKids, String appName)
   {
   	this((Clearable)null, popupName, preferencesLabel, justShowKids, appName);
   }
  
   /**
    * @param popupName Name of the pop up.  getName()
    * @param preferencesLabel Used to store the preferences.  Use file name of the file calling this class.
    * @param justShowKids False if you want to show the selected directory also.
    */
   public DirListController(Clearable clear, String popupName, String preferencesLabel, boolean justShowKids, String appName)
   {
   	this.myClearable = clear;
      this.myPopupName = popupName;
      this.myPreferencesLabel = preferencesLabel;
      this.iJustShowKids = justShowKids;
      this.myPrefs = Preferences.userRoot().node("/" + appName + "/org/gerhardb/lib/io/DirListController");
      this.myDirList.getSelectionModel().setSelectionMode(
         ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
      layout();
   }

   /*
    =============================================================================
                             Public Functions
    =============================================================================
    */

   public void add(File[] files)
   {
   	this.myDirPanel.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      for (int i = 0; i < files.length; i++)
      {
         this.myDirListModel.addDirectory(files[i]);
      }
      this.myDirPanel.setCursor(Cursor.getDefaultCursor());
   }

   public void add(File file)
   {
     this.myDirListModel.addDirectory(file);
    }

   public DirListModel getModel()
   {
      return this.myDirListModel;
   }

   public File[] getDirArray()
   {
      int dirCount = this.myDirListModel.getSize();
      File[] dirArray = new File[dirCount];
      for (int i = 0; i < dirCount; i++)
      {
         dirArray[i] = (File)this.myDirListModel.getElementAt(i);
      }
      return dirArray;
   }

   public JPanel getPanel()
   {
      return this.myDirPanel;
   }

   /*
    =============================================================================
                             SAVE Functions
    =============================================================================
    */
   void saveDirectoryList()
   {
      this.myStatusBar.setText(" ");
      JFileChooser chooser = new JFileChooser();
      chooser.setDialogTitle(this.myPopupName + ": " + AppStarter.getString("DirListController.3")); //$NON-NLS-1$
      chooser.setFileFilter(new SimpleFileFilter("txt", "Text file")); //$NON-NLS-1$ //$NON-NLS-2$
      chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
      // In case of problem with myLastSave
      chooser.setCurrentDirectory(
         new File(getSaveDirectory()));
      if (this.myLastSave != null)
      {
         File lastFile = new File(this.myLastSave);
         if (lastFile.exists())
         {
            chooser.setCurrentDirectory(lastFile);
            if (lastFile.isFile())
            {
               chooser.setSelectedFile(lastFile);
            }
            else
            {
               chooser.setSelectedFile(new File("*.txt")); //$NON-NLS-1$
            }
         }
      }
      else
      {
         chooser.setSelectedFile(new File("*.txt")); //$NON-NLS-1$
      }
      int returnVal = chooser.showSaveDialog(this.myDirPanel);
      if (returnVal == JFileChooser.APPROVE_OPTION)
      {
         try
         {
            File saveAsFile = chooser.getSelectedFile();
            if (saveAsFile.exists())
            {
               int answer = JOptionPane.showConfirmDialog(
                            this.myDirPanel,
                            AppStarter.getString("DirListController.8"), AppStarter.getString("DirListController.9"), //$NON-NLS-1$ //$NON-NLS-2$
                            JOptionPane.YES_NO_OPTION);
               if (answer == JOptionPane.NO_OPTION)
               {
                  return;
               }

            }
            //System.out.println(saveAsPath);
            String newLine = System.getProperty("line.separator"); //$NON-NLS-1$
            try
            {
               FileWriter writer = new FileWriter(saveAsFile);
               int size = this.myDirListModel.getSize();
               for (int i = 0; i < size; i++)
               {
                  Object entry = this.myDirListModel.getElementAt(i);
                  writer.write(entry + newLine);
               }
               writer.flush();
               writer.close();
               rememberSave(saveAsFile);
            }
            catch (Exception ex)
            {
               ex.printStackTrace();
               JOptionPane.showMessageDialog(
                  this.myDirPanel,
                  AppStarter.getString("DirListController.11") //$NON-NLS-1$
                  + ex.getMessage(),
                  AppStarter.getString("DirListController.12"), //$NON-NLS-1$
                  JOptionPane.ERROR_MESSAGE);
            }
         }
         catch (Exception ex)
         {
            // So remote as to ignore
            ex.printStackTrace();
         }
      }
   }

   void restoreDirectoryList()
   {
      this.myStatusBar.setText(" ");
      JFileChooser chooser = new JFileChooser();
      chooser.setDialogTitle(this.myPopupName + ": " + AppStarter.getString("DirListController.13")); //$NON-NLS-1$
      chooser.setFileFilter(new SimpleFileFilter("txt", AppStarter.getString("DirListController.15"))); //$NON-NLS-1$ //$NON-NLS-2$
      //chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);
      chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
      // In case of problem with myLastSave
      chooser.setCurrentDirectory(
         new File(getSaveDirectory()));
      if (this.myLastSave != null)
      {
         File lastFile = new File(this.myLastSave);
         if (lastFile.exists())
         {
            chooser.setCurrentDirectory(lastFile);
            if (lastFile.isFile())
            {
               chooser.setSelectedFile(lastFile);
            }
            else
            {
               chooser.setSelectedFile(new File("*.txt")); //$NON-NLS-1$
            }
         }
      }
      else
      {
         chooser.setSelectedFile(new File("*.txt")); //$NON-NLS-1$
      }
      int returnVal = chooser.showOpenDialog(this.myDirPanel);
      if (returnVal == JFileChooser.APPROVE_OPTION)
      {
         File saveAsFile = chooser.getSelectedFile();
         restoreDirectoryList(saveAsFile);
      }
   }

   void restoreDirectoryList(File saveAsFile)
   {
      this.myStatusBar.setText(" ");
      if (saveAsFile == null ||
         !saveAsFile.exists() || !saveAsFile.isFile())
      {
         return;
      }
      BufferedReader buffRead = null;
      try
      {
         buffRead =
            new BufferedReader(new FileReader(saveAsFile));
         String appendMe = buffRead.readLine();
         while (appendMe != null)
         {
            this.add(new File(appendMe));
            appendMe = buffRead.readLine();
         }
         rememberSave(saveAsFile);
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
         JOptionPane.showMessageDialog(
            this.myDirPanel,
            AppStarter.getString("DirListController.1") + "\n" //$NON-NLS-1$ //$NON-NLS-2$
            + ex.getMessage(),
            AppStarter.getString("DirListController.2"), //$NON-NLS-1$
            JOptionPane.ERROR_MESSAGE);
         rememberSave(saveAsFile);
      }
      finally
      {
      	if (buffRead!= null)
      	{
	      	try{buffRead.close();}
	      	catch(Exception ex){ex.printStackTrace();}
      	}
      }
   }

   void rememberSave(File file)
   {
      this.myStatusBar.setText(" ");
      this.myLastSave = file.getAbsolutePath();
      this.myPrefs.put(this.myPreferencesLabel + LAST_SAVE, this.myLastSave);
      try
      {
         this.myPrefs.flush();
      }
      catch (Exception ex)
      {
      	ex.printStackTrace();
      }
   }
   
   private static final String SAVE_DIRECTORY = "saveDirectory"; //$NON-NLS-1$
   public String getSaveDirectory()
   {
      String rtnMe = FileUtil.checkDirectory(this.myPrefs.get(
      		SAVE_DIRECTORY,
         System.getProperty("user.home"))); 
      if (rtnMe == null)
      {
      	rtnMe = System.getProperty("user.home");
      }
      return rtnMe;
   }

   public void setSaveDirectory(String dir)

   {
      String validated = FileUtil.safeValidateDirectory(dir);
      if (validated != null)
      {
         this.myPrefs.put(SAVE_DIRECTORY, validated);
      }
      else
      {
         this.myPrefs.remove(SAVE_DIRECTORY);
      }
   }


   /*
    =============================================================================
                             Private Functions
    =============================================================================
    */

   private void layout()
   {
      //addFilePopup();

      this.myDirList.setMinimumSize(new java.awt.Dimension(200, 100));

      JButton btnAdd = new JButton(AppStarter.getString("DirListController.21")); //$NON-NLS-1$
      JButton btnSub = new JButton(AppStarter.getString("DirListController.22")); //$NON-NLS-1$
      JButton btnRemove = new JButton(AppStarter.getString("DirListController.23")); //$NON-NLS-1$
      JButton btnClear = new JButton(AppStarter.getString("DirListController.24")); //$NON-NLS-1$
      JButton btnSave = new JButton(AppStarter.getString("save")); //$NON-NLS-1$
      JButton btnRestore = new JButton(AppStarter.getString("DirListController.26")); //$NON-NLS-1$
      final JCheckBox boxAutoRestore = new JCheckBox(AppStarter.getString("DirListController.27")); //$NON-NLS-1$
      boxAutoRestore.setToolTipText(
         AppStarter.getString("DirListController.28")); //$NON-NLS-1$

      boolean autoRestore =
         this.myPrefs.getBoolean(this.myPreferencesLabel + AUTO_RESTORE, false);
      boxAutoRestore.setSelected(autoRestore);
      boxAutoRestore.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            DirListController.this.myPrefs.putBoolean(DirListController.this.myPreferencesLabel + AUTO_RESTORE,
               boxAutoRestore.isSelected());
            try
            {
               DirListController.this.myPrefs.flush();
            }
            catch (Exception ex)
            {
            	ex.printStackTrace();
            }
         }
      });
      if (autoRestore)
      {
         this.myLastSave = this.myPrefs.get(this.myPreferencesLabel + LAST_SAVE, null);
         if (this.myLastSave != null)
         {
            File restoreMe = new File(this.myLastSave);
            restoreDirectoryList(restoreMe);
         }
      }

      JPanelRows btnPanel = new JPanelRows(FlowLayout.CENTER);
      JPanel aRow = btnPanel.topRow();
      aRow.add(btnAdd);
      aRow.add(btnSub);
      aRow.add(btnRemove);
      aRow.add(btnClear);
      aRow = btnPanel.nextRow();
      aRow.add(btnSave);
      aRow.add(btnRestore);
      aRow.add(boxAutoRestore);
      
      aRow = btnPanel.nextRow(FlowLayout.LEFT);
      aRow.add(this.myStatusBar);

      btnAdd.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            addBtnActionPerformed();
         }
      });

      btnSub.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            addRecursiveActionPerformed();
         }
      });

      btnRemove.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            removeBtnActionPerformed();
         }
      });

      btnClear.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            clearBtnActionPerformed();
         }
      });

      btnSave.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            saveBtnActionPerformed();
         }
      });

      btnRestore.addActionListener(new java.awt.event.ActionListener()
      {
         public void actionPerformed(java.awt.event.ActionEvent evt)
         {
            restoreBtnActionPerformed();
         }
      });

      this.myDirPanel.setLayout(new BorderLayout());
      this.myDirPanel.add(this.myDirScroll, BorderLayout.CENTER);
      this.myDirPanel.add(btnPanel, BorderLayout.SOUTH);
   }

   void addBtnActionPerformed()
   {
   	clear();
      JFileChooser chooser =
         new JFileChooser(getAddDirectory());
      chooser.setDialogTitle(this.myPopupName + ": " + AppStarter.getString("DirListController.29")); //$NON-NLS-1$
      chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

      int returnVal = chooser.showOpenDialog(this.myDirPanel);
      if (returnVal == JFileChooser.APPROVE_OPTION)
      {
         File picked = chooser.getSelectedFile();
         setAddDirectoryParent(picked);
         this.myDirListModel.addDirectory(picked);
      }
   }

   void addRecursiveActionPerformed()
   {
   	clear();
      final JFileChooser chooser = new JFileChooser(getAddDirectory());
      chooser.setDialogTitle(this.myPopupName + ": " + AppStarter.getString("DirListController.30")); //$NON-NLS-1$
      chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);

      int returnVal = chooser.showOpenDialog(this.myDirPanel);
      if (returnVal == JFileChooser.APPROVE_OPTION)
      {
         this.myDirPanel.setCursor(Cursor.getPredefinedCursor(Cursor.
            WAIT_CURSOR));
         Runnable runMe = new Runnable()
         {
            public void run()
            {
               File picked = chooser.getSelectedFile();
               //System.out.println("You chose to recurse: " + picked);
               setAddDirectoryParent(picked);
               if (DirListController.this.iJustShowKids)
               {
               	DirListController.this.myDirListModel.justAddKids(picked); 
                  DirListController.this.myStatusBar.setText("Done");
               }
               else
               {
	               // Show kids before parent
	               DirListController.this.myDirListModel.recurseAddDirectory(picked);  
	               DirListController.this.myDirListModel.addDirectory(picked);
                  DirListController.this.myStatusBar.setText("Done");
               }
               DirListController.this.myDirPanel.setCursor(Cursor.getPredefinedCursor(Cursor.
                     DEFAULT_CURSOR));
             }
         };
         // Feb 2011 trying to solve sometimes doesn's show after adding...
         // Apparently running this in a separate thread causes the updates
         // not to show up occasionally. (About every 3rd time.)
         //SwingUtilities.invokeLater(runMe);
         //new Thread(runMe).run();
         //Thread t = new Thread(runMe);
         //t.start();
         
         runMe.run();
      }
   }

   void removeBtnActionPerformed()
   {
   	clear();
   	this.myDirList.removeSelected();
   }

   void clearBtnActionPerformed()
   {
   	clear();
      this.myDirListModel.removeAllElements();
      //myDirList.removeAll();
      //myDirList.invalidate();
      //myDirList.repaint();
   }

   void saveBtnActionPerformed()
   {
      this.saveDirectoryList();
   }

   void restoreBtnActionPerformed()
   {
      this.restoreDirectoryList();
   }

   /*
    =============================================================================
                             Popup Functions
    =============================================================================
    */
   /*
   private void addFilePopup()
   {
      myDirList.addMouseListener(new MouseAdapter()
      {
         public void mouseClicked(MouseEvent e)
         {
            if (e.getClickCount() == 2)
            {
               popUp(e);
               return;
            }
            else if ((e.getModifiers() & MouseEvent.BUTTON2_MASK) != 0)
            {
               popUp(e);
               return;
            }
            else if ((e.getModifiers() & MouseEvent.BUTTON3_MASK) != 0)
            {
               popUp(e);
               return;
            }
         }
      });
   }

   private void popUp(MouseEvent e)
   {

             // SAMPLE CODE for picking individual items.
             //final int index = myDirList.locationToIndex(e.getPoint());
             //if (index < 0){return;}
             //File breakFile = (File)myDirList.getModel().getElementAt(index);

      JPopupMenu menu = new JPopupMenu();
      JPanel center = new JPanel(new FlowLayout(FlowLayout.CENTER));
      center.add(new JLabel(myPopupName), 0);
      menu.insert(center, 0);
      menu.addSeparator();
      JMenuItem breakItem = menu.add("Save List");
      JMenuItem clearItem = menu.add("Restore List");

      breakItem.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent event)
         {
            // Make the cells actually update.
            //myDirList.repaint();
         }
      });

      clearItem.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent event)
         {
            myDirList.invalidate();
         }
      });

      menu.show((Component)e.getSource(), e.getX(), e.getY());
   }
*/

   //===========================================================================
   // The directory selected when the user hits the Add or Add Subdirectory
   // buttons.
   //===========================================================================
   String getAddDirectory()
   {
      String addDir = this.myPrefs.get(this.myPreferencesLabel, System.getProperty("user.home")); //$NON-NLS-1$
      return IOUtils.validDirectory(addDir);
   }

   /**
    * Set the Add Directory to the parent of this directory.
    * Use getAddDirectory() to get the value set.
    * @param dir
    */
   void setAddDirectoryParent(File dir)
   {
   	if (dir.getParentFile() != null)
   	{
   		dir = dir.getParentFile(); 		
   	}
   	if (!dir.isFile())
      { 
      	this.myPrefs.put(this.myPreferencesLabel,
      			dir.toString());
      }
   }

   private void clear()
   {
   	if (this.myClearable != null)
   	{
   		this.myClearable.clear();
   	}
   }
}
