/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.io;

import java.awt.Component;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.StringTokenizer;

import javax.swing.BoundedRangeModel;
import javax.swing.DefaultBoundedRangeModel;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.util.Conversions;
import org.gerhardb.lib.util.StopCheck;

/**
 */
public class FileUtil
{
	public static final String SPACE = " "; //$NON-NLS-1$
	public static final String NEWLINE = "\n"; //$NON-NLS-1$
	public static final String SYS_NEWLINE = System.getProperty("line.separator"); //$NON-NLS-1$

	private FileUtil()
	{
		// Prevent public creation.
	}

	/**
	 * Size ready to display to user in bytes, k, mb, gb, etc.
	 */
	public static String getSize(File file)
	{
		return Conversions.formattedBinaryBytes(file.length()) + " ("
				+ Conversions.formattedMetricBytes(file.length()) + ")";
	}

	/**
	 * Removes the file name extension.
	 * <p>
	 * The file name extension is defined as the last dot something.
	 * Examples:<br>
	 * frodo.jpg ==> frodo
	 * frodo.bar.jpg ==> frodo.bar
	 * frodo.bar.not.jpg ==> frodo.bar.not
	 * sometextfile.txt ==> sometextfile
	 * @return file name without extension
	 */
	public static String fileNameNoExtension(String oldName)
	{
		final StringTokenizer tokens = new StringTokenizer(oldName, "."); //$NON-NLS-1$
		final StringBuffer baseName = new StringBuffer(50);
		String ending = null;
		// Because file can't be null, there must be at least one token.
		// We will ignore the possible problem of a file named "."
		baseName.append(tokens.nextToken());
		while (tokens.hasMoreTokens())
		{
			// Put back any intermediate periods in the name.
			if (ending != null)
			{
				baseName.append(ending + "."); //$NON-NLS-1$
			}
			// Last ending will be ignored.
			ending = tokens.nextToken();
		}
		return baseName.toString();
	}

	/**
	 * Returns the file name extension.
	 * <p>
	 * The file name extension is defined as the last dot something.
	 * Returns null if there is no dot.
	 * Examples:<br>
	 * frodo.jpg ==> jpg
	 * @return fextension
	 */
	public static String getExtension(String name)
	{
		if (name == null) { return null; }
		final StringTokenizer tokens = new StringTokenizer(name, "."); //$NON-NLS-1$
		String ending = null;
		if (tokens.countTokens() == 1) { return null; }
		while (tokens.hasMoreTokens())
		{
			ending = tokens.nextToken();
		}
		return ending;
	}

	/**
	 * Physically copy a file to a different name.
	 */
	public static void copyFile(File toFile, File copyThis)
			throws TargetFileExistsException, FileNotFoundException, IOException
	{
		if (copyThis == null || toFile == null) { throw new IllegalArgumentException(
				"null arguements are not allowed"); } //$NON-NLS-1$

		if (!copyThis.isFile()) { throw new IllegalArgumentException(copyThis
				+ " is NOT a File"); } //$NON-NLS-1$

		// See if the file already exits, and if so take appropriate action.
		if (toFile.exists()) { throw new TargetFileExistsException(toFile
				.getName(), toFile.getAbsolutePath()); }

		// A little debug...
		/*
		 System.out.println( "FileUtil - Copying File--------------------------- " );
		 System.out.println( "From: " + copyThis.getAbsolutePath() );
		 System.out.println( "To: " + newName.getAbsolutePath() );
		 */

		final BufferedInputStream in = new BufferedInputStream(
				new FileInputStream(copyThis));
		final BufferedOutputStream out = new BufferedOutputStream(
				new FileOutputStream(toFile));

		try
		{
			/*
			int aByte = in.read();
			while (aByte != -1)
			{
				out.write(aByte);
				aByte = in.read();
			}
			 */
			byte[] buf = new byte[1024];
			int len = 0;
			while ((len = in.read(buf)) > 0)
			{
				out.write(buf, 0, len);
			}
		}
		catch (IOException ex)
		{
			try
			{
				in.close();
				out.close();
				toFile.delete();
			}
			catch (Exception doubleEx)
			{
				doubleEx.printStackTrace();
			}
			throw ex;
		}
		out.close();
		in.close();
	}

	/**
	 * Physically copy a file to a different directory.
	 */
	public static void copyFileToDir(File toDir, File copyThis)
			throws TargetFileExistsException, FileNotFoundException, IOException
	{
		// First, let's make sure we're talking about moving a file to a directory.
		if (copyThis == null || toDir == null) { throw new IllegalArgumentException(
				"null arguements are not allowed"); } //$NON-NLS-1$

		if (!copyThis.isFile()) { throw new IllegalArgumentException(copyThis
				+ " is NOT a File"); } //$NON-NLS-1$

		if (!toDir.isDirectory()) { throw new IllegalArgumentException(toDir
				+ " is NOT a Directory"); } //$NON-NLS-1$

		// Now that we know what we're talking about, let's get started!

		// First, figure out where the move will put the file.
		final File newName = new File(toDir.getAbsolutePath() + File.separator
				+ copyThis.getName());

		copyFile(newName, copyThis);
	}

	/**
	 * Optimized for where called from...
	 * @param toDir
	 * @param moveThis
	 * @throws Exception
	 */
	public static File moveFile(File toDir, File moveThis) throws Exception
	{
		// First, figure out where the move will put the file.
		File newName = new File(toDir.getAbsolutePath() + System.getProperty("file.separator") + moveThis.getName());
		if (newName.exists()) 
		{ 
			throw new TargetFileExistsException(newName.getName(), toDir.getAbsolutePath()); 
		}

		// Try the easy way first.
		// This won't work across drives.
		if (moveThis.renameTo(newName)) 
		{ 
			//System.out.println("FileUtil renamed to: " + newName);
			return newName; 
		}

		// If the easy way didn't work,
		// do the old-fashioned copy then delete.
		//System.out.println("FileUtil copy " + moveThis + " to: " + toDir);
		copyFileToDir(toDir, moveThis);
		if (moveThis.delete()) { return newName; }
		throw new FileUtilDeleteException(AppStarter.getString("FileUtil.23") //$NON-NLS-1$
				+ moveThis.getAbsolutePath());
	}

	/**
	 * Physically move a file to a different directory.
	 * @throws TargetFileExistsException
	 */
	public static void moveFile(File toDir, File moveThis, StringBuffer report,
			EzLogger logger, boolean makeNewName) throws Exception
	{
		// First, let's make sure we're talking about moving a file to a directory.
		if (moveThis == null || toDir == null) { throw new IllegalArgumentException(
				"null arguements are not allowed"); } //$NON-NLS-1$

		if (!moveThis.isFile()) { throw new IllegalArgumentException(moveThis
				+ " is NOT a File"); } //$NON-NLS-1$

		if (!toDir.isDirectory()) { throw new IllegalArgumentException(toDir
				+ " is NOT a Directory"); } //$NON-NLS-1$

		// Now that we know what we're talking about, let's get started!

		// First, figure out where the move will put the file.
		File newName = new File(toDir.getAbsolutePath() + File.separator
				+ moveThis.getName());

		// See if the file already exits, and if so take appropriate action.
		String nameNoPath = newName.getName();
		if (!makeNewName && newName.exists()) { throw new TargetFileExistsException(
				nameNoPath, toDir.getAbsolutePath()); }

		while (newName.exists())
		{
			String rootNameNoExtension = fileNameNoExtension(nameNoPath);
			String ending = ""; //$NON-NLS-1$
			int period = nameNoPath.lastIndexOf("."); //$NON-NLS-1$
			if (period > -1)
			{
				ending = nameNoPath.substring(period);
			}
			boolean makeItUp = true;
			int underscore = rootNameNoExtension.lastIndexOf("_"); //$NON-NLS-1$
			if (underscore > -1)
			{
				String possibleNumber = rootNameNoExtension
						.substring(underscore + 1);
				try
				{
					int number = Integer.parseInt(possibleNumber) + 1;
					makeItUp = false;
					String rootBase = rootNameNoExtension.substring(0, underscore);
					String newRoot = rootBase + "_" + number + ending; //$NON-NLS-1$
					newName = new File(toDir.getAbsolutePath() + File.separator
							+ newRoot);
					nameNoPath = newName.getName();
				}
				catch (Exception ex)
				{
					ex.printStackTrace();
				}
			}
			if (makeItUp)
			{
				newName = new File(toDir.getAbsolutePath() + File.separator
						+ rootNameNoExtension + "_1" + ending); //$NON-NLS-1$
				nameNoPath = newName.getName();
			}
		}

		// A little debug...
		/*
		 System.out.println("FileUtil - Moving File--------------------------- ");
		 System.out.println("Old Name: " + moveThis.getAbsolutePath());
		 System.out.println("New name: " + newName.getAbsolutePath());
		 */

		// Try the easy way first.
		// This won't work across drives.
		if (moveThis.renameTo(newName))
		{
			// Apparently report is only valided for park operations.
			if (report != null)
			{
				report
						.append("   " + AppStarter.getString("FileUtil.14") + AppStarter.getString("colon") + " " + newName + "\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
			}
			if (logger != null)
			{
				logger
						.logLine("   " + AppStarter.getString("FileUtil.17") + AppStarter.getString("colon") + " " + newName); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			}
			return;
		}

		// If the easy way didn't work,
		// do the old-fashioned copy then delete.
		copyFileToDir(toDir, moveThis);
		// Apparently report is only valided for park operations.
		if (report != null)
		{
			report
					.append("   " + AppStarter.getString("FileUtil.19") + AppStarter.getString("colon") + " " + newName + "\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$ //$NON-NLS-5$
		}
		if (logger != null)
		{
			logger
					.logLine("   " + AppStarter.getString("FileUtil.22") + AppStarter.getString("colon") + " " + newName); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		}
		if (moveThis.delete()) { return; }
		throw new FileUtilDeleteException(
				AppStarter.getString("FileUtil.23") + AppStarter.getString("colon") + " " //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						+ moveThis.getAbsolutePath());
	}

	/**
	 * Concatenates a list of files to a new file.
	 * Will destroy the new file so be sure the new file is NOT in the list
	 * of files to be concatentated.
	 */
	public static void concatFiles(File newFile, final File[] concat,
			BoundedRangeModel range, StopCheck check)
			throws TargetFileExistsException, FileNotFoundException, IOException
	{
		// First, let's make sure we're talking about moving a file to a directory.
		if (newFile == null || concat == null) { throw new IllegalArgumentException(
				"concatFiles does not take non-null arguments"); } //$NON-NLS-1$

		if (concat.length == 0) { throw new IllegalArgumentException(
				"no files passed in to concatinate"); } //$NON-NLS-1$

		if (newFile.isFile()) { throw new TargetFileExistsException(newFile
				.getName(), newFile.getAbsolutePath()); }

		if (range == null)
		{
			range = new DefaultBoundedRangeModel();
		}
		range.setRangeProperties(0, 0, 0, concat.length, false);

		// Now that we know what we're talking about, let's get started!
		final BufferedOutputStream out = new BufferedOutputStream(
				new FileOutputStream(newFile));

		for (int i = 0; i < concat.length; i++)
		{
			if (check != null && check.isStopped())
			{
				break;
			}
			final BufferedInputStream in = new BufferedInputStream(
					new FileInputStream(concat[i]));

			int aByte = in.read();
			while (aByte != -1)
			{
				out.write(aByte);
				aByte = in.read();
			}
			in.close();
			range.setValue(i + 1);
		}
		out.close();
	} // concatFiles(  )

	//============================================================================
	// Tell User Functions
	// No logging, User is told of problem immediately
	//============================================================================

	/**
	 * Physically copy a file to a different directory.
	 * All problems are handled here.
	 */
	public static boolean copyFileToDir(File toDir, File copyThis,
			java.awt.Component onTopOf) // NO_UCD
	{
		try
		{
			copyFileToDir(toDir, copyThis);
			return true;
		}
		catch (TargetFileExistsException ex)
		{
			// Not very appropriate, but better than doing nothing.
			// IMPROVEMENT - should ask user whether to overwrite, cancel,
			// auto-rename, or rename.
			// which should default to whatever was done last time.
			JOptionPane
					.showMessageDialog(
							onTopOf,
							ex.getMessage()
									+ " - " + AppStarter.getString("FileUtil.27"), //$NON-NLS-1$ //$NON-NLS-2$
							AppStarter.getString("FileUtil.28"), JOptionPane.WARNING_MESSAGE); //$NON-NLS-1$
			return false;
		}
		catch (Exception ex)
		{
			JOptionPane.showMessageDialog(onTopOf, ex.getMessage(), AppStarter
					.getString("FileUtil.29"), //$NON-NLS-1$
					JOptionPane.ERROR_MESSAGE);
			return false;
		}
	} // copyFile(  )

	/**
	 * Physically move a file to a different directory,
	 * WITH on screen error reporting.
	 */
	public static boolean moveFile(File toDir, File moveThis,
			java.awt.Component onTopOf)
	{
		try
		{
			FileUtil.moveFile(toDir, moveThis);
			return true;
		}
		catch (TargetFileExistsException ex)
		{
			// Not very appropriate, but better than doing nothing.
			// IMPROVEMENT - should ask user whether to overwrite, cancel,
			// auto-rename, or rename.
			// which should default to whatever was done last time.
			JOptionPane
					.showMessageDialog(
							onTopOf,
							ex.getMessage()
									+ " - " + AppStarter.getString("FileUtil.31"), //$NON-NLS-1$ //$NON-NLS-2$
							AppStarter.getString("FileUtil.32"), JOptionPane.WARNING_MESSAGE); //$NON-NLS-1$
			return false;
		}
		catch (Exception ex)
		{
			JOptionPane.showMessageDialog(onTopOf, ex.getMessage(), AppStarter
					.getString("FileUtil.33"), //$NON-NLS-1$
					JOptionPane.ERROR_MESSAGE);
			return false;
		}
	}

	/**
	 * Check for a direcotry with no errors reported.
	 * @param dirName
	 * @return null for any problems
	 */
	public static String safeValidateDirectory(String dirName)
	{
		try
		{
			return validateDirectory(dirName);
		}
		catch (Exception ex)
		{
			//ex.printStackTrace();
		}
		return null;
	}

	public static String validateDirectory(String dirName)
			throws FileNotFoundException
	{
		if (dirName == null || dirName.length() == 0) { throw new IllegalArgumentException(
				"directory name required"); } //$NON-NLS-1$

		File dir = new File(dirName);
		if (!dir.exists()) { throw new FileNotFoundException(dirName
				+ " does not exist. "); } //$NON-NLS-1$

		// If this is a file instead of a directory,
		// just get the directory part of it.
		if (!dir.isDirectory())
		{
			dir = dir.getParentFile();
			if (!dir.isDirectory()) { throw new FileNotFoundException(dir
					+ " " + AppStarter.getString("FileUtil.36")); } //$NON-NLS-1$ //$NON-NLS-2$
			return dir.getAbsolutePath();
		}
		return dirName;
	}

	/**
	 * Returns null if directory not found.
	 * @param dirName
	 * @return name of directory if it exists.
	 */
	public static String checkDirectory(String dirName)
	{
		if (dirName == null) { return null; }

		if (dirName.length() == 0) { return null; }

		File dir = new File(dirName);
		if (!dir.exists()) { return null; }

		// If this is a file instead of a directory,
		// just get the directory part of it.
		if (!dir.isDirectory())
		{
			dir = dir.getParentFile();
			if (!dir.isDirectory()) { return null; }
			return dir.getAbsolutePath();
		}
		return dirName;
	}

	/**
	 * Gets a string (for example an XML string) from a file.
	 * @param file The file to read.
	 * @return contents The contents of the file as a String.
	 * @throws java.io.FileNotFoundException
	 * @throws java.io.IOException
	 */
	public static String getStringFromFile(java.io.File file)
			throws java.io.FileNotFoundException, java.io.IOException
	{
		BufferedReader buffRead = new BufferedReader(new FileReader(file));
		StringBuffer buff = new StringBuffer(5000);
		String appendMe = buffRead.readLine();
		while (appendMe != null)
		{
			buff.append(appendMe + "\n"); //$NON-NLS-1$
			appendMe = buffRead.readLine();
		}
		buffRead.close();
		String xml = buff.toString();
		return xml;
	}

	public static String lookupDir(String msg, String priorDir, Component comp)
	{
		if (priorDir == null)
		{
			priorDir = System.getProperty("user.home"); //$NON-NLS-1$
		}

		JFileChooser chooser = new JFileChooser();
		chooser.setDialogTitle(msg);
		chooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		chooser.setCurrentDirectory(new File(priorDir));

		int returnVal = chooser.showOpenDialog(comp);
		if (returnVal == JFileChooser.APPROVE_OPTION)
		{
			try
			{
				return chooser.getSelectedFile().getCanonicalPath();
			}
			catch (IOException ex)
			{
				ex.printStackTrace();
			}
		}
		return null;
	}

	public static File[] listFilesIncludingSubDirectories(File rootDir) // NO_UCD
	{
		ArrayList<File> fileList = new ArrayList<File>(1000);
		addSubdirectoriesRecursive(rootDir, fileList);
		File[] rtnMe = new File[fileList.size()];
		rtnMe = fileList.toArray(rtnMe);
		return rtnMe;
	}

	private static void addSubdirectoriesRecursive(File root, ArrayList<File> list)
	{
		File[] files = root.listFiles();
		Arrays.sort(files);
		for (int i = 0; i < files.length; i++)
		{
			if (files[i].isDirectory())
			{
				addSubdirectoriesRecursive(files[i], list);
			}
			else
			{
				list.add(files[i]);
			}
		}
	}

	// ==========================================================================
	//                       Test Harness
	// ==========================================================================
	public static void main(String[] args)
	{
		/*
		final String fromDir = "e:/testpics/dedupBig/"; //$NON-NLS-1$
		final File toDir = new File("e:/testpics/park/"); //$NON-NLS-1$
		System.out.println(toDir.getAbsolutePath());
		
		StringBuffer buff = new StringBuffer();

		try
		{
			org.gerhardb.lib.io.FileUtil.moveFile(toDir, new File(fromDir + "aaamoveMe.jpg"), buff, null, true); //$NON-NLS-1$

		}
		catch (Exception ex)
		{
			System.out.println("Move ERROR: " + ex.getMessage()); //$NON-NLS-1$
		}
		*/
		
		File[] files = listFilesIncludingSubDirectories(new File("/media/rename"));
		for(int i=0; i<files.length; i++)
		{
			System.out.println(files[i].getAbsolutePath());
			if (files[i].getName().endsWith("wmv"))
			{
				String newName = files[i].getParentFile().getAbsolutePath() + "/" + "goof" + i + ".wmv" ;
				System.out.println(newName);
				files[i].renameTo(new File(newName));
			}
		}
	}
}
