/*
 * Copyright (c) 2011 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print;

import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;


/**
 * This class will print header and footer.
 * It will also give updated PageLayout for stuff that will be printing
 * inside the header and footer.  That is, it update the usable area for the 
 * amount the header and footer uses up.
 */
public class HeaderFooter
{
	public static final int DEFAULT_MARGIN_PADDING_DPI = 6; 
	
	InteriorLayout interior;
	int pageCount;

	// This will come from BufferedPrinting mostly...
	public HeaderFooter(InteriorLayout ilIn, int pageCountIn)
	{
		this.interior = ilIn;
		this.pageCount = pageCountIn;
	}
	
	public InteriorLayout draw(Graphics gIn, int pageIndex, HeaderFooterInfo infoIn, Pageable p)
	{
		//System.out.println("_________________________________________________");
		//System.out.println("Header Footer Drawing page: " + pageIndex);
		
		HeaderFooterInfo info = infoIn.makePageCountAdjusted(pageIndex, p);	
		
		// Make sure we don't pollute the main graphics.
		Graphics g = gIn.create();
		
		// May want to assume this is done before we get to here.
		// Header and Footer is assumed to have same background as margin.
		/*
		g.setColor(info.backgroundColor);
		g.fillRect(
				this.interior.getLeftMargin(), 
				this.interior.getTopMargin(), 
				this.interior.getWidth(),
				this.interior.getHeight());
		 */
		g.setColor(info.textColor);
		
		int topHeight = drawHeader(g, info);
		int bottomHeight = drawFooter(g, info);
		
		// Get rid of the graphics we created.
		g.dispose();		
		
		// Compute and return the interior space left after the margin.
		if (topHeight > 0)
		{
			topHeight = topHeight + DEFAULT_MARGIN_PADDING_DPI;
		}
		if (bottomHeight > 0)
		{
			bottomHeight = bottomHeight + DEFAULT_MARGIN_PADDING_DPI;
		}	
		
		int ajdustedTopMargin = this.interior.getTopMargin() + topHeight;		
		int ajdustedHeight = this.interior.getHeight() - topHeight - bottomHeight;
		return new InteriorLayout(
				this.interior.getLeftMargin(), 
				ajdustedTopMargin, 
				this.interior.getWidth(), 
				ajdustedHeight);
	}
	
	private int drawHeader(Graphics g, HeaderFooterInfo info)
	{
		Font headerFont = new Font(
				info.headerFontFamily, Font.PLAIN,
				info.headerFontSize);
		//System.out.println("headerFont: " + headerFont);
		
		// DO SOME CALCULATIONS TO DETERMINE THE SIZE OF THE HEADER.
		g.setFont(headerFont);
		FontMetrics fm = g.getFontMetrics();
		int textHeight = fm.getAscent() + fm.getDescent();
		int bottomOfText = this.interior.getTopMargin() + textHeight;
		
		// Text is draw up from the Ascent and down from the Descent.
		// This puts text at the top of the printable margin.
		int textX = (this.interior.getTopMargin() + fm.getAscent());
		
		// Now we are ready to do some work!
		if (info.showLines)
		{
			// Draw a line just underneath the text.		
			g.drawLine(
					this.interior.getLeftMargin(), bottomOfText, 
					this.interior.getRightMargin(), bottomOfText);

		}
	
		// Print the LEFT Header
		String printString = info.headerLeft;
		if (printString != null && printString.trim().length() > 0)
		{
			// Left Justified
			g.drawString(printString, this.interior.getLeftMargin(), textX);
		}

		// Print the CENTER Header
		printString = info.headerCenter;
		if (printString != null && printString.trim().length() > 0)
		{
			// Centered
			int headerWidth = fm.stringWidth(printString);
			int textY = this.interior.getLeftMargin() + 
					((this.interior.getWidth() - headerWidth) / 2);

			g.drawString(printString, textY, textX);
		}
	
		// Print the RIGHT Header
		printString = info.headerRight;
		if (printString != null && printString.trim().length() > 0)
		{
			// Right Justified
			int headerWidth = fm.stringWidth(printString);
			int y = this.interior.getRightMargin() - headerWidth;
			g.drawString(printString, y, textX);
		}
		
		return textHeight;
	}

	private int drawFooter(Graphics g, HeaderFooterInfo info)
	{
		Font footerFont = new Font(
				info.footerFontFamily, Font.PLAIN,
				info.footerFontSize);
		//System.out.println("footerFont: " + footerFont);
		
		// DO SOME CALCULATIONS TO DETERMINE THE SIZE OF THE HEADER.
		g.setFont(footerFont);
		FontMetrics fm = g.getFontMetrics();
		int textHeight = fm.getAscent() + fm.getDescent();
		int footerHeight = textHeight;
		int topOfText = this.interior.getBottomMargin() - footerHeight;
		
		if (info.showLines)
		{
			// Draw a line just on top of the text.		
			g.drawLine(
					this.interior.getLeftMargin(), topOfText, 
					this.interior.getRightMargin(), topOfText);
		}

		// Text is draw up from the Ascent and down from the Descent.
		int textX = (this.interior.getBottomMargin() - fm.getDescent());

		// ------------- Actual Printing Follows -----------------------

		// Print the LEFT Footer
		String printString = info.footerLeft;
		if (printString != null && printString.trim().length() > 0)
		{
			// LEFT Justified
			g.drawString(printString, this.interior.getLeftMargin(), textX);
		}

		// Print the CENTER Footer
		printString = info.footerCenter;
		if (printString != null && printString.trim().length() > 0)
		{
			// Centered
			int headerWidth = fm.stringWidth(printString);
			int textY = this.interior.getLeftMargin() + 
			((this.interior.getWidth() - headerWidth) / 2);
			g.drawString(printString, textY, textX);
		}
		
		// Print the RIGHT Footer
		printString = info.footerRight;
		if (printString != null && printString.trim().length() > 0)
		{
			// Right Justified
			int headerWidth = fm.stringWidth(printString);
			int y = this.interior.getRightMargin() - headerWidth;
			g.drawString(printString, y, textX);
		}
		return footerHeight;
	}
	
	// ==========================================================================
	//                       Static Methods
	// ==========================================================================
	
	public static InteriorLayout calcInterior(HeaderFooterInfo info, InteriorLayout layout)
	{
		/*
		Font headerFont = new Font(
				info.headerFontFamily, Font.PLAIN,
				info.headerFontSize);
		Font footerFont = new Font(
				info.footerFontFamily, Font.PLAIN,
				info.footerFontSize);
		*/
		int topHeight = 0;
		if (drawHeaderText(info))
		{
			topHeight = info.headerFontSize;
		}
		
		int bottomHeight = 0;
		if (drawFooterText(info))
		{
			bottomHeight = info.footerFontSize;
		}
		
		// Compute and return the interior space left after the margin.
		if (topHeight > 0 || info.showLines)
		{
			topHeight = topHeight + DEFAULT_MARGIN_PADDING_DPI;
		}
			
		if (bottomHeight > 0 || info.showLines)
		{
			bottomHeight = bottomHeight + DEFAULT_MARGIN_PADDING_DPI;
		}	
		
		int ajdustedTopMargin = layout.getTopMargin() + topHeight;		
		int ajdustedHeight = layout.getHeight() - topHeight - bottomHeight;
		return new InteriorLayout(
				layout.getLeftMargin(), 
				ajdustedTopMargin, 
				layout.getWidth(), 
				ajdustedHeight);
	}

	private static boolean drawHeaderText(HeaderFooterInfo info)
	{
		if (info.headerLeft != null && info.headerLeft.trim().length() > 0)
		{
			return true;
		}

		if (info.headerCenter != null && info.headerCenter.trim().length() > 0)
		{
			return true;
		}

		if (info.headerRight != null && info.headerRight.trim().length() > 0)
		{
			return true;
		}
		return false;
	}
	
	private static boolean drawFooterText(HeaderFooterInfo info)
	{
		if (info.footerLeft != null && info.footerLeft.trim().length() > 0)
		{
			return true;
		}

		if (info.footerCenter != null && info.footerCenter.trim().length() > 0)
		{
			return true;
		}

		if (info.footerRight != null && info.footerRight.trim().length() > 0)
		{
			return true;
		}
		return false;
	}

	// ==========================================================================
	//                       Main
	// ==========================================================================
	public static void main(String[] args)
	{
		/*
		String[] families = java.awt.GraphicsEnvironment.getLocalGraphicsEnvironment().getAvailableFontFamilyNames();
		for(int i=0; i<families.length; i++)
		{
			System.out.println(families[i]);
		}
		*/
		org.gerhardb.lib.util.startup.AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		HeaderFooterTest hfTest = new HeaderFooterTest();
		new PageablePreviewer(
				hfTest, 
	   		true, null);
	} 	
}
