/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print;

/**
 * Simple PageLayout.
 * Assumes 72 DPI like java.
 * Everything is stored in DPI.
 * No notion of Landscape or Portrait.
 */
public class PageLayout
{
	private static final int DPI = 72;

	private InteriorLayout interior;
	
	// Passed In - STORED IN DPI
	private int pageWidth;
	private int pageHeight;

	// Calculated - STORED IN DPI
	private int rightMarginSize;
	private int bottomMarginSize;
	private int centerX;
	private int centerY;
	
	//private boolean portrait;

	private PageLayout(InteriorLayout il)
	{
		this.interior = il;
	}

	/**
	 * Default PageLayout.
	 * 8.5 x 11
	 */
	public static PageLayout makeDefault()
	{
		return makeInches(8.5f, 11f, 1f, 1f, 6.5f, 9f);
	}

	public static PageLayout makeInches(float pageWidthIn, float pageHeightIn, 
			float leftMarginIn, float topMarginIn,  
			float usableWidthIn, float usableHeightIn)
	{
		return makeDPI((int) (pageWidthIn * DPI), (int) (pageHeightIn * DPI),
				(int) (leftMarginIn * DPI), (int) (topMarginIn * DPI),
				(int) (usableWidthIn * DPI), (int) (usableHeightIn * DPI));
	}
	
	public static PageLayout makeDPI(int pageWidthIn, int pageHeightIn,
			int leftMarginIn, int topMarginIn, 
			int usableWidthIn, int usableHeightIn)
	{
		InteriorLayout il = new InteriorLayout(leftMarginIn, topMarginIn, 
				usableWidthIn, usableHeightIn);
		
		PageLayout pl = new PageLayout(il);

		pl.pageWidth = pageWidthIn;
		pl.pageHeight = pageHeightIn;

		pl.rightMarginSize = pl.pageWidth - il.getRightMargin();
		pl.bottomMarginSize = pl.pageHeight - il.getBottomMargin();
		
		pl.centerX = pl.pageWidth / 2;
		pl.centerY = pl.pageHeight / 2;
		
		return pl;
	}

	/*
	public void setPortrait(boolean p)
	{
		this.portrait = p;
	}

	public boolean getPortrait()
	{
		return this.portrait;
	}
   */
	
	public int getPageHeight()
	{
		return this.pageHeight;
	}

	public int getPageWidth()
	{
		return this.pageWidth;
	}

	public InteriorLayout getInteriorLayout()
	{
		return this.interior;
	}
	
	public int getRightMarginSize()
	{
		return this.rightMarginSize;
	}

	public int getBottomMarginSize()
	{
		return this.bottomMarginSize;
	}

	public int getCenterX()
	{
		return this.centerX;
	}

	public int getCenterY()
	{
		return this.centerY;
	}

	public String toStringInches()
	{
		StringBuffer buff = new StringBuffer();

		buff.append("Page W x H: " + (float) this.pageWidth / DPI + " x "
				+ (float) this.pageHeight / DPI + "\n");
		buff.append("Page Center X x Y: " + (float) this.centerX / DPI + " x "
				+ (float) this.centerY / DPI + "\n");

		buff.append("Top x Bottom Margin Size: " + (float) this.interior.getTopMargin() / DPI + " x "
				+ (float) this.bottomMarginSize / DPI + "\n");
		buff.append("Left x Right Margin Size: " + (float) this.interior.getLeftMargin() / DPI + " x "
				+ (float) this.rightMarginSize / DPI + "\n");

		buff.append(this.interior.toStringInches());
		
		return buff.toString();

	}

	public String toStringDPI()
	{
		StringBuffer buff = new StringBuffer();

		buff.append("Page W x H: " + this.pageWidth + " x " + this.pageHeight + "\n");
		buff.append("Page Center X x Y: " + this.centerX + " x " + this.centerY
				+ "\n");

		buff.append("Top x Bottom Margin Size: " + this.interior.getTopMargin() + " x " + this.bottomMarginSize + "\n");
		buff.append("Left x Right Margin Size: " + this.interior.getLeftMargin() + " x " + this.rightMarginSize + "\n");
		
		buff.append(this.interior.toStringDPI());

		return buff.toString();
	}

	// ==========================================================================
	//                       Main
	// ==========================================================================
	public static void main(String[] args)
	{
		PageLayout pl = PageLayout.makeDefault();
		System.out.println(pl.toStringDPI());
		System.out.println(pl.toStringInches());
	}
}
