/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print;

import java.awt.Dialog;
import java.awt.GridLayout;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import org.gerhardb.jibs.Jibs;
import org.gerhardb.lib.swing.JPanelRows;
import org.gerhardb.lib.swing.OnlyFloat;
import org.gerhardb.lib.swing.SwingUtils;

/**
 * ContactSheetDisplay
 */
public class PageLayoutDialog
{
	int FIELD_WIDTH = 10;
	JDialog dialog;

	JComboBox size = new JComboBox(PageSize.values());

	AdjustingTextField pageWidth = new AdjustingTextField();
	AdjustingTextField pageHeight = new AdjustingTextField();

	AdjustingTextField topMargin = new AdjustingTextField();
	AdjustingTextField bottomMargin = new AdjustingTextField();
	AdjustingTextField leftMargin = new AdjustingTextField();
	AdjustingTextField rightMargin = new AdjustingTextField();

	JRadioButton inches = new JRadioButton("inches");
	JRadioButton millimeters = new JRadioButton("millimeters");
	JRadioButton dpi = new JRadioButton("dots per inch (DPI)");

	public PageLayoutDialog(JFrame parent, PageLayout layout)
	{
		this.dialog = new JDialog(parent, "Page Setup",
				Dialog.ModalityType.APPLICATION_MODAL);
		
		if (layout == null)
		{
			layout = PageLayout.makeDefault();
		}
		this.pageWidth.setDPI(layout.getPageWidth());
		this.pageHeight.setDPI(layout.getPageHeight());
		this.topMargin.setDPI(layout.getInteriorLayout().getTopMargin());
		this.leftMargin.setDPI(layout.getInteriorLayout().getLeftMargin());
		this.bottomMargin.setDPI(layout.getPageHeight() - layout.getInteriorLayout().getBottomMargin());
		this.rightMargin.setDPI(layout.getPageWidth() - layout.getInteriorLayout().getRightMargin());		
		
		for(PageSize ps: PageSize.values())
		{
			if (ps.width == layout.getPageWidth() && ps.height == layout.getPageHeight())
			{
				this.size.setSelectedItem(ps);
				break;
			}
		}

		JButton okBtn = new JButton("OK"); //$NON-NLS-1$
		okBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent ae)
			{
				PageLayoutDialog.this.dialog.dispose();
			}
		});

		JButton cancelBtn = new JButton(Jibs.getString("cancel")); //$NON-NLS-1$ //new JButton(Jibs.getString("ContactSheetPrintOptions.4")); //$NON-NLS-1$
		cancelBtn.addActionListener(new java.awt.event.ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent ae)
			{
				PageLayoutDialog.this.dialog.dispose();
			}
		});

		this.dialog.addWindowListener(new WindowAdapter()
		{
			@Override
			public void windowClosing(WindowEvent e)
			{
				PageLayoutDialog.this.dialog.dispose();
			}
		});

		this.size.addActionListener(new ActionListener()
		{
			public void actionPerformed(ActionEvent ae)
			{
				adjustPageSize();
			}
		});

		ChangeListener updateMeasurements = new ChangeListener()
		{
			public void stateChanged(ChangeEvent e)
			{
				PageLayoutDialog.this.pageWidth.update();
				PageLayoutDialog.this.pageHeight.update();
				PageLayoutDialog.this.topMargin.update();
				PageLayoutDialog.this.bottomMargin.update();
				PageLayoutDialog.this.leftMargin.update();
				PageLayoutDialog.this.rightMargin.update();
			}
		};
		this.inches.addChangeListener(updateMeasurements);
		this.millimeters.addChangeListener(updateMeasurements);
		this.dpi.addChangeListener(updateMeasurements);

		this.size.setEditable(false);
		adjustPageSize();

		ButtonGroup type = new ButtonGroup();
		type.add(this.inches);
		type.add(this.millimeters);
		type.add(this.dpi);
		this.inches.setSelected(true);

		JPanel colA = new JPanel(new GridLayout(3, 1));
		colA.add(new JLabel("Width: "));
		colA.add(new JLabel("Top: "));
		colA.add(new JLabel("Left: "));

		JPanel colB = new JPanel(new GridLayout(3, 1));
		colB.add(this.pageWidth);
		colB.add(this.topMargin);
		colB.add(this.leftMargin);

		JPanel colC = new JPanel(new GridLayout(3, 1));
		colC.add(new JLabel("Height: "));
		colC.add(new JLabel("Bottom: "));
		colC.add(new JLabel("Right: "));

		JPanel colD = new JPanel(new GridLayout(3, 1));
		colD.add(this.pageHeight);
		colD.add(this.bottomMargin);
		colD.add(this.rightMargin);

		JPanelRows panel = new JPanelRows();
		JPanel row = panel.topRow();
		row.add(this.size);

		row = panel.nextRow();
		row.add(this.inches);
		row.add(this.millimeters);
		row.add(this.dpi);

		row = panel.nextRow();
		row.add(colA);
		row.add(colB);
		row.add(colC);
		row.add(colD);

		row = panel.nextRow();
		row.add(okBtn);
		row.add(cancelBtn);

		this.dialog.setContentPane(panel);
		this.dialog.pack();
		SwingUtils.centerOnScreen(this.dialog);
		this.dialog.setVisible(true);
	}

	public PageLayout getPageLayout()
	{
		return PageLayout.makeDPI(
				this.pageWidth.atf_DPI, 
				this.pageHeight.atf_DPI, 
				this.leftMargin.atf_DPI,
				this.topMargin.atf_DPI, 
				this.pageWidth.atf_DPI - this.leftMargin.atf_DPI - this.rightMargin.atf_DPI, 
				this.pageHeight.atf_DPI - this.topMargin.atf_DPI - this.bottomMargin.atf_DPI);
	}

	void adjustPageSize()
	{
		PageSize pageSize = (PageSize) this.size.getSelectedItem();
		this.pageWidth.setDPI(pageSize.width);
		this.pageHeight.setDPI(pageSize.height);
	}

	static int mmToDPI(float mm)
	{
		float rtnMe = mm * 0.03937007874f * 72;
		//System.out.println("mmToDPI(" + mm + "): " + rtnMe);
		return (int) rtnMe;
	}

	public enum PageSize
	{
		LETTER("Letter 8 1/2 x 11 inches", 612, 792), A2("A2 420 x 594 mm",
				mmToDPI(420), mmToDPI(594)), A3("A3 297 x 420 mm", mmToDPI(297),
				mmToDPI(420)), A4("A4 210 x 297 mm", mmToDPI(210), mmToDPI(297)), A6(
				"A6 105 x 148 mm", mmToDPI(105), mmToDPI(148)), LEGAL(
				"Legal 8 1/2 x 14 inches", 612, 1008), HALF_LETTER(
				"Half Letter 5 1/2 X 8 1/2 inches", 396, 612), SUPER_B(
				"Super B 13 x 19 inches", 936, 1368), US_B("US B 11 x 17 inches",
				792, 1224), US_C("US C 17 x 22 inches", 1224, 1584), IN_3_5x5(
				"3 1/2 x 5 inches", 252, 360), IN_4x6("4 x 6 inches", 288, 432), IN_5x7(
				"5 x 7 inches", 360, 504), IN_8x10("8 x 10 inches", 576, 720), IN_11x14(
				"11 x 14 inches", 792, 1008), OTHER("User Defined", 0, 0);

		private final String label;
		final int width; // in dpi
		final int height; // in dpi

		PageSize(String s, int w, int h)
		{
			this.label = s;
			this.width = w;
			this.height = h;
		}

		// public int widthDPI() { return this.width; }
		//public int heightDPI() { return this.height; }
		@Override
		public String toString()
		{
			return this.label;
		}
	}

	private class AdjustingTextField extends JTextField //JFormattedTextField
	{
		int atf_DPI;

		AdjustingTextField()
		{
			super(PageLayoutDialog.this.FIELD_WIDTH);
			
			super.setDocument(new OnlyFloat());
			//super(NumberFormat.getNumberInstance());
			//super.setSize(PageLayoutDialog.this.FIELD_WIDTH, super.getHeight());

			super.addFocusListener(new FocusListener()
			{
				public void focusGained(FocusEvent e)
				{
					// we don't care
				}

				public void focusLost(FocusEvent e)
				{
					try
					{
						if (PageLayoutDialog.this.inches.isSelected())
						{
							float parsedInches = Float.parseFloat(AdjustingTextField.this.getText());
							AdjustingTextField.this.setDPI((int)(parsedInches * 72));
						}
						else if (PageLayoutDialog.this.millimeters.isSelected())
						{
							//super.setText(Integer.toString(Math.round(this.atf_DPI / 72f * 25.4000f)));   	
							int mm = Integer.parseInt(AdjustingTextField.this.getText());	
							AdjustingTextField.this.setDPI((int)(mm * 0.0393f * 72f));
						}
						else
						{
							int parsedDPI = Integer.parseInt(AdjustingTextField.this.getText());
							AdjustingTextField.this.setDPI(parsedDPI);
						}
					}
					catch(Exception ex)
					{
						ex.printStackTrace();
					}
				}
			});
		}

		void setDPI(int dpiIn)
		{
			this.atf_DPI = dpiIn;
			this.update();
		}

		void update()
		{
			if (PageLayoutDialog.this.inches.isSelected())
			{
				super.setText(Float.toString(this.atf_DPI / 72f));
			}
			else if (PageLayoutDialog.this.millimeters.isSelected())
			{
				super.setText(Integer.toString(Math
						.round(this.atf_DPI / 72f * 25.4000f)));
			}
			else
			{
				super.setText(Integer.toString(this.atf_DPI));
			}
		}
	}

	// **************************************************************************
	// MAIN
	// **************************************************************************
	public static void main(String[] args)
	{
		org.gerhardb.lib.util.startup.AppStarter.startUpApp(args, "org.gerhardb.jibs.Jibs", true);
		try
		{
			new PageLayoutDialog(null, null);
			System.out.println("Done");
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
	}
}
