/*
 * Copyright (c) 2011 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print.contactSheet;

import java.io.File;
import java.io.FileOutputStream;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JProgressBar;
import org.gerhardb.lib.print.*;
import org.gerhardb.lib.util.startup.AppStarter;

import com.itextpdf.text.BaseColor;
import com.itextpdf.text.Document;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfContentByte;
import com.itextpdf.text.pdf.PdfWriter;


/**
 * Easier to user replacement for java.Pageable.
 * Not really, but it servers the same purpose in this framework.
 */
public class ContactSheetPDF implements Convertible 
{
	static final String ALPHABET = "ABDCEFGHIJKLMNOPQRSTUVWXYZ abcdefghijklmnopqrstuvwxyz 123456789 ~!@#$%^&*()_+ `-={}|[]<>?,.:;'/\\\"";
	static final boolean SHOW_TEST = false;
	
	// Passed In
	private PageLayout pageLayout;
	ContactSheetInfo csInfo;
	private HeaderFooterInfo hfInfo;
	private boolean showFilesNotDirectories;
	private JFrame topFrame;
	
	// Computed
	private String[] fonts = new String[3];
	private InteriorLayout interior;
	private PicBoxSizes picBoxSizes;

	// iText
	Document document;
	PdfContentByte cb;
	BaseFont bf_helv;	
	private PdfWriter writer;
	
	// Other
	private JProgressBar picBar;	
	private ListToDraw drawIt;	
	private DrawAPic drawAPic;
	private ResizedImageFactory resizedImageFactory;
	
	public ContactSheetPDF(File[] f, boolean showFilesNotDirectoriesIn, 
			PageLayout pl, HeaderFooterInfo hfi, ContactSheetInfo csi, 
			File printTo, JFrame topFrameIn, JProgressBar bar)
	{
		this.showFilesNotDirectories = showFilesNotDirectoriesIn;
		this.pageLayout = pl;
		this.hfInfo = hfi;
		this.csInfo = csi;	
		this.topFrame = topFrameIn;
		this.picBar = bar;
		
		this.fonts[0] = this.hfInfo.headerFontFamily;
		this.fonts[1] = this.hfInfo.footerFontFamily;
		this.fonts[2] = this.csInfo.pictureFontFamily;
				
		// This needs picsPerPage
		// This will correctly set number of pages needed for HeaderFooter.
		this.drawIt = ListToDraw.makeListToDraw(f, 
				this.csInfo.rows, this.csInfo.cols, this.showFilesNotDirectories);
		this.picBar.setMaximum(this.drawIt.picsPerPage);
	
		this.interior = HeaderFooter.calcInterior(this.hfInfo, this.pageLayout.getInteriorLayout());	
		this.picBoxSizes = new PicBoxSizes(this.interior, this.csInfo, this.drawIt, this.pageLayout, true);

		this.drawAPic = new DrawAPic(
				this.csInfo.pictureFontSize, 
				this.picBoxSizes.picBoxWidth, 
				this.picBoxSizes.picBoxHeight, 
				this.csInfo);
		
		this.resizedImageFactory = new ResizedImageFactory(
				this.csInfo, 
				this.drawAPic.netPicWidth, 
				this.drawAPic.netPicHeight);
		
		try
		{
			printPDF(printTo);
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
			if (this.topFrame != null)
			{
		        JOptionPane.showMessageDialog(
		      	 this.topFrame,
		          ex.getMessage(),
		          AppStarter.getString("save.failed"), //$NON-NLS-1$
		          JOptionPane.ERROR_MESSAGE);				
			}
		}
		//System.out.println("ContactSheetPDF All Done!");
	}
	
	// ==========================================================================
	// Convertible Implementation
	// ==========================================================================
	public String convert(String text, int pageIndex)
	{
		if (text == null){return null;}
		text = text.replaceAll(HeaderFooterInfo.PAGE_INDEX, Integer.toString(pageIndex));
		text = text.replaceAll(HeaderFooterInfo.PAGE_COUNT, Integer.toString(ContactSheetPDF.this.drawIt.pageCount));		
		return this.drawIt.convert(text, pageIndex);
	} 

	// ==========================================================================
	// Working Code
	// ==========================================================================
	
	private void printPDF(File printTo)
	throws Exception
	{
		this.bf_helv = BaseFont.createFont(BaseFont.HELVETICA, "Cp1252", false);	
		
		//------------------------------------------------------------------------
		// Starting Up
		//------------------------------------------------------------------------
		// I think default size is A4.
		this.document = new Document(com.itextpdf.text.PageSize.LETTER);
		this.writer = PdfWriter.getInstance(this.document, new FileOutputStream(printTo));
		this.document.open();	
		this.cb = this.writer.getDirectContent();
		
		//------------------------------------------------------------------------
		// Print Loop
		//------------------------------------------------------------------------
		int pageCount = this.drawIt.pageCount;
		for(int i=1; i <= pageCount; i++)
		{
			printPage(i);
			this.document.newPage();
		}

		//------------------------------------------------------------------------
		// Closing down
		//------------------------------------------------------------------------
		
		// Don't forget to close!
		this.document.close();		
	}
	
	private void printPage(int pageNumber) throws Exception
	{
		//System.out.println("_________________________________________________");
		//System.out.println("Drawing page: " + pageNumber);
		
		// Fill the background with the appropriate color.
		this.cb.setColorFill(getBaseColor(this.csInfo.backgroundColor));
		this.cb.rectangle(0,0,this.pageLayout.getPageWidth(),this.pageLayout.getPageHeight());
		this.cb.fill();

		// Need to set text color...
		//g.setColor(info.textColor);	
		
		HeaderFooterInfo info = this.hfInfo.makePageCountAdjusted(pageNumber, this);	
		drawHeader(info);
		drawFooter(info);		
		
		ListToDraw.FilesForPage ffp = this.drawIt.getFilesForPage(pageNumber);
		if (ffp.filesToPrint())
		{
			drawImages(ffp);
		}
		else
		{				
			// Handle edge case of printing directories and the directory is empty.
			// So if nothing drawn on the first page, we will assume empty directory.
			// Only really applies if the DrawIt in question happens to have been
			// a directory.
			if (pageNumber == 1)
			{
				String printMe = "Directory had no pictures";	
				int textY = this.pageLayout.getPageHeight()/2;					
				this.cb.beginText();
				this.cb.setFontAndSize(this.bf_helv, 12.0f);
				this.cb.setColorFill(getBaseColor(this.hfInfo.textColor));
				this.cb.showTextAligned(
						PdfContentByte.ALIGN_CENTER, 
						printMe, 
						this.pageLayout.getInteriorLayout().getCenterX(),
						textY, 0);
				this.cb.endText();
			}
		}
	}
	
	private void drawImages(ListToDraw.FilesForPage ffp) 
	{
		for (int i=0; i<this.drawIt.picsPerPage; i++)
		{
			int drawIndex = ffp.startPicIndex + i;
			//System.out.println("drawImages: " + drawIndex);
			if (drawIndex < ffp.files.length)
			{			
				File imageFile = ffp.files[drawIndex];
				if (imageFile != null)
				{
					ResizedImage resizedImage = this.resizedImageFactory.makeResizedImage(imageFile, drawIndex, true);
					if (resizedImage != null)
					{
						resizedImage.xOffset = this.picBoxSizes.xPicBoxOffsets[i]; 
						resizedImage.yOffset = this.picBoxSizes.yPicBoxOffsets[i];
						//System.out.println("x: " + resizedImage.xOffset + " y: " + resizedImage.yOffset);
						this.drawAPic.print(this, resizedImage);
					}
				}
			}
			this.picBar.setValue(i + 1);
		}
	}
	
	private void drawHeader(HeaderFooterInfo info)
	throws Exception
	{
		if (SHOW_TEST)
		{
			this.cb.setColorStroke(BaseColor.RED);	
			this.cb.moveTo(
					this.pageLayout.getInteriorLayout().getLeftMargin(), 
					pdfY(this.pageLayout.getInteriorLayout().getTopMargin()));
			this.cb.lineTo(
					this.pageLayout.getInteriorLayout().getRightMargin(), 
					pdfY(this.pageLayout.getInteriorLayout().getTopMargin()));
			this.cb.stroke();
		}

		int topMargin = pdfY(this.interior.getTopMargin());
		if (info.showLines)
		{
			// Draw a line just underneath the text.	
			this.cb.setColorStroke(getBaseColor(this.hfInfo.textColor));	
			this.cb.moveTo(
					this.pageLayout.getInteriorLayout().getLeftMargin(), 
					topMargin);
			this.cb.lineTo(
					this.pageLayout.getInteriorLayout().getRightMargin(), 
					topMargin);
			this.cb.stroke();
		}
		
		/*
		java.awt.Font headerFont = new java.awt.Font(
				info.headerFontFamily, 
				java.awt.Font.PLAIN,
				info.headerFontSize);
		System.out.println("headerFont: " + headerFont);
		*/
		
		// PDF does text from the bottom up.
		int textY = topMargin + HeaderFooter.DEFAULT_MARGIN_PADDING_DPI; 

		this.cb.beginText();
		this.cb.setFontAndSize(this.bf_helv, info.headerFontSize);
		this.cb.setColorFill(getBaseColor(this.hfInfo.textColor));
		
		// Print the LEFT Header
		String printString = info.headerLeft;
		if (printString != null && printString.trim().length() > 0)
		{
			// Left Justified
			this.cb.showTextAligned(
					PdfContentByte.ALIGN_LEFT, 
					printString, 
					this.pageLayout.getInteriorLayout().getLeftMargin(),
					textY, 0);
		}
		
		// Print the CENTER Header
		printString = info.headerCenter;
		if (printString != null && printString.trim().length() > 0)
		{
			// Centered
			this.cb.showTextAligned(
					PdfContentByte.ALIGN_CENTER, 
					printString, 
					this.pageLayout.getInteriorLayout().getCenterX(),
					textY, 0);
		}

		// Right Header
		printString = info.headerRight;
		if (printString != null && printString.trim().length() > 0)
		{
			// Right Justified
			this.cb.showTextAligned(
					PdfContentByte.ALIGN_RIGHT, 
					printString, 
					this.pageLayout.getInteriorLayout().getRightMargin(),
					textY, 0);
		}		
		this.cb.endText();
	}

	private void drawFooter(HeaderFooterInfo info)
	throws Exception
	{
		if (SHOW_TEST)
		{
			this.cb.setColorStroke(BaseColor.RED);	
			this.cb.moveTo(
					this.pageLayout.getInteriorLayout().getLeftMargin(), 
					pdfY(this.pageLayout.getInteriorLayout().getBottomMargin()));
			this.cb.lineTo(
					this.pageLayout.getInteriorLayout().getRightMargin(), 
					pdfY(this.pageLayout.getInteriorLayout().getBottomMargin()));
			this.cb.stroke();
		}

		int bottomMargin = pdfY(this.interior.getBottomMargin());
		if (info.showLines)
		{
			// Draw a line just underneath the text.	
			this.cb.setColorStroke(getBaseColor(this.hfInfo.textColor));	
			this.cb.moveTo(
					this.pageLayout.getInteriorLayout().getLeftMargin(), 
					bottomMargin);
			this.cb.lineTo(
					this.pageLayout.getInteriorLayout().getRightMargin(), 
					bottomMargin);
			this.cb.stroke();
		}
		
		/*
		java.awt.Font headerFont = new java.awt.Font(
				info.headerFontFamily, 
				java.awt.Font.PLAIN,
				info.headerFontSize);
		System.out.println("headerFont: " + headerFont);
		*/
		
		// PDF does text from the bottom up.
		int textY = bottomMargin - info.footerFontSize;  //HeaderFooter.DEFAULT_MARGIN_PADDING_DPI; // 

		this.cb.beginText();
		this.cb.setFontAndSize(this.bf_helv, info.footerFontSize);
		this.cb.setColorFill(getBaseColor(this.hfInfo.textColor));
		
		// Print the LEFT Footer
		String printString = info.footerLeft;
		if (printString != null && printString.trim().length() > 0)
		{
			// Left Justified
			this.cb.showTextAligned(
					PdfContentByte.ALIGN_LEFT, 
					printString, 
					this.pageLayout.getInteriorLayout().getLeftMargin(),
					textY, 0);
		}
		
		// Print the CENTER Footer
		printString = info.footerCenter;
		if (printString != null && printString.trim().length() > 0)
		{
			// Centered
			this.cb.showTextAligned(
					PdfContentByte.ALIGN_CENTER, 
					printString, 
					this.pageLayout.getInteriorLayout().getCenterX(),
					textY, 0);
		}

		// Right Footer
		printString = info.footerRight;
		if (printString != null && printString.trim().length() > 0)
		{
			// Right Justified
			this.cb.showTextAligned(
					PdfContentByte.ALIGN_RIGHT, 
					printString, 
					this.pageLayout.getInteriorLayout().getRightMargin(),
					textY, 0);
		}		
		this.cb.endText();
	}
	
	/**
	 * Convert top down Y in Java to bottom up Y in PDF.
	 * @param Y
	 * @return
	 */
	public int pdfY(int y)
	{
		return this.pageLayout.getPageHeight() - y;
	}
	
	public static BaseColor getBaseColor(java.awt.Color color)
	{
		return new BaseColor(
				color.getRed(), 
				color.getGreen(), 
				color.getBlue(), 
				color.getAlpha());
	}
	
	// ==========================================================================
	//                       Main
	// ==========================================================================
	public static void main(String[] args)
	{
		org.gerhardb.lib.util.startup.AppStarter.startUpApp(
				args, "org.gerhardb.jibs.Jibs", true);
	
			File[] theList;
			boolean filesNotDir = true;
			if (filesNotDir)
			{
				//File dir = new File("D:/testpics/Fixed Files/contactSheet/a");
				//theList = dir.listFiles();
				
				theList = new File[]{new File("D:/testpics/Fixed Files/contactSheet/a/a.jpg")};
				theList = new File[]{new File("D:/testpics/Fixed Files/contactSheet/c/aaa-a.jpg")};
				theList = new File("D:/testpics/Fixed Files/contactSheet/b").listFiles();
			}
			else
			{
				File dirA = new File("D:/testpics/Fixed Files/contactSheet/a");
				File dirB = new File("D:/testpics/Fixed Files/contactSheet/b");
				File dirC = new File("D:/testpics/Fixed Files/contactSheet/c");
				theList = new File[]{dirA, dirB, dirC};
			}
			
			PageLayout pl = PageLayout.makeDefault();
			
			HeaderFooterInfo hfi = new HeaderFooterInfo();
			hfi.headerLeft = "Left";
			hfi.headerCenter = "<%DIR%>";
			hfi.headerRight = "Right";	
			
			ContactSheetInfo csi = new ContactSheetInfo();
			csi.rows = 2;
			csi.cols = 2;
			csi.showPictureName = true;
			csi.showPictureNumber = true;
			csi.showPictureSize = true;
			csi.frameSize = 1;
			csi.shadowSize = 4;
			csi.pictureFontSize = 10;
			csi.pictureFontFamily = "SansSerif";
			csi.pictureTextColor = java.awt.Color.BLACK;
			csi.backgroundColor = java.awt.Color.WHITE;
			csi.foregroundColor = java.awt.Color.BLACK;
			csi.resizeType = ContactSheetInfo.ResizeType.SHARP;
			csi.resizeDPI = ContactSheetInfo.ResizeDPI.DPI_300;
			
			try
			{
				File printTo = new File("d:/trash/hello-300.pdf");
				new ContactSheetPDF(
						theList, filesNotDir, pl, hfi, csi, printTo,
						null,
						new JProgressBar());
				
				Runtime.getRuntime().exec("\"C:/Program Files (x86)/Adobe/Reader 10.0/Reader/AcroRd32.exe\" " + printTo.getAbsolutePath() );
			}
			catch(Exception ex)
			{
				ex.printStackTrace();
			}
	} 		
}
