/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.print.contactSheet;

import java.awt.image.BufferedImage;
import java.io.File;

import org.gerhardb.lib.image.ImageChangeUtil;
import org.gerhardb.lib.io.FileUtil;

/**
 * ContactSheet
 */
public abstract class ResizedImage
{
	File file;
	private int originalWidth;
	private int originalHeight;
	protected int resizedWidth;
	protected int resizedHeight;
	protected int dpiHeight;
	protected int dpiWidth;

	public int xOffset;
	public int yOffset;
	String picName;
	String picSize;

	ResizedImage(File imageFile, ContactSheetInfo csInfo, int picCntList)
	{
		this.file = imageFile;
		if (csInfo.showPictureName)
		{
			// First let's make the string so we can see how big it is.
			int picNumber = picCntList + 1;
			if (csInfo.showPictureNumber)
			{
				this.picName = Integer.toString(picNumber)
						+ ": " + this.file.getName(); //$NON-NLS-1$
			}
			else
			{
				this.picName = this.file.getName();
			}
		}
	}

	public File getFile()
	{
		return this.file;
	}

	public int getOriginalHeight()
	{
		return this.originalWidth;
	}

	public int getOriginalWidth()
	{
		return this.originalHeight;
	}

	public int getResizedWidth()
	{
		return this.resizedWidth;
	}

	public int getResizedHeight()
	{
		return this.resizedHeight;
	}

	public int getDpiHeight()
	{
		return this.dpiHeight;
	}

	public int getDpiWidth()
	{
		return this.dpiWidth;
	}

	public abstract BufferedImage getImage();

	protected void setOriginalWidth(int originalWidthIn, int originalHeightIn,
			ContactSheetInfo csInfo)
	{
		this.resizedWidth = this.originalWidth = this.dpiWidth = originalWidthIn;
		this.resizedHeight = this.originalHeight = this.dpiHeight = originalHeightIn;
		if (csInfo.showPictureSize)
		{
			this.picSize = this.originalWidth
					+ "x" + this.originalHeight + "  " + FileUtil.getSize(this.file); //$NON-NLS-1$ //$NON-NLS-2$
		}
	}

	protected void calcToSize(int targetWidth, int targetHeight, int dpi)
	{
		if (targetWidth > 0 && targetHeight > 0)
		{
			// Separate name space for the theta's
			{
				float theta = ImageChangeUtil.calcThetaDown(this.originalWidth,
						this.originalHeight, targetWidth, targetHeight);
				// theta will equal zero if the image is smaller than the box it needs to go into.
				// theta greater than one would make the image bigger not smaller.
				if (theta > 0 && theta < 1)
				{
					this.resizedWidth = (int) (this.originalWidth * theta);
					this.resizedHeight = (int) (this.originalHeight * theta);
				}
			}
			
			if (dpi > 72)
			{
				float dpiTheta = ImageChangeUtil.calcThetaDown(
						this.originalWidth,
						this.originalHeight, 
						targetWidth * dpi / 72, 
						targetHeight * dpi / 72);
				// theta will equal zero if the image is smaller than the box it needs to go into.
				// theta greater than one would make the image bigger not smaller.
				if (dpiTheta > 0 && dpiTheta < 1)
				{
					this.dpiWidth = (int) (this.originalWidth * dpiTheta);
					this.dpiHeight = (int) (this.originalHeight * dpiTheta);
				}
				//System.out.println("Blow Up Target: " + this.dpiWidth + " " + this.dpiHeight);
			}
			else
			{
				this.dpiWidth = this.resizedWidth;
				this.dpiHeight = this.resizedHeight;				
			}

		}
		//System.out.println("ResizedImage OUTPUT width x height: " + this.resizedWidth + "x" + this.resizedHeight);
	}
}