/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.scroller;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.EventQueue;
import java.awt.KeyboardFocusManager;
import java.awt.event.ActionListener;
import java.awt.event.KeyListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;

import javax.imageio.ImageIO;
import javax.swing.BoundedRangeModel;
import javax.swing.ImageIcon;
import javax.swing.JFrame;
import javax.swing.ListModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import org.gerhardb.lib.util.startup.AppStarter;
import org.gerhardb.lib.dirtree.filelist.FileList;
import org.gerhardb.lib.image.IOImage;
import org.gerhardb.lib.image.ImageFactory;
import org.gerhardb.lib.util.Icons;
import org.gerhardb.lib.io.*;

/**
 * Commands to scroll among a group of pages. Used for making a slide show with
 * sound. Class is a singleton.
 */
public class Scroller implements IScroll, ListModel, BoundedRangeModel 
{
	// Not yet used...
	// What re between virtual machines?
	// THIS JUST FLAT DOES NOT WORK!!!
	/*
	 * public static final DataFlavor dataFlavor = new DataFlavor(
	 * "application/x-jibs-scroller/", "ScrollerDataFlavor");
	 */

	ArrayList<File> myList = new ArrayList<File>(1000);
	FileList myFileList; // pooh Why MyList and MyFileList?  Won't one do?  How related?
	ArrayList<ListDataListener> myListDataListeners = new ArrayList<ListDataListener>(5);
	ArrayList<ChangeListener> myBoundedRangeModelDataListeners = new ArrayList<ChangeListener>(5);
	ArrayList<ScrollerListener> myScrollerListeners = new ArrayList<ScrollerListener>(5);
	ArrayList<SlideShowListener> myShowListeners = new ArrayList<SlideShowListener>(5);

	boolean iBoundedRangeModelValueIsAdjusting = false;
	int mySortType = Scroller.SORT_OFF;
	int myPageSize = 10;
	int myIndex = -1;
	int myTimerDelay = 3000;

	ListMaker myListMaker;
	Component myFocusComponent;

	IOImage myCurrentIOImage;

	javax.swing.Timer myTimer;
	KeyListener myShowViewKeyListener = null;
	boolean iShowEndMessages = false;
	JFrame myFrame;
	private ScrollerKeyListener myScrollerKeyListener = new ScrollerKeyListener(this);

	// =========================================================================
	// Constructor
	// =========================================================================
	public Scroller()
	{
		// We don't care
	}

	public void setListMaker(ListMaker lm)
	{
		this.myListMaker = lm;
		reloadScroller();
	}
	
	public void setFileList(FileList fileList)
	{
		this.myFileList = fileList;
		this.myScrollerKeyListener.setListSelectionModel(this.myFileList.getFileListSelectionModel());	
	}
	
	public void updateSelectionInterval()
	{
		this.myScrollerKeyListener.updateSelectionInterval();
	}

	// =========================================================================
	// Static Methods
	// =========================================================================
	/*
	 * public static File[] convertFileList(File[] fileList) { File[]
	 * pages = new File[fileList.length]; for (int i = 0; i < pages.length;
	 * i++) { pages[i] = new File(fileList[i]); } return pages; } //
	 * convertFileList()
	 */

	// =========================================================================
	// IScroll
	// =========================================================================
	public ListMaker getListMaker(){return this.myListMaker;}
	
	public void reloadScroller(int index, boolean reloadeCache)
	{
		//new Throwable().printStackTrace();
		//System.out.println("========= Scroller RELOAD =========");
		Cursor oldCursor = this.myFrame.getCursor();
		this.myFrame.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try
		{
			this.myFileList.clearSelection();
			
			this.iBoundedRangeModelValueIsAdjusting = true;
			this.myIndex = -1;
			this.myList = new ArrayList<File>();
			if (this.myListMaker != null)
			{

				File[] list = this.myListMaker.getFileList();
				if (list != null)
				{
					int length = list.length;
					this.myList = new ArrayList<File>(length);
					for (int i = 0; i < length; i++)
					{
						this.myList.add(list[i]);
					}
					if (length > 0)
					{
						this.myIndex = 0;
					}
					doSort(false);
				}
			}

			// Silently ignore bad values
			// Negative OK if empty directory.
			if (index >= this.myList.size())
			{
				this.myIndex = this.myList.size() - 1;
			}
			else
			{
				this.myIndex = index;
			}

			// Now figure out what the correct size for File up, File down should
			// be.
			if (this.myList.size() < 51)
			{
				this.myPageSize = 5;
			}
			else if (this.myList.size() < 101)
			{
				this.myPageSize = 10;
			}
			else
			{
				this.myPageSize = this.myList.size() / 10;
			}

			this.iBoundedRangeModelValueIsAdjusting = false;
			
			// Now update the image.
			// Must do a structure changed to see the tick marks update.
			updateImage(ScrollerChangeEvent.LIST_RELOADED, reloadeCache);  

			kickListeners(new ScrollerListDataEvent(this, ListDataEvent.CONTENTS_CHANGED,
					0, this.myList.size(), reloadeCache));
		}
		catch (Exception ex)
		{
			// I have no reason to think that this could ever be used.
			// It's here because I'm trying to track down the mystry loss
			// of keyboard control.
			ex.printStackTrace();
		}
		this.myFrame.setCursor(oldCursor);
		//System.out.println("========= Scroller RELOAD Finished =========");
	}

	// =========================================================================
	// Getters and Setters
	// =========================================================================
	
	
	public String getDescription()
	{
		if (this.myListMaker != null)
		{
			return this.myListMaker.getDescription();
		}
		return null;
	}

	public ScrollerKeyListener getScrollerKeyListener()
	{
		return this.myScrollerKeyListener;
	}
	
	public String getStatusBarCount()
	{
		File File = getCurrentFile();
		if (File != null)
		{
			int pageShowing = getValueZeroBased() + 1;
			int count = getMaximumZeroBased() + 1;
			//return pageShowing + " " + AppStarter.getString("Scroller.0") + " " + count + " - " + File.toString(); //$NON-NLS-1$ //$NON-NLS-2$
			return pageShowing + " " + AppStarter.getString("Scroller.0") + " " + count; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		return ""; //$NON-NLS-1$
	}
	
	public String getStatusBarFileName()
	{
		File File = getCurrentFile();
		if (File != null)
		{
			return File.toString(); 
		}
		else if (this.myListMaker != null)
		{
			return this.myListMaker.getDescription();
		}
		return ""; //$NON-NLS-1$
	}

	public void setShowViewKeyListener(KeyListener k)
	{
		this.myShowViewKeyListener = k;
	}

	public void setEndMessages(boolean showEndMessages)
	{
		this.iShowEndMessages = showEndMessages;
	}

	public File[] getSortedFiles()
	{
		File[] rtnMe = new File[this.myList.size()];
		this.myList.toArray(rtnMe);
		return rtnMe;
	}
	
	// =========================================================================
	// ListModel Implementation - REALLY ONLY USED BY THE SWING SLIDER.
	// =========================================================================
	public void addListDataListener(ListDataListener l)
	{
		synchronized (this.myListDataListeners)
		{
			this.myListDataListeners.add(l);
		}
	}

	public Object getElementAt(int index)
	{
		if (this.myList == null) { return null; }
		if (index >= this.myList.size()) { return null; }
		return this.myList.get(index);
	}

	public int getSize()
	{
		return this.myList.size();
	}

	public void removeListDataListener(ListDataListener l)
	{
		synchronized (this.myListDataListeners)
		{
			this.myListDataListeners.remove(l);
		}
	}

	// =========================================================================
	// BoundedRangeModel Implementation - Used for other stuff
	// =========================================================================
	public int getValueZeroBased()
	{
		return this.myIndex;
	}

	public int getMaximumZeroBased()
	{
		return this.myList.size() - 1;
	}

	// =========================================================================
	// BoundedRangeModel Implementation
	// =========================================================================
	public int getValue()
	{
		// Real problem with counting from one is that the tick marks are bad.
		// They go 1, 6, 11 instead of 1, 5, 10
		return this.myIndex + 1;
		//return getValueZeroBased();
	}

	/**
	 * Min and max are zero based which works well for the JSlider.
	 * 
	 * @return -1 if no items
	 */
	public int getMaximum()
	{
		// See note on getValue()
		return this.myList.size();
		//return getMaximumZeroBased();
	}


	public int getMinimum()
	{
		return 0;
	}
	
	/**
	 * 
	 * @param newValue
	 *           Invalid values are silently ignored.
	 */
	public void setValue(int newValue)
	{
		if (newValue >= 0 && newValue < this.myList.size())
		{
			Cursor priorCursor = this.myFrame.getCursor();
			this.myFrame.setCursor(Cursor.getPredefinedCursor(
					Cursor.WAIT_CURSOR));
			this.myIndex = newValue;
			updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);					
			this.myFrame.setCursor(priorCursor);
		}
	}

	public void addChangeListener(ChangeListener x)
	{
		synchronized (this.myBoundedRangeModelDataListeners)
		{
			this.myBoundedRangeModelDataListeners.add(x);
		}
	}

	public void removeChangeListener(ChangeListener x)
	{
		synchronized (this.myBoundedRangeModelDataListeners)
		{
			this.myBoundedRangeModelDataListeners.remove(x);
		}
	}

	public int getExtent()
	{
		return 0;
	}

	public boolean getValueIsAdjusting()
	{
		return this.iBoundedRangeModelValueIsAdjusting;
	}

	public void setValueIsAdjusting(boolean b)
	{
		this.iBoundedRangeModelValueIsAdjusting = b;
		if (false == b)
		{
			updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
		}
	}

	// We just ignore this since the internal list controls this.
	public void setRangeProperties(int value, int extent, int min, int max,
			boolean adjusting)
	{
		setValueIsAdjusting(adjusting);
	}

	// We just ignore this since the internal list controls this.
	public void setExtent(int newExtent)
	{
		// We don't care
	}

	// We just ignore this since the internal list controls this.
	public void setMaximum(int newMaximum)
	{
		// We don't care
	}

	// We just ignore this since the internal list controls this.
	public void setMinimum(int newMinimum)
	{
		// We don't care
	}
	
	// =========================================================================
	// Sort Related Items
	// =========================================================================
	public static final int SORT_OFF = 0;
	public static final int SORT_NAME_CASE_INSENSATIVE = 1;
	public static final int SORT_NAME_CASE_SENSATIVE = 2;
	public static final int SORT_DATE = 3;
	public static final int SORT_LENGTH = 4;

	public void sort(int type)
	{
		if (type < SORT_OFF || type > SORT_LENGTH) { throw new IllegalArgumentException(
				"sort type out of range"); } //$NON-NLS-1$
		this.mySortType = type;
		doSort(true);
	}

	public void doSort(final boolean realSort)
	{
		if (this.myFrame != null)
		{
			SwingUtilities.invokeLater(new Runnable()
			{
				public void run()
				{
					Scroller.this.myFrame.setCursor(Cursor.getPredefinedCursor(
							Cursor.WAIT_CURSOR));
				}
			});
		}

		File returnTo = this.getCurrentFile();
		
		try
		{
			// Now decide how to sort the list.
			switch (this.mySortType)
			{
			case SORT_NAME_CASE_INSENSATIVE:
				Collections.sort(this.myList, new FileNameComparatorInsensative());
				break;
			case SORT_NAME_CASE_SENSATIVE:
				Collections.sort(this.myList, new FileNameComparatorSensative());
				break;
			case SORT_DATE:
				Collections.sort(this.myList, new FileDateComparator());
				break;
			case SORT_LENGTH:
				Collections.sort(this.myList, new FileLengthComparator());
				break;
			case SORT_OFF:
			default:
				// Do nothing.
				if (realSort)
				{
					reloadScroller(0, IScroll.KEEP_CACHE);
				}
			}
			if (realSort)
			{
				kickListeners(new ScrollerChangeEvent(ScrollerChangeEvent.CURRENT_PAGE_CHANGED, IScroll.KEEP_CACHE), true);
				this.selectFile(returnTo);
			}
		}
		finally
		{
			if (this.myFrame != null)
			{
				SwingUtilities.invokeLater(new Runnable()
				{
					public void run()
					{
						Scroller.this.myFrame.setCursor(Cursor.getDefaultCursor());
					}
				});
			}
		}
	}

	// =========================================================================
	// File Related Items
	// =========================================================================
	public File getCurrentFile()
	{
		return getFile(this.myIndex);
	}

	public File getFile(int index)
	{
		if (index < 0 || index >= this.myList.size()) { return null; }
		return this.myList.get(index);
	}

	public void reloadScroller()
	{
		reloadScroller(0, IScroll.RELOAD_CACHE);
	}

	public void selectFile(File file)
	{
		if (file == null)
		{
			return;
		}
		File[] sortedFiles = new File[this.myList.size()];
		this.myList.toArray(sortedFiles);
		for(int i=0; i<sortedFiles.length; i++)
		{
			if (file.equals(sortedFiles[i]))
			{
				this.myIndex = i;
				updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
				break;
			}
		}
	}
	

	/**
	 * Used for DnD. Note that it only removes from scroll list, not the physical
	 * File.
	 */
	public void removeCurrentPage()
	{
		if (this.myList.size() < 1 || this.myIndex < 0) { return; }

		// Record event before changing myIndex.
		int indexRemoved = this.myIndex;

		// Get rid of the old current item.
		File removedPage = this.myList.remove(indexRemoved);

		kickListeners(new ListDataEvent(this, ListDataEvent.INTERVAL_REMOVED,
				indexRemoved, indexRemoved));
		
		ScrollerChangeEvent event = new ScrollerChangeEvent(
				ScrollerChangeEvent.ONE_FILE_REMOVED, indexRemoved,
				removedPage);

		if (this.myIndex == this.myList.size())
		{
			this.myIndex = this.myList.size() - 1;
		}

		// Force getting a new image, and update of tick marks.
		kickListeners(event, true);
		requestFocus();
	}

	public File[] getPictureFiles()
	{
		File[] rtnMe = new File[this.myList.size()];
		for (int i = 0; i < rtnMe.length; i++)
		{
			rtnMe[i] = this.myList.get(i);
		}
		return rtnMe;
	}

	// =========================================================================
	// Movement Items
	// =========================================================================

	/**
	 * Set File size.
	 * 
	 * @param pageSize
	 *           How many items File up and File down should Scroller.
	 */
	public void setPageSize(int pageSize)
	{
		this.myPageSize = pageSize;
	}

	public void up()
	{
		if (this.myIndex > 0)
		{
			this.myIndex--;
			updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
		}
		else
		{
			if (this.iShowEndMessages)
			{
				endOfList(false);
			}
		}
	}

	public void down()
	{
		// IF YOU CHANGE THIS LOGIC, ADJUST THE SHOW LOGIC
		// AT startSlideShow(). THIS IS COPIED THERE.
		if (this.myIndex < (this.myList.size() - 1))
		{
			this.myIndex++;
			updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
		}
		else
		{
			if (this.iShowEndMessages)
			{
				endOfList(true);
			}
		}
	}

	public void pageUp()
	{
		if (this.myIndex > this.myPageSize)
		{
			this.myIndex = this.myIndex - this.myPageSize;
		}
		else if (this.myIndex <= 0)
		{
			endOfList(false);
		}
		else
		{
			this.myIndex = 0;
		}
		updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
	}

	public void pageDown()

	{
		System.out.println("Scroller pageDown()");
		int endOfList = this.myList.size() - 1;
		if (this.myIndex < (endOfList - this.myPageSize))
		{
			this.myIndex = this.myIndex + this.myPageSize;
		}
		else if (this.myIndex >= endOfList)
		{
			endOfList(true);
		}
		else
		{
			this.myIndex = endOfList;
		}
		updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
	}

	public void home()
	{
		if (this.myList.size() > 0)
		{
			this.myIndex = 0;
		}
		else
		{
			this.myIndex = -1;
		}
		updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
	}

	public void end()
	{
		this.myIndex = this.myList.size() - 1;
		updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
	}

	/**
	 * Jumps to some percentage down the list.
	 * 
	 * @param jumpTo
	 *           between 0 and 100
	 */
	/*
	public void jumpToPercent(int jumpTo)
	{
		if (jumpTo < 0 || jumpTo > 100)
		{
			throw new IllegalArgumentException("jump must be between 0 and 100"); //$NON-NLS-1$
		}
		else if (jumpTo == 0)
		{
			home();
		}
		else if (jumpTo == 100)
		{
			end();
		}
		else
		{
			myIndex = (int) (((double) myList.size()) * ((double) jumpTo) / 100);
		}
		updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
	}
*/
	
	/**
	 * Jump to a particular object.
	 * 
	 * @param jumpTo
	 *           where to jump
	 * @return true if jump worked
	 */
	public boolean jumpTo(Object jumpTo)
	{
		int index = this.myList.indexOf(jumpTo);
		if (index > 0)
		{
			setValue(index);
			return true;
		}
		return false;
	}

	/**
	 * Jump to a particular index.
	 * 
	 * @param index
	 *           where to jump
	 * @return true if jump worked
	 */
	public void jumpTo(int index)
	{
		// Don't do anything if nothing is showing.
		if (this.myList.size() == 0) { return; }

		if (index > this.myList.size())
		{
			setValue(this.myList.size());
		}
		else if (index < 0)
		{
			setValue(-1);
		}
		else
		{
			setValue(index);
		}
	}

	// =========================================================================
	// Slide Show Items
	// =========================================================================

	public synchronized void addSlideShowListener(SlideShowListener l)
	{
		this.myShowListeners.add(l);
	}

	public synchronized void removeSlideShowListener(SlideShowListener l) // NO_UCD
	{
		this.myShowListeners.remove(l);
	}

	public synchronized void startSlideShow(final boolean continuous)
	{
		if (this.myTimer != null) { return; }

		notifySlideShowListeners(true);

		// Start from the beginning
		//home();

		// Start the actual show
		this.myTimer = new javax.swing.Timer(this.myTimerDelay, new ActionListener()
		{
			public void actionPerformed(java.awt.event.ActionEvent ae)
			{
				boolean moreFiles = true;
				// Loop till we find a File that exists.
				while (moreFiles)
				{
					if (Scroller.this.myIndex < (Scroller.this.myList.size() - 1))
					{
						Scroller.this.myIndex++;
						File currentFile = getCurrentFile();
						if (currentFile != null && currentFile.exists())
						{
							break;
						}
					}
					else
					{
						moreFiles = false;
					}
				}

				if (moreFiles)
				{
					updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
				}
				else
				{
					if (continuous)
					{
						reloadScroller();
						home();
					}
					else
					{
						// This will make the end of File message come up.
						endOfList(true);
						stopSlideShow();
					}
				}

				/*
				 * // Don't keep going past end of list! if (!down(false)) { if
				 * (continuous) { reload(); home(); } else { // This will make the
				 * end of File message come up. down(); stopSlideShow(); } }
				 */
			}
		});
		this.myTimer.start();
	}

	public synchronized void stopSlideShow()
	{
		if (this.myTimer != null)
		{
			this.myTimer.stop();
			this.myTimer = null;
			notifySlideShowListeners(false);
		}
	}

	public boolean isSlideShowRunning()
	{
		if (this.myTimer == null)
		{
			return false;
		}
		return true;
	}

	private void notifySlideShowListeners(final boolean running)
	{
		EventQueue.invokeLater(new Runnable()
		{
			public void run()
			{
				Iterator<SlideShowListener> loop = Scroller.this.myShowListeners.iterator();
				while (loop.hasNext())
				{
					loop.next().slideShow(running);
				}
			}
		});
	}

	// =========================================================================
	// Focus Items
	// =========================================================================

	public void setFrame(JFrame frame)
	{
		this.myFrame = frame;
	}

	/**
	 * Component to get focus when value is changed.
	 * 
	 * @param focus
	 *           what get focus
	 */
	public void setAutoFocus(Component focus)
	{
		this.myFocusComponent = focus;
		this.myFocusComponent.addMouseWheelListener(getMouseWheelListener());
		requestFocus();
	}

	public void requestFocus()
	{
		if (this.myFocusComponent != null)
		{
			// null is returned if none of the components in this application has the focus
		  Component compFocusOwner =
		        KeyboardFocusManager.getCurrentKeyboardFocusManager().getFocusOwner();
		  if (compFocusOwner != null)
		  {
			  if (!(compFocusOwner instanceof FileList))
			  {
  				  //System.out.println(compFocusOwner);
			  		this.myFocusComponent.requestFocus();
			  }
		  }
		  else
		  {
			  this.myFocusComponent.requestFocus();			  
		  }
		}
	}
	
	public void forceFocus()
	{
		if (this.myFocusComponent != null)
		{
			this.myFocusComponent.requestFocus();		
		}
	}

	public Component getFocusComponent()
	{
		return this.myFocusComponent;
	}

	// =========================================================================
	// Image Items
	// =========================================================================

	public void refreshCurrentImage()
	{
		updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
	}

	public IOImage getCurrentIOImage()
	{
		// myCurrentIOImage is set to null to trigger reloading.
		if (this.myCurrentIOImage == null)
		{
			File file = getCurrentFile();
			if (file != null)
			{
				this.myCurrentIOImage = ImageFactory.getImageFactory().makeImageEZ(file);
			}
		}
		return this.myCurrentIOImage;
	}

	 public BufferedImage getCurrentImage()
	 {
		 IOImage current = getCurrentIOImage();
		 if (current == null)
		 {
			 return null;
		 }
		 try
		 {
			 return current.getImage();
		 }
		 catch (Exception ex)
		 {
			 System.out.println("Scroller.getCurrentImage FAILED"); //$NON-NLS-1$
			 System.out.println(ex.getMessage());
			 ex.printStackTrace();
		 }
		 return null;
	 }
	 
	// =========================================================================
	// ScrollerListener
	// =========================================================================
	public void addScrollerListener(ScrollerListener x)
	{
		synchronized (this.myScrollerListeners)
		{
			this.myScrollerListeners.add(x);
		}
	}

	public void removeScrollerListener(ScrollerListener x)
	{
		synchronized (this.myScrollerListeners)
		{
			this.myScrollerListeners.remove(x);
		}
	}

	// =========================================================================
	// Public Methods
	// =========================================================================
	public void setSlideFlipDelay(int millisedonds)
	{
		this.myTimerDelay = millisedonds;
	}

	public int getSlideFlipDelay()
	{
		return this.myTimerDelay;
	}

	public MouseWheelListener getMouseWheelListener()
	{
		return new MouseWheelListener()
		{
			public void mouseWheelMoved(MouseWheelEvent event)
			{
				int units = event.getUnitsToScroll();
				/*
				 * int amount = event.getScrollAmount(); int type =
				 * event.getScrollType(); System.out.println( "u: " + units + " a: " +
				 * amount + " t: " + type);
				 */
				if (units > 0)
				{
					down();
				}
				else if (units < 0)
				{
					up();
				}
			}
		};
	}

	public boolean isBeyond()
	{
		if (this.myIndex < 0 || this.myIndex >= this.myList.size())
		{
			return true;
		}
		return false;
	}

	public BufferedImage getBeyondBounds()
	{
		java.net.URL url = null;
		if (getSize() == 0)
		{
			url = Icons.getURL(Icons.NO_PICTURES);
		}
		else if (this.myIndex < 0)
		{
			url = Icons.getURL(Icons.BEGINNING);
		}
		else if (this.myIndex >= this.myList.size())
		{
			url = Icons.getURL(Icons.END);
		}
		if (url == null) { return null; }
		try
		{
			return ImageIO.read(url);
		}
		catch (IOException ex)
		{
			ex.printStackTrace();
		}
		return null;
	}

	public ImageIcon getBeyondBoundsIcon()
	{
		BufferedImage img = getBeyondBounds();
		if (img == null) { return null; }
		return new ImageIcon(img);
	}

	// =========================================================================
	// Private Items
	// =========================================================================

	void updateImage(int scrollerChangeEventType)
	{
		updateImage(scrollerChangeEventType, IScroll.KEEP_CACHE);
	}
	
	void updateImage(int scrollerChangeEventType, boolean reloadCache)
	{
		//System.out.println("ds index: " + myIndex);
		// This is how we tell JIBS components what's up!
		kickListeners(new ScrollerChangeEvent(scrollerChangeEventType, reloadCache), true);
		// This is the Swing default type event we are sending.
		kickListeners(new ChangeEvent(this));
		// System.out.println("updateImage requesting focus");
		requestFocus();
	}

	/**
	 * Does not reset the slider.
	 */
	public void editedImage()
	{
		// This is how we tell JIBS components what's up!
		System.out.println("Scroller Kicking Listeners");
		ScrollerChangeEvent sce = new ScrollerChangeEvent(ScrollerChangeEvent.CURRENT_IMAGE_CHANGED, IScroll.KEEP_CACHE);
		kickListeners(sce, false);
		System.out.println("Scroller Kicked Listeners");
		requestFocus();
	}

	/**
	 * Used for File sorting, deletion, etc when the image needs to be updated
	 * WITHOUT replaying the current sounds.
	 * 
	 * @param event
	 *           event to send to listeners
	 */
	private void kickListeners(ScrollerChangeEvent event, boolean invalidate)
	{
		// First, invalidate the current File to force the File to be looked up.
		if (invalidate)
		{
			//	myCurrentIOImage is set to null to trigger reloading.
			this.myCurrentIOImage = null;
		}

		// Now tell the world about it.
		ScrollerListener[] listeners = null;
		synchronized (this.myScrollerListeners)
		{
			listeners = new ScrollerListener[this.myScrollerListeners.size()];
			this.myScrollerListeners.toArray(listeners);
		}
		for (int i = 0; i < listeners.length; i++)
		{
			listeners[i].scrollerChanged(event);
		}
	}

	private void kickListeners(ListDataEvent event)
	{
		ListDataListener[] listeners = null;
		synchronized (this.myListDataListeners)
		{
			listeners = new ListDataListener[this.myListDataListeners.size()];
			this.myListDataListeners.toArray(listeners);
		}
		for (int i = 0; i < listeners.length; i++)
		{
			switch (event.getType())
			{
			case ListDataEvent.CONTENTS_CHANGED:
				listeners[i].contentsChanged(event);
				break;
			case ListDataEvent.INTERVAL_ADDED:
				listeners[i].intervalAdded(event);
				break;
			case ListDataEvent.INTERVAL_REMOVED:
				listeners[i].intervalRemoved(event);
				break;
			}
		}
	}

	private void kickListeners(ChangeEvent event)
	{
		ChangeListener[] listeners = null;
		synchronized (this.myBoundedRangeModelDataListeners)
		{
			listeners = new ChangeListener[this.myBoundedRangeModelDataListeners.size()];
			this.myBoundedRangeModelDataListeners.toArray(listeners);
		}
		for (int i = 0; i < listeners.length; i++)
		{
			listeners[i].stateChanged(event);
		}
	}

	void endOfList(boolean end)
	{
		if (end)
		{
			this.myIndex = this.myList.size();
		}
		else
		{
			this.myIndex = -1;
		}
		updateImage(ScrollerChangeEvent.CURRENT_PAGE_CHANGED);
	}
}
