/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.swing;

import java.awt.Component;
import java.awt.HeadlessException;
import java.awt.Rectangle;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.prefs.Preferences;

import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileSystemView;

import org.gerhardb.lib.io.IOUtils;

/**
 * JFileChooser which remembers its size.
 */
public class JFileChooserExtra extends JFileChooser
{
	private static final String	SAVE_HEIGHT	= "saveHeight"; //$NON-NLS-1$
	private static final String	SAVE_WIDTH	= "pickWidth"; //$NON-NLS-1$
	private static final String	SAVE_SET	= "pickSet"; //$NON-NLS-1$
	
	Preferences myPrefs;
	JDialog myDialog;
	
	public JFileChooserExtra()
	{
		super();
	}

	public JFileChooserExtra(File currentDirectory) // NO_UCD
	{
		super(IOUtils.validDirectory(currentDirectory));
	}

	public JFileChooserExtra(File currentDirectory, FileSystemView fsv) // NO_UCD
	{
		super(IOUtils.validDirectory(currentDirectory), fsv);
	}

	public JFileChooserExtra(FileSystemView fsv) // NO_UCD
	{
		super(fsv);
	}

	public JFileChooserExtra(String currentDirectoryPath)
	{
		super(IOUtils.validDirectory(currentDirectoryPath));
	}

	public JFileChooserExtra(String currentDirectoryPath, FileSystemView fsv) // NO_UCD
	{
		super(IOUtils.validDirectory(currentDirectoryPath), fsv);
	}

	/**
	 * Gets added to end of prefs so either one word or forward slashed tree.
	 * Do not include an beginning slash.
	 * @param name
	 */
	public void setSaveName(String appName, String name)
	{
		if ( name != null && name.length() > 0)
		{
			this.myPrefs = Preferences.userRoot().node("/" + appName + "/org/gerhardb/lib/swing/JFileChooser/" + name); //$NON-NLS-1$
		}
	}
	
	@Override
	protected JDialog createDialog(Component parent) throws HeadlessException
	{
		//System.out.println("JFileChooserExtra");
		this.myDialog = super.createDialog(parent);
		if (this.myPrefs == null)
		{
			return this.myDialog;			
		}
		
		this.myDialog.addWindowListener(new WindowAdapter()
		{
			@Override
			public void windowDeactivated(WindowEvent e) 
			{
				//System.out.println("JFileChooserExtra closing DEACTIVATED");
				saveInfo();				
			}
		});
		
		if (this.myPrefs.getBoolean(SAVE_SET, false))
		{
			try
			{
				Rectangle rect = this.myDialog.getBounds();
				int width = (int) rect.getWidth();
				int height = (int) rect.getHeight();
				//System.out.println("initial: " + width + " " + height);
				width = this.myPrefs.getInt(SAVE_WIDTH, width);
				height = this.myPrefs.getInt(SAVE_HEIGHT, height);
				//System.out.println("looked up: " + width + " " + height);
				rect.setSize(width, height);
				this.myDialog.setBounds(rect);

				this.myDialog.setLocationRelativeTo(parent);
			} 
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}		

		return this.myDialog;
	}
	
	void saveInfo()
	{
		//System.out.println("===JFileChooserExtra.saveInfo===");
		try
		{
			Rectangle rect = this.myDialog.getBounds();
			int width = (int) rect.getWidth();
			int height = (int) rect.getHeight();
			//System.out.println("saving: " + width + " " + height);
			this.myPrefs.putInt(SAVE_WIDTH, width);
			this.myPrefs.putInt(SAVE_HEIGHT, height);
			this.myPrefs.putBoolean(SAVE_SET, true);
		} catch (Exception ex)
		{
			this.myPrefs.putBoolean(SAVE_SET, false);
			ex.printStackTrace();
		}

	}
}