/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.swing;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.LayoutManager;

import javax.swing.JPanel;

/**
 Makes bands of Panels with a left flow layout.
 <p>
 Used to make a left justified display.
 First, create a top row.  This gives back a panel to add stuff to.
 Then, create a next row and add stuff to it.
 <p>
 Implemented as a BorderLayout using North and Center.
 This means you can use east, west, and south directly.
 However, using south would be poor form.
 */
public class JPanelRows
   extends JPanel
{
   JPanel myLastPanel;
   int myDefaultAlignment = FlowLayout.LEFT;

   public JPanelRows()
   {
      super(new BorderLayout());
      this.myLastPanel = this;
   }

   public JPanelRows(int defaultFlowLayoutAlignment)
   {
      super(new BorderLayout());
      this.myLastPanel = this;
      this.myDefaultAlignment = defaultFlowLayoutAlignment;
   }

   public JPanel topRow()
   {
      return topRow(this.myDefaultAlignment);
   }

   public JPanel topRow(int flowLayoutAlignment)
   {
      JPanel row = new JPanel(new FlowLayout(flowLayoutAlignment));
      row.setBackground(this.getBackground());
      add(row, BorderLayout.NORTH);
      return row;
   }
   
   public JPanel topRow(LayoutManager layout)
   {
      JPanel row = new JPanel(layout);
      row.setBackground(this.getBackground());
      add(row, BorderLayout.NORTH);
      return row;
   }

   public JPanel nextRow()
   {
      return nextRow(this.myDefaultAlignment);
   }

   /**
      Add a row using only the top.
      Returns new row.
      flowLayoutAlignment: <br>
      FlowLayout.LEFT<br>
      FlowLayout.CENTER<br>
      FlowLayout.RIGHT<br>
    */
   public JPanel nextRow(int flowLayoutAlignment)
   {
      // Put on north end of new panel.
      JPanel newPanel = new JPanel(new BorderLayout());
      JPanel row = new JPanel(new FlowLayout(flowLayoutAlignment));
      newPanel.setBackground(this.getBackground());
      row.setBackground(this.getBackground());
      newPanel.add(row, BorderLayout.NORTH);
      // New panel goes in middle of last panel.
      this.myLastPanel.add(newPanel, BorderLayout.CENTER);
      // New panel is now the last panel - to which next center goes.
      this.myLastPanel = newPanel;
      return row;
   }

   public JPanel nextRow(LayoutManager layout)
   {
      // Put on north end of new panel.
      JPanel newPanel = new JPanel(new BorderLayout());
      JPanel row = new JPanel(layout);
      newPanel.setBackground(this.getBackground());
      row.setBackground(this.getBackground());
      newPanel.add(row, BorderLayout.NORTH);
      // New panel goes in middle of last panel.
      this.myLastPanel.add(newPanel, BorderLayout.CENTER);
      // New panel is now the last panel - to which next center goes.
      this.myLastPanel = newPanel;
      return row;
   }

   @Override
	public Component add(Component addMe)
   {
      super.add(addMe);
      return addMe;
   }
} 
