/*
 * Copyright (c) 2007 Gerhard Beck.  All rights reserved.
 *
 * Subject to the GNU GENERAL PUBLIC LICENSE,
 * Version 3, 29 June 2007 http://www.gnu.org/licenses/gpl.html
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL GERHARD BECK OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.gerhardb.lib.util;

import java.awt.AWTEvent;
import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.EventQueue;
import java.awt.KeyboardFocusManager;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyVetoException;
import java.beans.VetoableChangeListener;
import java.util.Date;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BoundedRangeModel;
import javax.swing.DefaultBoundedRangeModel;
import javax.swing.Icon;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.KeyStroke;
import javax.swing.RootPaneContainer;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.WindowConstants;

import org.gerhardb.lib.swing.JPanelRows;

/**
 * Blocks user interaction during which the application
 * needs to behave modally providing a modal dialog box with a progress meter.  
 * 
 * All done out of the AWT thread so the application
 * can be providing visual feedback to the user as the operation is
 * processed.
 */
public class ModalProgressDialogFast
{
   Window myTopWindow;
   String myTitle;
   String myMessage;
   String myCancelBtnText;
   BoundedRangeModel myRange;
   Thread myTaskThread;
   
   boolean iWasDeactivated = false;
   boolean iWasUnistalled = false;
   
   public static final Icon WAIT_ICON = Icons.getIcon(Icons.HOUR_GLASS);
   JDialog myDialog;
   Date showTime;
   String finishedSync = "sync"; //$NON-NLS-1$
   boolean iAmFinished = false;
   KeyboardFocusManager myKeyboardFocusManager;

   public ModalProgressDialogFast(Window topWindow, String title, String message,
			String cancelBtnText, BoundedRangeModel range, Runnable runMe)
	{
		this.myTaskThread = new Thread(runMe);
		this.myTopWindow = topWindow;
		this.myTitle = title;
		this.myMessage = message;
		this.myCancelBtnText = cancelBtnText;
		this.myRange = range;
		
		install();
      activate();
		
		Thread dialogThread = new Thread()
      {
         @Override
			public void run()
         {
            EventQueue.invokeLater(new Runnable()
            {
               public void run()
               {
                  //activate();
                  ModalProgressDialogFast.this.myDialog.setVisible(true);
                  ModalProgressDialogFast.this.showTime = new Date();
               }
            });
         }
      };   		
		dialogThread.start();
	}

   // =========================================================================
   // Public Methods
   // =========================================================================

   public void start()
   {		
   	// *** START EVERYTHING UP ***
		// Starts the actual work!
   	// Do after created so recreation dialog will return before this starts.
		this.myTaskThread.start();
   }
   
   /**
    * Halts any executing task, and hides this object's dialog if it
    * is displayed.  The executing task, if any, will be interrupted;
    * it is up to that task to terminate itself based on its
    * interruption status.
    */
   public void interrupt() // NO_UCD
   {
      if (this.myDialog != null)
      {
         this.myTaskThread.interrupt();
      }
   }
   
   /**
    * Called when the task is done to make sure dialog goes away fast...
    */
   public void done()
   {
		try
		{
			uninstallGlassPane();
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
		}
		deactivateDialog();  	
   }

   // =========================================================================
   // Private Methods
   // =========================================================================

   private synchronized void activate()
   {
      // Don't bother if the work is already done.
      synchronized (this.finishedSync)
      {
         if (this.iAmFinished == true)
         {
            return;
         }
      }

      //System.out.println("ACTIVATE CALLED");

      Action cancelAction = new AbstractAction(
                            UIManager.getString("OptionPane.cancelButtonText")) //$NON-NLS-1$
      {
         public void actionPerformed(ActionEvent event)
         {
            //System.out.println("CANCEL BUTTON PRESSED");
         	ModalProgressDialogFast.this.myTaskThread.interrupt();
         }
      };

      // Set up cancel button
      JButton cancelButton = new JButton(cancelAction);
      cancelButton.setText(this.myCancelBtnText);
      cancelButton.setDefaultCapable(true);

      // Default keystroke actions for cancel button
      Object actionID = "cancel"; //$NON-NLS-1$
      cancelButton.getActionMap().put(actionID, cancelAction);
      InputMap inputMap =
         cancelButton.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
      inputMap.put(KeyStroke.getKeyStroke("ESCAPE"), actionID); //$NON-NLS-1$
      inputMap.put(KeyStroke.getKeyStroke("CANCEL"), actionID); //$NON-NLS-1$
      inputMap.put(KeyStroke.getKeyStroke("STOP"), actionID); //$NON-NLS-1$

      // Set up viewable panel
      JPanel option = new JPanel(new BorderLayout());
      option.add(new JProgressBar(this.myRange), BorderLayout.NORTH);
      option.add(cancelButton, BorderLayout.SOUTH);

      JOptionPane optionPane = new JOptionPane(
                               " ", //$NON-NLS-1$
                               JOptionPane.PLAIN_MESSAGE,
                               JOptionPane.DEFAULT_OPTION,
                               WAIT_ICON,
                               new Object[]
                               {option},
                               cancelButton);

      optionPane.setMessage(this.myMessage);
      this.myDialog = optionPane.createDialog(this.myTopWindow, this.myTitle);

      this.myDialog.setModal(false);
      this.myDialog.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

      // If the owning windows goes away, we want to also.
      // Also, if user closes the dialog stop everything.
      this.myDialog.addHierarchyListener(new HierarchyListener()
      {
         public void hierarchyChanged(HierarchyEvent event)
         {
            long flags = event.getChangeFlags();
            Component component = event.getComponent();
            if ((flags & HierarchyEvent.SHOWING_CHANGED) != 0)
            {
               if (!component.isShowing())
               {
                  //System.out.println("hierarchyChanged");
               	ModalProgressDialogFast.this.myTaskThread.interrupt();
               }
            }
         }
      });

      //myDialog.setVisible(true);
      //showTime = new Date();

      //System.out.println("ACTIVATE COMPLETED");
   }

   public synchronized void deactivateDialog()
   {
   	this.iAmFinished = true;
   	if (this.iWasDeactivated)
   	{
   		return;
   	}
     	this.iWasDeactivated = true;
   	
     	// let's make sure it always comes up...
     	/*
      if (showTime == null)
      {
         return;
      }
      */

      // Show for at least a second, even if done.
      // To avoid flicker.
     	long time = 0;
     	if (this.showTime != null)
     	{
         time = new Date().getTime() - this.showTime.getTime();     		
     	}
      if (time < 1000)
      {
         try
         {
            Thread.sleep(1000);
         }
         catch (InterruptedException e)
         {
         	ModalProgressDialogFast.this.myTaskThread.interrupt();
         }
      }

      //System.out.println("DEACTIVATE CALLED");
      this.myDialog.setVisible(false);
      //myDialog.dispose();
      //System.out.println("DEACTIVATE COMPLETED");
   }

   // =========================================================================
   // Glass Pane Methods (from DialogModality)
   // =========================================================================
   /**
    * Makes this object's dialog appear modal with
    * respect to its owner window.
    *
     */
   public void install()
   {
   	this.myTopWindow.requestFocus();

      // Prevents window from getting focus.
      // This (windowFocusPreventer) is what prevents keystrokes
      // from working in the window!!!
      this.myKeyboardFocusManager = KeyboardFocusManager.getCurrentKeyboardFocusManager();
      //myKeyboardFocusManager.addVetoableChangeListener("focusedWindow", windowFocusPreventer);

      this.myTopWindow.setCursor(
         Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

      // Adjusts the size of the mouseBlocker if the user changes
      // the screen size.  mouseBlocker is a clear panel which eats
      // mouse clicks.
      this.myTopWindow.addComponentListener(this.mouseBlockerSynchronizer);

      if (this.myTopWindow instanceof RootPaneContainer)
      {
         JLayeredPane l =
            ((RootPaneContainer)this.myTopWindow).getLayeredPane();
         this.mouseBlocker.setSize(l.getSize());
         l.add(this.mouseBlocker, new Integer(Integer.MAX_VALUE));
      }
      else
      {
         // better than nothing...
      	this.myTopWindow.setEnabled(false);
      }
      //System.out.println("INSTALL COMPLETED");
   }

   /**
    * Removes all influences of this object over its dialog.
    */
   public synchronized void uninstallGlassPane()
   {
   	this.iAmFinished = true;
   	if (this.iWasUnistalled)
   	{
   		return;
   	}
   	this.iWasUnistalled = true;
   	
      //System.out.println("UNINSTALL CALLED");

      this.myKeyboardFocusManager.removeVetoableChangeListener("focusedWindow", this.windowFocusPreventer); //$NON-NLS-1$

      this.myTopWindow.removeComponentListener(this.mouseBlockerSynchronizer);

      if (this.myTopWindow instanceof RootPaneContainer)
      {
         //System.out.println("removing from RootPaneContainer");
         JLayeredPane l = ((RootPaneContainer)this.myTopWindow).getLayeredPane();
         l.remove(this.mouseBlocker);
      }
      else
      {
         //System.out.println("EMERGENCY REMOVE");
         // better than nothing...
      	this.myTopWindow.setEnabled(true);
      }

      this.myTopWindow.setCursor(
         Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));

      //System.out.println("UNINSTALL COMPLETED");
   }

   // =========================================================================
   // Some final variables which are custom classes.
   // =========================================================================
   final Component mouseBlocker = new JPanel()
   {
      {
         setOpaque(false);
         enableEvents(AWTEvent.MOUSE_EVENT_MASK);
      }

      @Override
		protected void processMouseEvent(MouseEvent event)
      {
      	/*
         if (logger.isLoggable(Level.FINE))
         {
            String type;
            switch (event.getID())
            {
               case MouseEvent.MOUSE_CLICKED:
                  type = "MOUSE_CLICKED"; //$NON-NLS-1$
                  break;
               case MouseEvent.MOUSE_PRESSED:
                  type = "MOUSE_PRESSED"; //$NON-NLS-1$
                  break;
               case MouseEvent.MOUSE_RELEASED:
                  type = "MOUSE_RELEASED"; //$NON-NLS-1$
                  break;
               case MouseEvent.MOUSE_ENTERED:
                  type = "MOUSE_ENTERED"; //$NON-NLS-1$
                  break;
               case MouseEvent.MOUSE_EXITED:
                  type = "MOUSE_EXITED"; //$NON-NLS-1$
                  break;
               default:
                  type = "(unknown)"; //$NON-NLS-1$
                  break;
            }
         }
         */
         event.consume();
      }
   };

   private final ComponentListener mouseBlockerSynchronizer =
      new ComponentAdapter()
   {
      @Override
		public void componentResized(ComponentEvent event)
      {
         //System.out.println("Parent window resized");
         ModalProgressDialogFast.this.mouseBlocker.setSize(event.getComponent().getSize());
      }
   };

   /**
    * Prevents window from getting focus.
    */
   private final VetoableChangeListener windowFocusPreventer =
      new VetoableChangeListener()
   {
      public void vetoableChange(PropertyChangeEvent event)
         throws PropertyVetoException
      {
         String property = event.getPropertyName();
         //System.out.println("vetoableChange: " + property);
         if (property.equals("focusedWindow")) //$NON-NLS-1$
         {
            //System.out.println("focusedWindow property changed");
            if (event.getNewValue() == ModalProgressDialogFast.this.myTopWindow)
            {
               EventQueue.invokeLater(this.refocuser);
               synchronized (ModalProgressDialogFast.this.finishedSync)
               {
                  if (ModalProgressDialogFast.this.iAmFinished == false)
                  {
                     throw new PropertyVetoException(
                        "Window may not take focus" //$NON-NLS-1$
                        + " while it has a modal dialog showing", //$NON-NLS-1$
                        event);
                  }
               }
            }
         }
      }

      private final Runnable refocuser = new Runnable()
      {
         public void run()
         {
            if (ModalProgressDialogFast.this.myDialog != null)
            {
            	ModalProgressDialogFast.this.myDialog.requestFocus();
            }
            else
            {
               ModalProgressDialogFast.this.myTopWindow.requestFocus();
            }
         }
      };
   };

   // =========================================================================
   // Test Stub
   // =========================================================================
   public static void main(String[] args)
   {
      final BoundedRangeModel range = new DefaultBoundedRangeModel();
      range.setMaximum(10);

      JButton hitMe = new JButton("Hit Me!"); //$NON-NLS-1$
      final JLabel wacked = new JLabel("Not Wacked Yet"); //$NON-NLS-1$
      JButton start = new JButton("Start"); //$NON-NLS-1$
      final JLabel countLbl = new JLabel("Inactive at moment"); //$NON-NLS-1$

      JPanelRows contents = new JPanelRows();
      JPanel row = contents.topRow();
      row.add(hitMe);
      row.add(wacked);

      row = contents.nextRow();
      row.add(start);
      row.add(countLbl);

      final JFrame testFrame = new JFrame("File Tree Test"); //$NON-NLS-1$
      testFrame.getContentPane().add(new JScrollPane(contents));
      testFrame.setSize(300, 300);
      testFrame.addWindowListener(
         new WindowAdapter()
      {
         @Override
			public void windowClosing(WindowEvent evt)
         {
            System.exit(0);
         }
      });
      org.gerhardb.lib.swing.SwingUtils.centerOnScreen(testFrame);

      hitMe.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent event)
         {
            if (wacked.getBackground().equals(Color.GREEN))
            {
               wacked.setBackground(Color.RED);
               wacked.setOpaque(true);
               wacked.setForeground(Color.YELLOW);
               wacked.setText("WACKED"); //$NON-NLS-1$
            }
            else
            {
               wacked.setBackground(Color.GREEN);
               wacked.setOpaque(true);
               wacked.setForeground(Color.BLUE);
               wacked.setText("unwacked at moment"); //$NON-NLS-1$
            }
         }
      });

      class ShowProgress
      {

         ShowProgress(final String msg)
         {
            SwingUtilities.invokeLater(new Runnable()
            {
               public void run()
               {
                  countLbl.setText(msg);
               }
            });
         }
      }

      class TestMoveIt
         implements Runnable
      {
      	ModalProgressDialogFast dialog;
      	
         private TestMoveIt()
         {
            this.dialog = new ModalProgressDialogFast(
                  testFrame,
                  "Moving Files", //$NON-NLS-1$
                  "The files are being moved...", //$NON-NLS-1$
                  "Stop Move", //$NON-NLS-1$
                  range,
                  TestMoveIt.this);
         }

         public void run()
         {
            int length = range.getMaximum() + 1; // CAREFULL
            for (int i = 0; i < length; i++)
            {
               range.setValue(i);
               new ShowProgress("Doing: " + i); //$NON-NLS-1$
               try
               {
                  //Thread.sleep(1000L);
                  Thread.sleep(111L);
               }
               catch (InterruptedException e)
               {
                  break;
               }
               // This does not seem to work even when I interrupt the
               // thread.
               if (Thread.currentThread().isInterrupted())
               {
                  //System.out.println("THREAD WAS INTERRUPTED");
                  break;
               }
            }
            new ShowProgress("All Done!"); //$NON-NLS-1$
            this.dialog.done();
            //System.out.println("All Done!");

         }
      }

      start.addActionListener(
         new ActionListener()
      {
         public void actionPerformed(ActionEvent event)
         {
            new TestMoveIt();
            //TestMoveIt moveIt = new TestMoveIt();
            //Thread runMe = new Thread(moveIt);
         }
      });

      testFrame.setVisible(true);
   }
}

