/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.util.startup;

import java.awt.Dimension;
import java.io.File;
import java.util.HashMap;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JMenuItem;
import javax.swing.KeyStroke;

import org.gerhardb.jibs.viewer.frame.KeyChangerXML;
import org.gerhardb.lib.util.ActionHelpers;

/**
 * Trying to generically manage
 * internationalized accelerated iconized menus.
 * 
 * Make descendant that implements getStartupActions();
 */
public abstract class ActiveActions
{
	HashMap<String, Action> myLinkedActions = new HashMap<String, Action>();

	public ActiveActions()
	{
		// Prevents this from being created.
	}
	
	protected void initialize()
	{
		this.myLinkedActions = getFactoryDefaultActions().getStartupActions();			
	}
	
	public class MenuOverrides
	{
		public MenuOverrides()
		{
      	if (AppStarter.isJava6Enabled())
      	{
				// Get any user overrides
				// This may fail because it is a Java 6 feature.
				KeyChangerXML kc = new KeyChangerXML();
				File readFile = org.gerhardb.jibs.viewer.ViewerPreferences.getKeyChangerFile();
				kc.remapActionsFromXML(readFile, ActiveActions.this.myLinkedActions);	
      	}
		}
	}
	
	abstract protected FactoryDefaultActions getFactoryDefaultActions();
	/*
	 * {
	 * 	return new StartupActions(); // well, descendant of this class
	 * }
	 */
	
	public Action getAction(String menu, String item)
	{
		String key = menu + "." + item; //$NON-NLS-1$
		//System.out.println("ActiveActions returning: " + key);
		return this.myLinkedActions.get(key);
	}
	
	public HashMap<String, Action> getCurrentActions()
	{
		return this.myLinkedActions;
	}
	
	public HashMap<String, Action> getDefaultActions()
	{
		return getFactoryDefaultActions().getStartupActions();
	}
	
	
	protected JButton getAccelleratedToolBarButton(String menu, String item)
	{
		Action action = getAction(menu, item);
		return makeToolBarButton(action);
	}

	public JButton getToolBarButton(String menu, String item)
	{
		JButton rtnMe = getAccelleratedToolBarButton(menu, item);
		return rtnMe;
	}

	protected JCheckBox makeCheckBox(String menu, String item)
	{
		JCheckBox rtnMe = null;
		Action action = getAction(menu, item);
		if (action != null)
		{
			rtnMe = new JCheckBox(action);
		}
		else
		{
			rtnMe = new JCheckBox("unknown"); //$NON-NLS-1$
		}
		return rtnMe;
	}

	protected JMenuItem makeMenuItem(String menu, String item)
	{
		JMenuItem rtnMe = null;
		Action action = getAction(menu, item);
		if (action != null)
		{
			rtnMe = new JMenuItem(action);
		}
		else
		{
			rtnMe = new JMenuItem("unknown"); //$NON-NLS-1$
		}
		return rtnMe;
	}
	
	// ==========================================================================
	// Static Methods
	// ==========================================================================
	
	public static JButton makeToolBarButton(Action action)
	{
		JButton rtnMe = new JButton(action);
		// smaller than on gb windows display
		Dimension size = new Dimension(23, 23);
		rtnMe.setMaximumSize(size);
		rtnMe.setPreferredSize(size);
		rtnMe.setEnabled(true);
		rtnMe.setText(null);
		return rtnMe;
	}

	public static String makeKey(String application, String menu, String item)
	{
		return application + "." + menu + "." + item; 
	}
	
	public static Action loadAction(String resourseKey, Action action, Icon icon)
	{
		String retrieved = null;
		String itemKey = null; 
		try
		{		
			itemKey = resourseKey + ".label"; 
			retrieved = AppStarter.getString(itemKey);
			action.putValue(Action.NAME, retrieved);
		}
		catch(Exception ex)
		{
			System.out.println("-------------------------------------------"); //$NON-NLS-1$
			System.out.println("loadAction label: " + ex.getLocalizedMessage()); //$NON-NLS-1$
			System.out.println(itemKey + ": " +  retrieved);	 //$NON-NLS-1$
		}
	
		try
		{		
			itemKey = resourseKey + ".tip"; 
			retrieved = AppStarter.getString(itemKey);
			action.putValue(Action.SHORT_DESCRIPTION, retrieved);
		}
		catch(Exception ex)
		{
			System.out.println("-------------------------------------------"); //$NON-NLS-1$
			System.out.println("loadAction tip: " + ex.getLocalizedMessage()); //$NON-NLS-1$
			System.out.println(itemKey + ": " +  retrieved);	 //$NON-NLS-1$
		}
	
		try
		{		
			itemKey = resourseKey + ".mnemonic"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			retrieved = AppStarter.getString(itemKey);
			int keyCode = ActionHelpers.getKeyCode(retrieved);
			//System.out.println("mnemonic from PROPERTIES: " + retrieved + "   Key Code: " + keyCode);
			action.putValue(Action.MNEMONIC_KEY, new Integer(keyCode));
		}
		catch(Exception ex)
		{
			System.out.println("-------------------------------------------"); //$NON-NLS-1$
			System.out.println("loadAction mnemonic: " + ex.getLocalizedMessage()); //$NON-NLS-1$
			System.out.println(itemKey + ": " +  retrieved);	 //$NON-NLS-1$
		}
		
		try
		{		
			itemKey = resourseKey + ".accelerator"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			retrieved = AppStarter.getString(itemKey);
			if ( !retrieved.trim().equals("null")) //$NON-NLS-1$
			{
				//System.out.println("resourceKey: " + retrieved);
				KeyStroke ks = KeyStroke.getKeyStroke(retrieved);
				action.putValue(Action.ACCELERATOR_KEY, ks);
			}
		}
		catch(Exception ex)
		{
			System.out.println("-------------------------------------------"); //$NON-NLS-1$
			System.out.println("loadAction mnemonic: " + ex.getLocalizedMessage()); //$NON-NLS-1$
			System.out.println(itemKey + ": " +  retrieved);	 //$NON-NLS-1$
		}
		
		action.putValue(Action.SMALL_ICON, icon);
		return action;
	}
	
	// ==========================================================================
	// Main
	// ==========================================================================
	/*
	public static void main(String[] args)
	{
		String key = "A"; //$NON-NLS-1$
		int keyCode = getKeyCode(key);
		char keyChar = getKeyChar(keyCode);
		String keyString = getKeyString(keyCode);
		
		System.out.println("keyCode: " + keyCode); //$NON-NLS-1$
		System.out.println("keyChar: " + keyChar); //$NON-NLS-1$
		System.out.println("keyString: " + keyString); //$NON-NLS-1$
		System.out.println("back to code from char: " + getKeyCode(keyChar)); //$NON-NLS-1$
		for(int i = 0; i < 128; i++) 
		{
         //System.out.println(i + " = " + (char)i);
      }
		
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.SHIFT_MASK).toString();
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.ALT_MASK).toString();
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.CTRL_MASK).toString();
		KeyStroke.getKeyStroke(keyChar, java.awt.event.InputEvent.META_MASK).toString();
		
	}	
	*/
}
