/*
 * Copyright (c) 2007 Gerhard Beck. All rights reserved.
 * 
 * Subject to the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007
 * http://www.gnu.org/licenses/gpl.html
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" AND ANY EXPRESSED OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GERHARD
 * BECK OR OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.gerhardb.lib.util.startup;

import java.io.*;
import java.net.URL;
import java.net.URLDecoder;
import java.util.Date;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.prefs.Preferences;

import javax.swing.JOptionPane;

import org.gerhardb.lib.dirtree.DTNWriter;
import org.gerhardb.lib.dirtree.rdp.PathManager;

import com.saic.isd.util.ArgList;

/**
 * Provides a single source of entry to be called from the Jar file. Picks the
 * appropriate application to start based on passed-in parameters.
 * 
 * -props D:\dev\workspaces\JIBS\bin\jibs.props
 * -thumbdrive
 * 
 * Thumbdrive takes precedence.
 */
public class AppStarter
{
	private static ResourceBundle resources;

	public static final Date TIME_STARTED = new Date();
	
	static final String PROPS = "-props";
	static final String THUMBDRIVE = "-thumbdrive";
	private static boolean clsThumbdrive = false;
	
	// Initialized by first call to: getMyDocumentsDir()
	// Don't use directly, only through getMyDocumentsDir().
	private static File clsMyDocsDir;
	
	static boolean iAmJava6Enabled = true;
	
	private AppStarter()
	{
		// Prevents this from being created.
	}

	public static boolean isJava6Enabled()
	{
		if (!iAmJava6Enabled)
		{
			System.out.println("Featurce called which is not Java 6 enabled while running Java 1.5 or Java 5.");
		}
		return iAmJava6Enabled;
	}
	
	/**
	 * Loads resource files.
	 * @param args
	 * @param bundleName
	 * @param usePropParams
	 * @param appName
	 */
	public static void startUpApp(String[] args, String bundleName, boolean usePropParams)
	{
		String version = System.getProperty("java.version");
		System.out.println("==================================> Jibs6Hacks java.version: " + version);
		if (version.startsWith("1.4") || version.startsWith("1.5"))
		{
			System.out.println("Java 1.4 or 1.5 found.  Turning off Java 6 features.");
			iAmJava6Enabled = false;
		}
		
		//showArgs(args);
		log("Bundle: " + bundleName);
		resources = ResourceBundle.getBundle(bundleName, Locale.getDefault());
		log("Bundle Resources: " + resources);
		
		ArgList argList = new ArgList(args);
		try
   	{
			if (usePropParams && useCustomAppPropertyFile(argList))
			{
	   		AppPreferencesFactory.initializeProps();				
			}
   		AppPreferencesFactory.InfoToConsole();
   	}
   	catch(Exception ex)
   	{
   		ex.printStackTrace();
   	}

  		log("********************************"); //$NON-NLS-1$
		log("AppStarter Starting Up the Application" );
		log("Loaded Jibs from: " + getStartupDirectory());
      log("Java Prefs: " + Preferences.userRoot().getClass());
      log("Pref File: " + System.getProperty(AppPreferencesFactory.PROP_FILE_LOOKUP));
  		log("********************************"); //$NON-NLS-1$
 		
  		// This is if you double click on a *.jibs file.
  		if (args != null && args.length == 1)
  		{
  			File clickedOnJibs = new File(args[0]);
  			if (clickedOnJibs.exists() && clickedOnJibs.isFile())
  			{
  		 		PathManager.setStartingJibsFile(clickedOnJibs);			
  			}
  		}
  		else
  		{
  			// Now that we have the right preferences loaded.
  	  		PathManager.overrideInitialDirectories(
  					argList.getOption("-tree"), argList.getOption("-view")); //$NON-NLS-1$ //$NON-NLS-2$ 
  		}  		
 	}
	
	public static void closeDownApp()
	{
		if (clsLogStream != null)
		{
			clsLogStream.flush();
			clsLogStream.close();
		}
	}
	public static PrintWriter clsLogStream;
	public static void log(String msg)
	{
		System.out.println(msg);
		if (clsLogStream != null)
		{
			clsLogStream.println(msg);
			clsLogStream.flush();
		}
	}
	
	public static String getString(String key)
	{
		try
		{
			return resources.getString(key);
		}
		catch (Exception ex)
		{
			ex.printStackTrace();
			return '!' + key + '!';
		}
	}
	
	public static boolean isThumbdrive()
	{
		return clsThumbdrive;
	}
	
	/**
	 * This will never return a null.
	 * If it would be a null, you get an error message and a system exit.
	 * @return
	 */
	public static File getStartupDirectory()
	{
		String errMsg = "No starting URL provided";
		try
		{
			URL startURL = AppStarter.class.getProtectionDomain().getCodeSource().getLocation();
			errMsg = startURL.toString();
			
			String startPath = startURL.getPath();
			startPath = URLDecoder.decode(startPath, "UTF-8");
			
			File startupFile = new File(startPath);
			if (startupFile.exists())
			{
				if (startupFile.isDirectory())
				{
					return startupFile;
				}
				return startupFile.getParentFile();
			}
		}
		catch(Exception ex)
		{
			ex.printStackTrace();
		}
		
      JOptionPane.showMessageDialog(
            null,
            "JIBS does not understand start path of: \n" + errMsg,
            "JIBS Start Problem",
            JOptionPane.ERROR_MESSAGE );
      System.exit(99);
      return null;
	}
	
	public static File getMyDocumentsDir()
	{
		if (clsMyDocsDir == null)
		{
			//String os = System.getProperty("os.name").toLowerCase(); //$NON-NLS-1$
			javax.swing.JFileChooser fr = new javax.swing.JFileChooser();
	      javax.swing.filechooser.FileSystemView fw=fr.getFileSystemView();
	      clsMyDocsDir = fw.getDefaultDirectory();
		}
		return clsMyDocsDir;
	}

	public static File getMyPicturesDir()
	{
		File rtnMe = getMyDocumentsDir();		
		String os = System.getProperty("os.name").toLowerCase(); //$NON-NLS-1$
		if (os.startsWith("windows")) //$NON-NLS-1$
		{
			// If we can find it use it.
			// Don't create it because user might not be English.
			File myPics = new File(rtnMe.getAbsolutePath() + "/My Pictures"); //$NON-NLS-1$
			if (myPics.exists() && myPics.isDirectory())
			{
				return myPics;
			}
		}
		return rtnMe;
	}


	// ==========================================================================
	// protected
	// ==========================================================================	
	
	/*
	 * Main function is to set the property file variable.
	 * Secondary function is to signal default load if that process fails.
	 */
	static boolean useCustomAppPropertyFile(ArgList argList)
	{
		//log("Jibs.useCustomAppPropertyFile()");
		if (argList.isOption(PROPS))
		{
			String propFileName = argList.getOption(PROPS);
			//log("Jibs.PropFile[PROPS]: " + propFileName);
			if (propFileName == null || propFileName.length() == 0)
			{
				//log("LOADED JIBS PREFERENCES FROM BECAUSE NO PROP FILE PASSED IN");
				return false;
			}
			System.setProperty(AppPreferencesFactory.PROP_FILE_LOOKUP, propFileName);
			return true;
		}
		else if (argList.isOption(THUMBDRIVE))
		{
			String propFileName = argList.getOption(THUMBDRIVE);
			//log("Jibs.PropFile[THUMBDRIVE]: " + propFileName);
			System.setProperty(AppPreferencesFactory.PROP_FILE_LOOKUP, propFileName);
			clsThumbdrive = true;
			return true;
		}
		else
		{
			//log("LOADED JIBS PREFERENCES FROM DEFAULT DATA STORE");
			return false;
		}
	}
	
	public static void showArgs(String[] args)  // NO_UCD
	{
      String fileName = "C:/gbroot/jibsLOG.txt"; //$NON-NLS-1$
      File file = new File(fileName);
      try{clsLogStream = new PrintWriter(new BufferedWriter(new FileWriter(file)));}catch(Exception ex)
      {
      	ex.printStackTrace();
      }
      
		log("Jibs Startup"); //$NON-NLS-1$
		log("##################################################"); //$NON-NLS-1$
		for(int i=0; i<args.length; i++)
		{
				log(args[i]);
		}
		log("##################################################"); //$NON-NLS-1$
		
		File[] roots = File.listRoots();
		for(int i=0; i<roots.length; i++)
		{
			log(DTNWriter.getNameOfDrive(roots[i].toString())); 
		}
		log("##################################################"); //$NON-NLS-1$
	}	
	
	
	// ==========================================================================
	// TEST MAIN
	// ==========================================================================	
	static public void testMain(String[] args)
	{
		log(AppStarter.class.getName());
		String language;
		String country;

		if (args.length != 2)
		{
			language = new String("en"); //$NON-NLS-1$
			country = new String("US"); //$NON-NLS-1$
		}
		else
		{
			language = new String(args[0]);
			country = new String(args[1]);
		}

		Locale currentLocale;
		ResourceBundle messages;

		currentLocale = new Locale(language, country);
		log("Locale: " + currentLocale); //$NON-NLS-1$

		currentLocale = Locale.getDefault();
		log("Locale: " + currentLocale); //$NON-NLS-1$

		messages = ResourceBundle.getBundle("org.gerhardb.jibs.Jibs"); //$NON-NLS-1$

		log(messages.getString("AboutBox.title")); //$NON-NLS-1$

		Locale[] locales = Locale.getAvailableLocales();
		for (int i = 0; i < locales.length; i++)
		{
			log("Available Locale: " + locales[i]); //$NON-NLS-1$

		}

		Locale.setDefault(Locale.GERMANY);
		log("German Locale: " + Locale.GERMANY); //$NON-NLS-1$
		log("Current Locale: " + Locale.getDefault()); //$NON-NLS-1$

		messages = ResourceBundle.getBundle("org.gerhardb.jibs.Jibs", Locale //$NON-NLS-1$
				.getDefault());
		log("German: " //$NON-NLS-1$
				+ messages.getString("AboutBox.title")); //$NON-NLS-1$

	}
}
